﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
* @file
* @brief 較正情報書き込みライブラリで使用する型や定数の定義。
*/

#include <nn/nn_Common.h>

namespace nn { namespace cal {

/**
* @brief       デバイス証明書を表す構造体です。
*/
struct DeviceCertificate
{
    ::nn::Bit8 data[384];   //!< デバイス証明書
};

/**
* @brief       e-Ticket 証明書を表す構造体です。
*/
struct ETicketCertificate
{
    ::nn::Bit8 data[384];   //!< e-Ticket 証明書
};

/**
* @brief       ECC-P256 版のデバイス秘密鍵を表す構造体です。
*/
struct EccP256DeviceKey
{
    ::nn::Bit8 data[48];
};

/**
* @brief       ECC-B233 版のデバイス秘密鍵を表す構造体です。
*/
struct EccB233DeviceKey
{
    ::nn::Bit8 data[48];
};

/**
* @brief       SSL クライアント秘密鍵を表す構造体です。
*/
struct SslKey
{
    ::nn::Bit8 data[272];
};

/**
* @brief       SSL クライアント証明書を表す構造体です。
*/
struct SslCertificate
{
    ::nn::Bit8 data[2048];
};

/**
* @brief       乱数ブロックを表す構造体です。
*/
struct RandomNumber
{
    ::nn::Bit8 data[4096];
};

/**
* @brief       ゲームカード相互認証秘密鍵を表す構造体です。
*/
struct GameCardKey
{
    ::nn::Bit8 data[272];
};

/**
* @brief       ゲームカード相互認証証明書を表す構造体です。
*/
struct GameCardCertificate
{
    ::nn::Bit8 data[1024];
};

/**
* @brief       RSA-2048 版の eTicket 発行用デバイス秘密鍵を表す構造体です。
*/
struct Rsa2048ETicketKey
{
    ::nn::Bit8 data[544];
};

/**
* @brief       RSA-2048 版の eTicket 発行用デバイス証明書を表す構造体です。
*/
struct Rsa2048ETicketCertificate
{
    ::nn::Bit8 data[576];
};

/**
* @brief       スピーカー補正値を表す構造体です。
*/
struct SpeakerCalibrationValue
{
    ::nn::Bit8 data[90];
};

/**
* @brief       amiibo 用本体秘密鍵(ECQV-ECDSA)を表す構造体です。
*/
struct AmiiboKey
{
    ::nn::Bit8 data[80];
};

/**
* @brief       amiibo 用 ECQV 証明書を表す構造体です。
*/
struct AmiiboEcqvCertificate
{
    ::nn::Bit8 data[20];
};

/**
* @brief       amiibo 用 ECDSA 証明書を表す構造体です。
*/
struct AmiiboEcdsaCertificate
{
    ::nn::Bit8 data[112];
};

/**
* @brief       amiibo 用本体秘密鍵 ECQV-BLS を表す構造体です。
*/
struct AmiiboEcqvBlsKey
{
    ::nn::Bit8 data[64];
};

/**
* @brief       amiibo 用 ECQV-BLS 証明書を表す構造体です。
*/
struct AmiiboEcqvBlsCertificate
{
    ::nn::Bit8 data[32];
};

/**
* @brief       amiibo 用 ECQV-BLS ルート証明書を表す構造体です。
*/
struct AmiiboEcqvBlsRootCertificate
{
    ::nn::Bit8 data[144];
};

/**
* @brief       本体モデルタイプを表す構造体です。
*/
struct ProductModel
{
    int32_t value;
};

/**
* @brief       本体のカラーバリエーションを表す構造体です。
*/
struct ColorVariation
{
    ::nn::Bit8 data[6];
};

/**
* @brief       LCD バックライト出力の写像の係数を表す構造体です。
*/
struct LcdBacklightBrightnessMapping
{
    float coefficient[3];
};

/**
* @brief       領域拡張された ECC-B233 版のデバイス秘密鍵を表す構造体です。
*/
struct ExtendedEccB233DeviceKey
{
    ::nn::Bit8 data[80];
};

/**
* @brief       領域拡張された ECC-P256 版のデバイス秘密鍵を表す構造体です。
*/
struct ExtendedEccP256DeviceKey
{
    ::nn::Bit8 data[80];
};

/**
* @brief       領域拡張された RSA-2048 版の eTicket 発行用デバイス秘密鍵を表す構造体です。
*/
struct ExtendedRsa2048ETicketKey
{
    ::nn::Bit8 data[576];
};

/**
* @brief       領域拡張された SSL クライアント秘密鍵を表す構造体です。
*/
struct ExtendedSslKey
{
    ::nn::Bit8 data[304];
};

/**
* @brief       領域拡張された ゲームカード相互認証秘密鍵を表す構造体です。
*/
struct ExtendedGameCardKey
{
    ::nn::Bit8 data[304];
};

/**
* @brief       LCD のベンダ ID を表す構造体です。
*/
struct LcdVendorId
{
    uint32_t value;
};

/**
* @brief       領域拡張された RSA-2048 版のデバイス秘密鍵を表す構造体です。
*/
struct ExtendedRsa2048DeviceKey
{
    ::nn::Bit8 data[576];
};

/**
* @brief       RSA-2048 版の eTicket 発行用デバイス証明書を表す構造体です。
*/
struct Rsa2048DeviceCertificate
{
    ::nn::Bit8 data[576];
};

/**
* @brief       USB Type-C Power Source 回路バージョンを表す構造体です。
*/
struct UsbTypeCPowerSourceCircuit
{
    ::nn::Bit8 version;
};

/**
* @brief       本体カラーを表す構造体です。
*/
struct HousingColor
{
    ::nn::Bit8 data[4];
};

/**
* @brief       生産時較正情報の raw 領域を表す構造体です。
*/
struct RawCalibrationInfo
{
    ::nn::Bit8 data[16 * 2048];
};

}}
