﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   較正情報書き込みライブラリの公開ヘッダ。
 */

#include <string>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/settings/factory/settings_BatteryLot.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/system/settings_Region.h>

#include "cal_Types.h"

/**
 * @namespace nn::cal
 * @brief     較正情報書き込みライブラリの名前空間です。
 */
namespace nn { namespace cal {

/**
* @brief       ゼロ初期化します。
*
* @details     較正情報の書き込み領域をゼロ埋めします。
*
* @return
*/
nn::Result ZeroInitialize() NN_NOEXCEPT;

/**
* @brief       仕上げ処理をします。
*
* @details     較正情報本体全体のハッシュ値を計算し、ヘッダーを確定させます。
*              書き込み回数のインクリメントは下記のコマンドが実行された際に発生します。
*
* @return
*/
nn::Result FinishUp() NN_NOEXCEPT;

/**
* @brief       ベリフィケーションをします。
*
* @details     較正情報の内容、 CRC、 SHA256 ハッシュの比較し結果を出力します。
*              任意の工程で実行可能です。
*
* @param[out]  outIsSuccess                    ベリファイ結果の合否
*
* @return
*/
nn::Result Verify(bool &outIsSuccess) NN_NOEXCEPT;

/**
* @brief       構成識別子 1 （Board Type） の書き換えをします。
*
* @details     構成識別子 1 を書き込みます。
*
* @param[in]  configurationId1             書き込みたい構成識別子
*
* @return
*/
nn::Result WriteConfigurationId1(const ::nn::settings::factory::ConfigurationId1& configurationId1) NN_NOEXCEPT;

/**
* @brief       無線 LAN 用カントリーコードの書き換えをします。
*
* @details     無線 LAN 用カントリーコードを書き込みます。
*
* @param[in]  countryCodeArray             書き込みたいカントリーコードの配列
* @param[in]  arrayLength				   countryCodeArray の要素数
*
* @return
*/
nn::Result WriteWlanCountryCodes(const ::nn::settings::factory::CountryCode *countryCodeArray, const size_t arrayLength) NN_NOEXCEPT;

/**
* @brief       無線 LAN 用 MAC アドレスの書き換えをします。
*
* @details     無線 LAN 用 MAC アドレスを書き込みます。
*
* @param[in]  macAddress             書き込みたい MAC アドレス
*
* @return
*/
nn::Result WriteWlanMacAddress(const ::nn::settings::factory::MacAddress& macAddress) NN_NOEXCEPT;

/**
* @brief       BD アドレスの書き換えをします。
*
* @details     BlueTooth 用 BD アドレスを書き込みます。
*
* @param[in]  bdAddress             書き込みたい BD アドレス
*
* @return
*/
nn::Result WriteBdAddress(const ::nn::settings::factory::BdAddress& bdAddress) NN_NOEXCEPT;

/**
* @brief       内蔵加速度センサのオフセット値の書き換えをします。
*
* @details     内蔵加速度センサのオフセット値を書き込みます。
*
* @param[in]  x             X軸オフセット値
* @param[in]  y             Y軸オフセット値
* @param[in]  z             Z軸オフセット値
*
* @return
*/
nn::Result WriteAccelerometerOffset(const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT;

/**
* @brief       内蔵加速度センサのスケール値の書き換えをします。
*
* @details     内蔵加速度センサのスケール値を書き込みます。
*
* @param[in]  x             X軸スケール値
* @param[in]  y             Y軸スケール値
* @param[in]  z             Z軸スケール値
*
* @return
*/
nn::Result WriteAccelerometerScale(const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT;

/**
* @brief       内蔵ジャイロスコープのオフセット値の書き換えをします。
*
* @details     内蔵ジャイロスコープのオフセット値を書き込みます。
*
* @param[in]  x             X軸オフセット値
* @param[in]  y             Y軸オフセット値
* @param[in]  z             Z軸オフセット値
*
* @return
*/
nn::Result WriteGyroscopeOffset(const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT;

/**
* @brief       内蔵ジャイロスコープのスケール値の書き換えをします。
*
* @details     内蔵ジャイロスコープのスケール値を書き込みます。
*
* @param[in]  x             X軸スケール値
* @param[in]  y             Y軸スケール値
* @param[in]  z             Z軸スケール値
*
* @return
*/
nn::Result WriteGyroscopeScale(const int16_t x, const int16_t y, const int16_t z) NN_NOEXCEPT;

/**
* @brief       製品シリアル番号書き換えをします。
*
* @details     製品シリアル番号を書き込みます。
*
* @param[in]  serialNumber             書き込みたい製品シリアル番号
*
* @return
*/
nn::Result WriteSerialNumber(const ::nn::settings::factory::SerialNumber& serialNumber) NN_NOEXCEPT;

/**
* @brief      デバイス登録用秘密鍵 （ECC-P256 版） の書き換えをします。
*
* @details    デバイス登録用秘密鍵 （ECC-P256 版）を書き込みます。
*
* @param[in]  deviceKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteEccP256DeviceKey(const EccP256DeviceKey& deviceKey) NN_NOEXCEPT;

/**
* @brief      デバイス登録用証明書（ECC-P256 版） の書き換えをします。
*
* @details    デバイス登録用証明書 （ECC-P256 版）を書き込みます。
*
* @param[in]  deviceCertificate             書き込みたいデバイス登録用証明書
*
* @return
*/
nn::Result WriteEccP256DeviceCertificate(const DeviceCertificate& deviceCertificate) NN_NOEXCEPT;

/**
* @brief      デバイス登録用秘密鍵 （ECC-B233 版） の書き換えをします。
*
* @details    デバイス登録用秘密鍵 （ECC-B233 版）を書き込みます。
*
* @param[in]  deviceKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteEccB233DeviceKey(const EccB233DeviceKey& deviceKey) NN_NOEXCEPT;

/**
* @brief      デバイス登録用証明書（ECC-B233 版） の書き換えをします。
*
* @details    デバイス登録用証明書 （ECC-B233 版）を書き込みます。
*
* @param[in]  deviceCertificate             書き込みたいデバイス登録用証明書
*
* @return
*/
nn::Result WriteEccB233DeviceCertificate(const DeviceCertificate& deviceCertificate) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用秘密鍵 （ECC-P256 版） の書き換えをします。
*
* @details    eTicket 発行用秘密鍵 （ECC-P256 版） を書き込みます。
*
* @param[in]  eTicketKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteEccP256ETicketKey(const EccP256DeviceKey& eTicketKey) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用証明書（ECC-P256 版） の書き換えをします。
*
* @details    eTicket 発行用証明書 （ECC-P256 版）を書き込みます。
*
* @param[in]  eTicketCertificate             書き込みたい eTicket 発行用証明書
*
* @return
*/
nn::Result WriteEccP256ETicketCertificate(const ETicketCertificate& eTicketCertificate) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用秘密鍵 （ECC-B233 版） の書き換えをします。
*
* @details    eTicket 発行用秘密鍵 （ECC-B233 版） を書き込みます。
*
* @param[in]  eTicketKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteEccB233ETicketKey(const EccB233DeviceKey& eTicketKey) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用証明書（ECC-B233 版） の書き換えをします。
*
* @details    eTicket 発行用証明書 （ECC-B233 版）を書き込みます。
*
* @param[in]  eTicketCertificate             書き込みたい eTicket 発行用証明書
*
* @return
*/
nn::Result WriteEccB233ETicketCertificate(const ETicketCertificate& eTicketCertificate) NN_NOEXCEPT;

/**
* @brief      SSL クライアント秘密鍵の書き換えをします。
*
* @details    SSL クライアント秘密鍵を書き込みます。
*
* @param[in]  sslKey             書き込みたい SSL クライアント秘密鍵
*
* @return
*/
nn::Result WriteSslKey(const SslKey& sslKey) NN_NOEXCEPT;

/**
* @brief      SSL クライアント証明書の書き換えをします。
*
* @details    SSL クライアント証明書を書き込みます。
*
* @param[in]  pSslCertificate            書き込みたい SSL クライアント証明書のバッファ
* @param[in]  certificateSize            sslCertificate のサイズ
*
* @return
*/
nn::Result WriteSslCertificate(const void* pSslCertificate, const size_t certificateSize) NN_NOEXCEPT;

/**
* @brief      乱数の書き換えをします。
*
* @details    乱数を書き込みます。
*
* @param[in]  randomNumber             書き込みたい乱数
*
* @return
*/
nn::Result WriteRandomNumber(const RandomNumber& randomNumber) NN_NOEXCEPT;

/**
* @brief      ゲームカード相互認証秘密鍵の書き換えをします。
*
* @details    ゲームカード相互認証秘密鍵を書き込みます。
*
* @param[in]  gamecardKey             書き込みたいゲームカード相互認証秘密鍵
*
* @return
*/
nn::Result WriteGameCardKey(const GameCardKey& gamecardKey) NN_NOEXCEPT;

/**
* @brief      ゲームカード相互認証証明書の書き換えをします。
*
* @details    ゲームカード相互認証証明書を書き込みます。
*
* @param[in]  gameCardCertificate             書き込みたいゲームカード相互認証証明書
*
* @return
*/
nn::Result WriteGameCardCertificate(const GameCardCertificate& gameCardCertificate) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用秘密鍵 （RSA-2048 版） の書き換えをします。
*
* @details    eTicket 発行用秘密鍵 （RSA-2048 版）を書き込みます。
*
* @param[in]  rsa2048DeviceKey             書き込みたい eTicket 発行用秘密鍵
*
* @return
*/
nn::Result WriteRsa2048ETicketKey(const Rsa2048ETicketKey& rsa2048DeviceKey) NN_NOEXCEPT;

/**
* @brief      eTicket 発行用証明書 （RSA-2048 版） の書き換えをします。
*
* @details    eTicket 発行用証明書 （RSA-2048 版）を書き込みます。
*
* @param[in]  rsa2048DeviceCertificate             書き込みたい eTicket 発行用証明書
*
* @return
*/
nn::Result WriteRsa2048ETicketCertificate(const Rsa2048ETicketCertificate& rsa2048DeviceCertificate) NN_NOEXCEPT;

/**
* @brief      電池 LOT の書き換えをします。
*
* @details    電池 LOT を書き込みます。
*
* @param[in]  batteryLot             書き込みたい電池 LOT
*
* @return
*/
nn::Result WriteBatteryLot(const ::nn::settings::factory::BatteryLot& batteryLot) NN_NOEXCEPT;

/**
* @brief      スピーカー補正値の書き換えをします。
*
* @details    スピーカー補正値を書き込みます。
*
* @param[in]  speakerCalibrationValue             書き込みたいスピーカー補正値
*
* @return
*/
nn::Result WriteSpeakerCalibrationValue(const SpeakerCalibrationValue& speakerCalibrationValue) NN_NOEXCEPT;

/**
* @brief      リージョンコードの書き換えをします。
*
* @details    リージョンコードを書き込みます。
*
* @param[in]  regionCode             書き込みたいリージョンコード
*
* @return
*/
nn::Result WriteRegionCode(const ::nn::settings::system::RegionCode regionCode) NN_NOEXCEPT;

/**
* @brief      amiibo 用本体秘密鍵の書き換えをします。
*
* @details    amiibo 用本体秘密鍵を書き込みます。
*
* @param[in]  amiiboKey             書き込みたい amiibo 用本体秘密鍵
*
* @return
*/
nn::Result WriteAmiiboKey(const AmiiboKey& amiiboKey) NN_NOEXCEPT;

/**
* @brief      amiibo 用 ECQV 証明書の書き換えをします。
*
* @details    amiibo 用 ECQV 証明書を書き込みます。
*
* @param[in]  amiiboEcqvCertificate             書き込みたい amiibo 用 ECQV 証明書
*
* @return
*/
nn::Result WriteAmiiboEcqvCertificate(const AmiiboEcqvCertificate& amiiboEcqvCertificate) NN_NOEXCEPT;

/**
* @brief      amiibo 用 ECDSA 証明書の書き換えをします。
*
* @details    amiibo 用 ECDSA 証明書を書き込みます。
*
* @param[in]  amiiboEcdsaCertificate             書き込みたい amiibo 用 ECDSA 証明書
*
* @return
*/
nn::Result WriteAmiiboEcdsaCertificate(const AmiiboEcdsaCertificate& amiiboEcdsaCertificate) NN_NOEXCEPT;

/**
* @brief      amiibo 用 ECQV-BLS 本体秘密鍵の書き換えをします。
*
* @details    amiibo 用 ECQV-BLS 本体秘密鍵を書き込みます。
*
* @param[in]  amiiboEcqvBlsKey             書き込みたい amiibo 用 ECQV-BLS 本体秘密鍵
*
* @return
*/
nn::Result WriteAmiiboEcqvBlsKey(const AmiiboEcqvBlsKey& amiiboEcqvBlsKey) NN_NOEXCEPT;

/**
* @brief      amiibo 用 ECQV-BLS 証明書の書き換えをします。
*
* @details    amiibo 用 ECQV-BLS 証明書を書き込みます。
*
* @param[in]  amiiboEcqvBlsCertificate             書き込みたい amiibo 用 ECQV-BLS 証明書
*
* @return
*/
nn::Result WriteAmiiboEcqvBlsCertificate(const AmiiboEcqvBlsCertificate& amiiboEcqvBlsCertificate) NN_NOEXCEPT;

/**
* @brief      amiibo 用 ルート証明書 (ECQV-BLS) の書き換えをします。
*
* @details    amiibo 用 ルート証明書 (ECQV-BLS) を書き込みます。
*
* @param[in]  amiiboEcqvBlsRootCertificate             書き込みたい amiibo 用 ECQV-BLS 証明書
*
* @return
*/
nn::Result WriteAmiiboEcqvBlsRootCertificate(const AmiiboEcqvBlsRootCertificate& amiiboEcqvBlsRootCertificate) NN_NOEXCEPT;

/**
* @brief      本体モデルタイプの書き換えをします。
*
* @details    本体モデルタイプの書き込みます。
*
* @param[in]  productModel             書き込みたい本体モデルタイプ
*
* @return
*/
nn::Result WriteProductModel(const ProductModel& productModel) NN_NOEXCEPT;

/**
* @brief      本体カラーバリエーションの書き換えをします。
*
* @details    本体カラーバリエーションを書き込みます。
*
* @param[in]  colorVariation             書き込みたい本体カラーバリエーション
*
* @return
*/
nn::Result WriteColorVariation(const ColorVariation& colorVariation) NN_NOEXCEPT;

/**
* @brief      LCDバックライト出力の写像の係数の書き換えをします。
*
* @details    LCDバックライト出力の写像の係数を書き込みます。
*
* @param[in]  lcdBacklightBrightnessMapping             書き込みたいLCDバックライト出力の写像の係数
*
* @return
*/
nn::Result WriteLcdBacklightBrightnessMapping(const LcdBacklightBrightnessMapping& lcdBacklightBrightnessMapping) NN_NOEXCEPT;

/**
* @brief      領域拡張されたデバイス登録用秘密鍵 （ECC-B233 版） の書き換えをします。
*
* @details    領域拡張されたデバイス登録用秘密鍵 （ECC-B233 版）を書き込みます。
*
* @param[in]  deviceKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteExtendedEccB233DeviceKey(const ExtendedEccB233DeviceKey& deviceKey) NN_NOEXCEPT;

/**
* @brief      領域拡張された eTicket 発行用秘密鍵 （ECC-P256 版） の書き換えをします。
*
* @details    領域拡張された eTicket 発行用秘密鍵 （ECC-P256 版） を書き込みます。
*
* @param[in]  eTicketKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteExtendedEccP256ETicketKey(const ExtendedEccP256DeviceKey& eTicketKey) NN_NOEXCEPT;

/**
* @brief      領域拡張された eTicket 発行用秘密鍵 （ECC-B233 版） の書き換えをします。
*
* @details    領域拡張された eTicket 発行用秘密鍵 （ECC-B233 版） を書き込みます。
*
* @param[in]  eTicketKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteExtendedEccB233ETicketKey(const ExtendedEccB233DeviceKey& eTicketKey) NN_NOEXCEPT;

/**
* @brief      領域拡張された eTicket 発行用秘密鍵 （RSA-2048 版） の書き換えをします。
*
* @details    領域拡張された eTicket 発行用秘密鍵 （RSA-2048 版）を書き込みます。
*
* @param[in]  rsa2048DeviceKey             書き込みたい eTicket 発行用秘密鍵
*
* @return
*/
nn::Result WriteExtendedRsa2048ETicketKey(const ExtendedRsa2048ETicketKey& rsa2048DeviceKey) NN_NOEXCEPT;

/**
* @brief      領域拡張された SSL クライアント秘密鍵の書き換えをします。
*
* @details    領域拡張された SSL クライアント秘密鍵を書き込みます。
*
* @param[in]  sslKey             書き込みたい SSL クライアント秘密鍵
*
* @return
*/
nn::Result WriteExtendedSslKey(const ExtendedSslKey& sslKey) NN_NOEXCEPT;

/**
* @brief      領域拡張されたゲームカード相互認証秘密鍵の書き換えをします。
*
* @details    領域拡張されたゲームカード相互認証秘密鍵を書き込みます。
*
* @param[in]  gamecardKey             書き込みたいゲームカード相互認証秘密鍵
*
* @return
*/
nn::Result WriteExtendedGameCardKey(const ExtendedGameCardKey& gamecardKey) NN_NOEXCEPT;

/**
* @brief      LCD ベンダ ID の書き換えをします。
*
* @details    LCD ベンダ ID の書き換えを書き込みます。
*
* @param[in]  lcdVendorId             書き込みたい LCD ベンダ ID
*
* @return
*/
nn::Result WriteLcdVendorId(const LcdVendorId& lcdVendorId) NN_NOEXCEPT;

/**
* @brief      領域拡張されたデバイス登録用秘密鍵 （RSA-2048 版） の書き換えをします。
*
* @details    領域拡張されたデバイス登録用秘密鍵 （RSA-2048 版）を書き込みます。
*
* @param[in]  deviceKey             書き込みたいデバイス登録用秘密鍵
*
* @return
*/
nn::Result WriteExtendedRsa2048DeviceKey(const ExtendedRsa2048DeviceKey& deviceKey) NN_NOEXCEPT;

/**
* @brief      デバイス登録用証明書 （RSA-2048 版） の書き換えをします。
*
* @details    デバイス登録用証明書 （RSA-2048 版）を書き込みます。
*
* @param[in]  deviceCertificate             書き込みたいデバイス登録用証明書
*
* @return
*/
nn::Result WriteRsa2048DeviceCertificate(const Rsa2048DeviceCertificate& rsa2048DeviceCertificate) NN_NOEXCEPT;

/**
* @brief      USB Type-C Power Source 回路バージョンの書き換えをします。
*
* @details    USB Type-C Power Source 回路バージョンを書き込みます。
*
* @param[in]  usbTypeCPowerSourceCircuit    書き込みたい USB Type-C Power Source 回路バージョン
*
* @return
*/
nn::Result WriteUsbTypeCPowerSourceCircuitVersion(const UsbTypeCPowerSourceCircuit& usbTypeCPowerSourceCircuit) NN_NOEXCEPT;

/**
* @brief      本体サブカラーの書き換えをします。
*
* @details    本体サブカラーを書き込みます。
*
* @param[in]  housingColor    書き込みたい本体サブカラー
*
* @return
*/
nn::Result WriteHousingSubColor(const HousingColor& housingColor) NN_NOEXCEPT;

/**
* @brief      本体液晶縁カラーの書き換えをします。
*
* @details    本体液晶縁カラーを書き込みます。
*
* @param[in]  housingColor    書き込みたい本体液晶縁カラー
*
* @return
*/
nn::Result WriteHousingBezelColor(const HousingColor& housingColor) NN_NOEXCEPT;

/**
* @brief      本体カラー 1 の書き換えをします。
*
* @details    本体カラー 1 を書き込みます。
*
* @param[in]  housingColor    書き込みたい本体カラー 1
*
* @return
*/
nn::Result WriteHousingMainColor1(const HousingColor& housingColor) NN_NOEXCEPT;

/**
* @brief      本体カラー 2 の書き換えをします。
*
* @details    本体カラー 2 を書き込みます。
*
* @param[in]  housingColor    書き込みたい本体カラー 2
*
* @return
*/
nn::Result WriteHousingMainColor2(const HousingColor& housingColor) NN_NOEXCEPT;

/**
* @brief      本体カラー 3 の書き換えをします。
*
* @details    本体カラー 3 を書き込みます。
*
* @param[in]  housingColor    書き込みたい本体カラー 3
*
* @return
*/
nn::Result WriteHousingMainColor3(const HousingColor& housingColor) NN_NOEXCEPT;

/**
* @brief      raw 領域の一括書き換えをします。
*
* @details    較正情報の raw 領域を一括で書き込みます。
*
* @param[in]  calibrationInfo             書き込みたい較正情報領域
*
* @return
*/
nn::Result PutCalibrationData(const RawCalibrationInfo& calibrationInfo) NN_NOEXCEPT;

/**
* @brief      フォーマットバージョンを取得します。
*
* @details    較正情報のフォーマットバージョンを取得します。
*
* @param[out]  pOutValue             フォーマットバージョン
*
* @return
*/
nn::Result ReadCalibrationInfoVersion(uint32_t* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体カラーバリエーションを取得します。
*
* @details    本体カラーバリエーションを取得します。
*
* @param[out]  pOutValue             本体カラーバリエーション
*
* @return
*/
nn::Result ReadColorVariation(ColorVariation* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体サブカラーを取得します。
*
* @details    本体サブカラーを取得します。
*
* @param[out]  pOutValue             本体サブカラー
*
* @return
*/
nn::Result ReadHousingSubColor(HousingColor* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体液晶縁カラーを取得します。
*
* @details    本体液晶縁カラーを取得します。
*
* @param[out]  pOutValue            本体液晶縁カラー
*
* @return
*/
nn::Result ReadHousingBezelColor(HousingColor* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体カラー 1 を取得します。
*
* @details    本体カラー 1 を取得します。
*
* @param[out]  pOutValue            本体カラー 1
*
* @return
*/
nn::Result ReadHousingMainColor1(HousingColor* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体カラー 2 を取得します。
*
* @details    本体カラー 2 を取得します。
*
* @param[out]  pOutValue            本体カラー 2
*
* @return
*/
nn::Result ReadHousingMainColor2(HousingColor* pOutValue) NN_NOEXCEPT;

/**
* @brief      本体カラー 3 を取得します。
*
* @details    本体カラー 3 を取得します。
*
* @param[out]  pOutValue            本体カラー 3
*
* @return
*/
nn::Result ReadHousingMainColor3(HousingColor* pOutValue) NN_NOEXCEPT;

/**
* @brief      新しいファイルを書き込みます。
*
* @details    FAT 領域の指定した新しいファイルを書き込みます。
*
* @param[in]  pWriteData             書き込みたいデータバッファ
* @param[in]  writeSize             writeData のバッファサイズ
* @param[in]  filePath              書き込み先のファイルパス
*
* @return
*/
nn::Result WriteFile(void* pWriteData, size_t writeSize, const ::std::string& filePath) NN_NOEXCEPT;

/**
* @brief      ファイルを読み込みます。
*
* @details    FAT 領域の指定されたファイルを読み込みます。
*
* @param[out]  pOutValue            読み込んだデータバッファ
* @param[in]   readSize             readData のバッファサイズ
* @param[in]   filePath             読み込みたいファイルのパス
*
* @return
*/
nn::Result ReadFile(void* pOutValue, size_t readSize, const ::std::string& filePath) NN_NOEXCEPT;

/**
* @brief      ファイルサイズを取得します。
*
* @details    FAT 領域の指定されたファイルのサイズを取得します。
*
* @param[out]  pOutValue            ファイルサイズ
* @param[in]   filePath             サイズを取得したいファイルのパス
*
* @return
*/
nn::Result GetFileSize(int64_t* pOutValue, const ::std::string& filePath) NN_NOEXCEPT;

/**
* @brief      新しいディレクトリを作成します。
*
* @details    FAT 領域の指定されたパスに新しいディレクトリを作成します。
*
* @param[in]  directoryPath             ディレクトリを作成するパス
*
* @return
*/
nn::Result CreateDirectory(const ::std::string& directoryPath) NN_NOEXCEPT;

/**
* @brief      ファイルを削除します。
*
* @details    FAT 領域の指定されたパスにあるファイルを削除します。
*
* @param[in]  filePath             削除したいファイルパス
*
* @return
*/
nn::Result DeleteFile(const ::std::string& filePath) NN_NOEXCEPT;

/**
* @brief      ディレクトリを削除します。
*
* @details    FAT 領域の指定されたディレクトリ以下にある全てのファイル、ディレクトリを削除します。
*
* @param[in]  directoryPath             削除したいディレクトリパス
*
* @return
*/
nn::Result DeleteDirectoryRecursively(const ::std::string& directoryPath) NN_NOEXCEPT;

/**
* @brief      ファイルの存在を確認します。
*
* @details    FAT 領域の指定されたパスに、ファイルが存在するか確かめます。
*
* @param[out] outIsExist           ファイルが存在するか？
* @param[in]  filePath             確認したいファイルのパス
*
* @return
*/
nn::Result IsFileExist(bool& outIsExist, const ::std::string& filePath) NN_NOEXCEPT;

/**
* @brief      ディレクトリの存在を確認します。
*
* @details    FAT 領域の指定されたパスに、ディレクトリが存在するか確かめます。
*
* @param[out] outIsExist                ディレクトリが存在するか？
* @param[in]  directoryPath             確認したいディレクトリのパス
*
* @return
*/
nn::Result IsDirectoryExist(bool& outIsExist, const ::std::string& directoryPath) NN_NOEXCEPT;

}}
