﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   本体機能向け API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>

/**
 * @namespace   nn::btm
 * @brief       システムプロセス向け btm API の名前空間です。
 */

/**
 * @namespace   nn::btm::system
 * @brief       システムソフトウェア向け btm API の名前空間です。
 */
namespace nn { namespace btm { namespace system {


//! @name 本体機能向け API
//! @{

/**
 * @brief       GamepadPairing の状態が変わった際にシグナルするシステムイベントを取得します
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 * @param[out]  pSystemEvent
 */
void AcquireGamepadPairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       コントローラを対象としたペアリング試行を開始します。
 * @details     ノンブロッキング API です。 Result にある事前条件確認ができた段階で処理が返ります。
 *              ペアリング試行の開始タイミングは GamepadPairingEvent で確認できます。
 *
 *              「本体にペアリングボタンがないプラットフォーム」や「複数のコントローラー登録においてユーザーに都度ボタンを押してもらう手間を省く」ことを目的とする機能です。
 *              ペアリングを成立させるためには、コントローラー側もペアリング試行を開始する必要があります。
 *              ペアリング試行中のコントローラーが見つかると、自動的にペアリング処理、ペアリング情報登録処理、接続処理が行われます。
 *
 *              ペアリング試行を終了するためには CancelGamepadPairing() を使用してください。
 *              本体スリープ移行時、ローカル通信開始時、Bluetooth 停波開始時は、自動でペアリング試行が終了します。時間経過による終了はしません。
 *
 *              コントローラーが最大接続数に達していると、追加のペアリングは発生しなくなります。
 *
 *              登録可能なペアリング情報の上限数はプラットフォーム毎に異なります。
 *              登録数が上限に達している場合、最も古いペアリング情報が削除され、新たなペアリング情報が登録されます。
 *
 *              Bluetooth コントローラーを登録できないプラットフォームでは、本関数は作用しません。
 * @retresult
 *   @handleresult{nn::ResultSuccess,            処理に成功しました。}
 *   @handleresult{nn::btm::system::ResultPairingFailureLdnEnabled,  ローカル通信中のため、ペアリング試行が開始できませんでした。}
 *   @handleresult{nn::btm::system::ResultPairingFailureRadioOff,  電波設定がOffのためペアリング試行を開始できませんでした。}
 * @endretresult
 */
nn::Result StartGamepadPairing() NN_NOEXCEPT;

/**
 * @brief       コントローラを対象としたペアリング試行を終了します。
 * @details     ノンブロッキングAPIです。
 *              ペアリング試行を終了します。終了タイミングは GamepadPairingEvent で確認できます。
 *              ペアリング試行が行われていないときは何も作用しません。
 *              ペアリング処理に入っているコントローラーについては、本関数が呼ばれても処理がキャンセルされず、登録と接続が行われます。
 */
void CancelGamepadPairing() NN_NOEXCEPT;

/**
 * @brief       コントローラを対象としたペアリング試行の動作状態を取得します
 * @details     コントローラを対象としたペアリング試行の動作状態を取得します
 * @return      現在の動作状態
 */
bool IsGamepadPairingStarted() NN_NOEXCEPT;

/**
 * @brief       ペアリング済のコントローラ数を取得します
 * @details     本体にペアリング登録されている Bluetooth コントローラの数を取得します
 * @return      Bluetooth コントローラの登録数
 */
int GetPairedGamepadCount() NN_NOEXCEPT;

/**
 * @brief       コントローラとのペアリング情報を削除します
 * @details     ノンブロッキングAPIです。呼び出しから数秒以内に削除、切断されます。
 *              本体に登録されているコントローラの ペアリング情報を全て削除します。
 *              本関数呼び出し時に Bluetooth 接続されているコントローラは全て切断されます。
 */
void ClearGamepadPairingDatabase() NN_NOEXCEPT;

/**
 * @brief       電波設定が変わった際にシグナルするシステムイベントを取得します
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。 取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 * @param[out]  pSystemEvent
 */
void AcquireRadioEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       電波設定を変更します
 * @details     ノンブロッキングAPIです。変更の完了タイミングは RadioEvent で確認できます。
 *              電波無効を指定すると、Bluetooth 通信が停止し、電波を出さない設定になります。  接続されているコントローラは、全て切断されます。
 *              電波有効を指定すると、Bluetooth 通信が再開します。コントローラとの再接続は、自動では行われません。
 *              システムシャットダウンを経由しても、以前の設定を保った状態で起動します。
 * @param[in]   enable                 電波設定
 */
void EnableRadio(bool enable) NN_NOEXCEPT;

/**
 * @brief       現在の電波設定を取得します
 * @details     現在の電波設定を取得します
 * @return      現在の電波設定
 */
bool IsRadioEnabled() NN_NOEXCEPT;

//! @}

}}} // namespace nn::btm::system

