﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Btm API
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/btm/btm_Types.h>

#include <nn/applet/applet_FundamentalTypes.h>

namespace nn { namespace btm {

//! @name SF機能
//! @{

/**
 * @brief       Btmサーバーとのセッションを取得します
 * @details     同期APIです。
 *              セッションはライブラリ内で保持されます。他のすべてのAPIより先に呼び出しておく必要があります。
 *              複数回呼び出した場合は、ライブラリ内で回数を記憶しますが、セッションは1つのみ保持します。
 */
void InitializeBtmInterface() NN_NOEXCEPT;

/**
 * @brief       Btmサーバーとのセッションを破棄します
 * @details     同期APIです。
 *              セッション破棄後は、再度セッションを取得するまでSF機能 以外のAPIは使用できません。
 *              InitializeBtmInterfaceを呼び出した回数分のFinalizeBtmInterfaceが必要です。
 */
void FinalizeBtmInterface() NN_NOEXCEPT;

//! @}



//! @name Btm機能
//! @{

/**
 * @brief       Btm の内部ステートを取得します
 * @details     同期APIです。
 * @param[out]  pState
 */
void GetState(BtmState* pState)  NN_NOEXCEPT;

/**
 * @brief       Btmプロセス, Bluetoothプロセス, Bluetoothチップの情報を取得します
 * @details     同期APIです。
 * @param[out]  pHostDeviceProperty
 */
void GetHostDeviceProperty(HostDeviceProperty* pHostDeviceProperty) NN_NOEXCEPT;

//! @}



//! @name Connected Device Condition機能
//! @{

/**
 * @brief       DeviceCondition変更タイミング取得用のシステムイベントを登録します
 * @details     同期APIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              サーバーのDeviceConditionが更新されるたび、シグナルされます。
 *              具体的には、Connected Device Condition 機能の Set系 API を呼んだ後、もしくはデバイスの接続と切断発生時にシグナルされます。
 *              デバイス接続時は、接続自体の完了時と、適切なDevice Condition に自動遷移時の、2回、シグナルが発生します。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *              イベントのクリアモードは自動クリアです。
 * @param[out]  pSystemEvent
 */
void RegisterSystemEventForConnectedDeviceCondition(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       DeviceConditionListを取得します
 * @details     同期APIです。
 *              サーバー内のDeviceConditionListを取得します。
 *              該当システムイベントのシグナル前に呼んだ場合は、その時点での Device Condition List が取得されます。
 * @param[out]  pList
 */
void GetConnectedDeviceCondition(DeviceConditionList* pList) NN_NOEXCEPT;

/**
 * @brief       指定デバイスのBurstModeを切り替えます
 * @details     （注意）このAPIの仕様は、現時点で確定していません。
 *              非同期APIです。
 *              変更を開始した、もしくは変更が開始できなかったタイミングで処理が返ります。
 * @param[in]   isBurstMode
 */
nn::Result SetBurstMode(const BdAddress* pAddress, bool isBurstMode) NN_NOEXCEPT;

/**
 * @brief       Bluetooth通信のスロットサイズを変更します
 * @details     非同期APIです。
 *              変更を開始した、もしくは変更が開始できなかったタイミングで処理が返ります。
 *              変更の完了、成否、変更に付随する Device Condition の変化は Device Condition List で確認できます。
 *              接続されていないデバイスに対する指示は無視されます。RadioOff中は ResultInvalidState が返ります。
 * @param[in]  pDeviceSlotModeList
 */
nn::Result SetSlotMode(const DeviceSlotModeList* pDeviceSlotModeList) NN_NOEXCEPT;

/**
 * @brief       Bluetoothの通信モードをbtmに設定します
 * @details     非同期APIです。
 *              変更を開始した、もしくは変更が開始できなかったタイミングで処理が返ります。
 *              変更の完了、成否、変更に付随する Device Condition の変化は Device Condition List で確認できます。
 * @param[in]   bluetoothMode
 */
nn::Result SetBluetoothMode(BluetoothMode bluetoothMode) NN_NOEXCEPT;

/**
 * @brief       WLANの通信モードをbtmに設定します
 * @details     非同期APIです。
 *              変更を開始した、もしくは変更が開始できなかったタイミングで処理が返ります。
 *              変更の完了、成否、変更に付随する Device Condition の変化は Device Condition List で確認できます。
 * @param[in]   wlanMode
 */
nn::Result SetWlanMode(WlanMode wlanMode) NN_NOEXCEPT;

//! @}


//! @name Registered Device Info機能
//! @{

/**
 * @brief       DeviceInfo 変更タイミング取得用のシステムイベントを登録します
 * @details     同期APIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              サーバーの DeviceInfo が更新されるたび、シグナルされます。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *              イベントのクリアモードは自動クリアです。
 * @param[out]  pSystemEvent
 */
void RegisterSystemEventForRegisteredDeviceInfo(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       ペアリング済みのデバイス情報を取得します
 * @details     同期APIです。
 *              該当システムイベントのシグナル前に呼んだ場合は、その時点での DeviceInfoList が取得されます。
 * @param[out]  pList
 */
void GetDeviceInfo(DeviceInfoList* pList) NN_NOEXCEPT;

/**
 * @brief       ペアリング情報をbtmに追加します
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              変更の完了、成否、変更に付随する Device Info の変化は Device Info List で確認できます。
 *              ペアリング済みデバイスの情報を追加した場合、上書きされます。
 *              ペアリングデバイス数が上限に達している場合、最も古い（もっとも昔に接続した）デバイスのペアリング情報が上書きされます。
 * @param[in]   pList
 */
void AddDeviceInfo(const DeviceInfo* pInfo) NN_NOEXCEPT;

/**
 * @brief       ペアリング情報をbtmから削除します。該当デバイスが接続されている場合、切断します
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              変更の完了、成否、変更に付随する Device Info の変化は Device Info List で確認できます。
 *              また、削除デバイスの切断は非同期です。切断の確認は、DeviceCondition の変化を確認してください。
 * @param[in]   pAddress
 */
void RemoveDeviceInfo(const BdAddress* pAddress) NN_NOEXCEPT;

/**
 * @brief       特定デバイスのペアリング情報の生存優先度を上げます
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              特定デバイスのペアリング情報の生存優先度を最大にします。
 *              指定した BdAddress を持つペアリング情報が存在しない場合、要求は無視されます。
 *              変更の完了、成否、変更に付随する Device Info の変化は Device Info List で確認できます。
 * @param[in]   pAddress
 */
void IncreaseDeviceInfoOrder(const BdAddress *pAddress) NN_NOEXCEPT;

//! @}



//! @name Power Mode機能
//! @{

/**
 * @brief       LLRの動作状態変化を取得するためのシステムイベントを登録します
 * @details     同期APIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              LLR が開始、終了するたび、シグナルされます。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *              イベントのクリアモードは自動クリアです。
 * @param[out]  pSystemEvent
 */
void AcquireLlrStateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       LLR の動作状態を取得します
 * @details     同期APIです。
 */
bool IsLlrStarted() NN_NOEXCEPT;

/**
 * @brief       ペアリング済みデバイスにLLR通知を送信します
 * @details     （注意）このAPIの仕様は、現時点で確定していません。
 *              同期APIです
 *              RadioOff中は ResultInvalidState が返ります。
 */
nn::Result LlrNotify(const BdAddress *pAddress) NN_NOEXCEPT;

/**
 * @brief       Bluetooth の電波設定を変更します
 * @details     同期APIです。
 *              RadioOffに付随する変化の内、全デバイスの切断まで同期します。その後のユースケース遷移には同期しません。
 *              同じ設定を複数回行った場合は、何も作用しません。
 *              不揮発メモリへの設定保存も行われます。
 */
void EnableRadio(bool isEnable) NN_NOEXCEPT;

/**
 * @brief       btmからの起用要求時にシグナルするシステムイベントを登録します
 * @details     同期APIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              シグナルは、MinimumAwakeステートおよび、システムとしての完全スリープからFullAwakeに遷移する途中のMinimumAwakeステートで発生し得ります。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *              イベントのクリアモードは自動クリアです。
 * @param[out]  pSystemEvent
 */
void RegisterSystemEventForAwakeReq(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       Bluetooth の通信頻度を下げます
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              変更は必ず成功します。変更の完了、変更に付随する Device Condition の変化は Device Condition List で確認できます。
 * @param[in]   isEnable
 */
void EnableSlotSaving(bool isEnable) NN_NOEXCEPT;

//! @}


//! @name DeviceTreatment 機能
//! @{

/**
 * @brief       指定したBdAddressを持つペアリング情報を、上限越えペアリングにおける自動削除からプロテクトするか否かを設定します
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              変更は必ず成功します。変更完了タイミングの取得はできません。
 *              プロテクト状態に設定すると、自動削除の対象から外れます。デフォルトは、非プロテクトです。
 *              この設定は Device Info List とは別に管理されます。当該デバイスのペアリング情報が無くとも設定可能です。設定値の取得は出来ません。
 *              設定は、このAPIにおける明示的な設定と、電源断でのみ変化します。ペアリング情報の削除では変化しません。
 *              設定数の上限は、プラットフォームにおけるペアリング数の上限と同様です。上限を超えて設定した場合、もっとも昔に設定した情報が上書きされます。
 *              自動削除できるペアリング情報が無い場合、この設定を無視して削除され得ります。
 * @param[in]   pAddress
 */
void ProtectDeviceInfo(const BdAddress* pAddress, bool isProtect) NN_NOEXCEPT;

//! @}


//! @name HID機能
//! @{

/**
 * @brief       特定のHIDデバイスを切断します
 * @details     非同期APIです。変更を要求した時点で処理が返ります。
 *              変更の完了、成否、変更に付随する Device Condition の変化は Device Condition List で確認できます。
 * @param[in]   pAddress
 */
void HidDisconnect(const BdAddress* pAddress) NN_NOEXCEPT;

/**
 * @brief       特定のHIDデバイスの再送モードを変更します
 * @details     非同期APIです。
 *              変更を開始した、もしくは変更が開始できなかったタイミングで処理が返ります。
 *              （注意）変更の完了、成否、変更に付随する Device Condition の変化は Device Condition List で確認できません。
 * @param[in]   pAddress
 * @param[in]   pZeroRetransmissionList
 */
nn::Result HidSetRetransmissionMode(const BdAddress* pAddress, const ZeroRetransmissionList* pZeroRetransmissionList) NN_NOEXCEPT;

//! @}

//! @name BLE Scan 機能
//! @{

/**
 * @brief       BLE Scan の状態が変わった際にシグナルするシステムイベントを取得します。
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleScanEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE スキャンのフィルタパラメータを取得します。
 *
 * @details     StartBleScanForGeneral() を使用してBLE デバイスをスキャンするために必要なパラメータを取得します。
 *              引数parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pFilter                         スキャンフィルタオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - 初期化済み
 *              - pFilter != nullptr
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultInvalidArgument,               引数の値が無効です。}
 * @endretresult
 */
nn::Result GetBleScanFilterParameter(user::BleAdvFilterForGeneral* pFilter, uint16_t parameterId) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスアプリを対象としたBLE スキャンのフィルタパラメータを取得します。
 *
 * @details     StartBleScanForSmartDevice() を使用してスマートデバイスアプリをスキャンするために必要なパラメータをシステムから取得します。
 *              引数parameterId に指定する値は、専用のヘッダに定義された値を使用してください。
 *
 * @param[out]  pFilter                         スキャンフィルタオブジェクトへのポインタ
 * @param[in]   parameterId                     BLE スキャンパラメータの識別子
 *
 * @pre
 *              - 初期化済み
 *              - pFilter != nullptr
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultInvalidArgument,               引数の値が無効です。}
 * @endretresult
 */
nn::Result GetBleScanFilterParameter(user::BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイス以外のBLE デバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              設定されたフィルタはStopBleScanGeneral() が呼ばれるまで有効であり、有効である間スキャンを継続します。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanGeneral() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、AcquireBleScanEvent() を使って取得したシステムイベントがシグナルされます。
 *              スキャン結果は、GetBleScanResultsForGeneral() を使って取得してください。
 *
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                  ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,       下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::ResultScanStartFailedFull,          スキャン対象が最大に達しています。}
 * @endretresult
 */
nn::Result StartBleScanForGeneral(const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       BLE デバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForGeneral() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForGeneral() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                  ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,       下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForGeneral() NN_NOEXCEPT;

/**
 * @brief       BLE Scan で発見されたデバイスを取得します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForGeneral() を使用して開始されたスキャンで発見されたデバイスを取得します。
 *              最大で (nn::bluetooth::BleScanResultCountMax / 2) 個の結果を取得することができます。
 *
 *
 * @param[out]  pResults                        スキャン結果を格納するための配列へのポインタ
 * @param[in]   inNum                           pResults の配列サイズ
 *
 * @pre
 *              - 初期化済み
 *              - pResults != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pResults にスキャン結果が格納されている。
 *
 * @return      取得したスキャン結果の数
 */
uint8_t GetBleScanResultsForGeneral(user::ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT;

/**
 * @brief       ペアリングされたBLE デバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイス以外の、ペアリングされたBLE デバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              設定されたフィルタはStopBleScanGeneral() が呼ばれるまで有効であり、有効である間スキャンを継続します。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanForPaired() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、接続を試行します。
 *              接続操作が完了すると、AcquireBleConnectionEvent() で取得したイベントがシグナルされます。
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                                   処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                          ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,               下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::ResultScanStartFailedFull,                  スキャン対象が最大に達しています。}
 *   @handleresult{nn::btm::ResultScanStartFailedPairingNotFound,       ペアリング済みBLE デバイスのスキャンを開始しようとしましたが、ペアリング情報が存在しません。}
 * @endretresult
 */
nn::Result StartBleScanForPaired(const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされたBLE デバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForPaired() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForPaired() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                      ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,           下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForPaired() NN_NOEXCEPT;

/**
 * @brief       スマートデバイスを対象としたBLE Scan を開始します。
 * @details     ブロッキング API です。
 *              スマートデバイスのアドバイスパケットをスキャンします。
 *              本API では、内部的にスキャンフィルタを設定し、指定のフォーマットに合致するアドバイスパケットのスキャンを有効にします。
 *              指定のフォーマットについては、別途ドキュメントを参照してください。
 *              スキャンフィルタの数には上限があり、それを超えるスキャンを同時に行うことはできません。
 *              スキャンフィルタの数が上限に達した場合、StopBleScanForSmartDevice() を使用して、スキャンフィルタを削除してください。
 *              スキャンフィルタの数の上限はプラットフォームごとに異なります。
 *
 *              アドバイスパケットを発見した場合、AcquireBleScanEvent() を使って取得したシステムイベントがシグナルされます。
 *              スキャン結果は、GetBleScanResultsForSmartDevice() を使って取得してください。
 *
 * @param[in]   filter                          スキャンフィルタのパラメータ
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                  ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,       下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 *   @handleresult{nn::btm::ResultScanStartFailedFull,          スキャン対象が最大に達しています。}
 * @endretresult
 */
nn::Result StartBleScanForSmartDevice(const user::BleAdvFilterForSmartDevice &filter) NN_NOEXCEPT;

/**
 * @brief       スマートデバイスを対象としたBLE Scan を停止します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForSmartDevice() によって開始されたスキャンを停止します。
 *              内部的に、StartBleScanForSmartDevice() によって有効化されたスキャンフィルタをすべて削除します。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                           処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                  ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 *   @handleresult{nn::btm::ResultFailureLowLayerGeneral,       下層（Bluetooth）が一般的なFailureを返しました。直前の処理は失敗しています。続行可能です。}
 * @endretresult
 */
nn::Result StopBleScanForSmartDevice() NN_NOEXCEPT;

/**
 * @brief       BLE Scan で発見されたデバイス一覧を取得します。
 * @details     ブロッキングAPIです。
 *              StartBleScanForSmartDevice() を使用して開始されたスキャンで発見されたデバイスを取得します。
 *              最大で (nn::bluetooth::BleScanResultCountMax / 2) 個の結果を取得することができます。
 *
 *
 * @param[out]  pResults                        スキャン結果を格納するための配列へのポインタ
 * @param[in]   inNum                           pResults の配列サイズ
 *
 * @pre
 *              - 初期化済み
 *              - pResults != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pResults にスキャン結果が格納されている。
 *
 * @return      取得したスキャン結果の数
 */
uint8_t GetBleScanResultsForSmartDevice(user::ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT;

//! @}

//! @name BLE 接続機能
//! @{

/**
 * @brief       BLE の接続状態が変わった際にシグナルするシステムイベントを取得します。
 * @details     ブロッキングAPIです。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       BLE の接続を試行します。
 * @details     ノンブロッキングAPIです。
 *              指定されたアドレスを持つデバイスに対し接続を試行します。
 *              接続の結果は、AcquireBleConnectionEvent() を使用して取得したシステムイベントを通じて行われます。
 *
 * @param[in]   address                         接続を試行するBLE デバイス、スマートデバイスのBluetooth アドレス
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::ResultBleConnectionFailedFull,          接続台数が最大に達しています。}
 * @endretresult
 */
nn::Result BleConnect(nn::bluetooth::Address address) NN_NOEXCEPT;

/**
 * @brief       既存の BLE の接続のオーナー情報をクリアします
 * @details     ブロッキングAPIです。
 *              指定された接続ハンドルを持つBLE 接続のオーナー情報を nn::applet::AppletResourceUserId::GetInvalidId() で上書きします。
 *              オーナー情報がクリアされたデバイスは、オーナーがフォアグラウンドでなくなっても接続が維持されます。
 *
 * @param[in]   connectionHandle                オーナー情報をクリアするBLE 接続の接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::ResultBleConnectionNotFound,            存在しない接続に対する操作です。}
 * @endretresult
 */
nn::Result BleOverrideConnection(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       BLE の接続を切断します。
 * @details     ノンブロッキングAPIです。
 *              指定された接続ハンドルを持つBLE 接続を切断します。
 *              切断の結果は、AcquireBleConnectionEvent() を使用して取得したシステムイベントを通じて行われます。
 *
 * @param[in]   connectionHandle                 切断するBLE 接続の接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::ResultBleConnectionNotFound,            存在しない接続に対する操作です。}
 * @endretresult
 */
nn::Result BleDisconnect(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       BLE の接続状態を返します。
 * @details     ブロッキングAPIです。
 *              最大でnn::bluetooth::BLE_GATT_CLIENT_NUM_MAX 個の接続状態を取得することができます。
 *
 * @param[out]  pConnState                      接続状態を格納するための配列へのポインタ
 * @param[in]   inNum                           pConnState の配列のサイズ
 *
 * @pre
 *              - 初期化済み
 *              - pConnState != nullptr
 *              - inNum != 0
 *
 * @return      取得した接続状態
 */
uint8_t BleGetConnectionState(user::BleClientConnState *pConnState, uint8_t inNum) NN_NOEXCEPT;

/**
 * @brief       GATT Client の詳細な接続情報を返します。
 * @details     ブロッキングAPIです。
 *
 * @param[out]  pGattClientConditionList        接続状態を格納するための配列へのポインタ
 *
 * @pre
 *              - 初期化済み
 *              - pGattClientConditionList != nullptr
 */
void BleGetGattClientConditionList(GattClientConditionList* pGattClientConditionList) NN_NOEXCEPT;


//! @}

//! @name GATT Service Discovery に関する機能
//! @{

/**
 * @brief       GATT Service Disvery が完了した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 *              BLE 接続が確立した後、自動的にGATT Service Discovery が実行され、GATT Attribute が見つかる度にシグナルされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server のGatt Service の一覧を取得します
 * @details     ブロッキングAPI です。
 *               GATT Service Discovery の完了前に呼び出すと、不完全なGatt Service の一覧が返る可能性があります。
 *
 * @param[out]  pServices                       取得したGatt Service を格納するための配列へのポインタ
 * @param[in]   inNum                           pServices の配列サイズ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pServices != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pServices にはGatt Service 一覧が格納されている
 *
 * @return      取得したGATT Service の数
 */
uint8_t GetGattServices(user::GattService *pServices, uint8_t inNum, uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の、指定したGatt Service に属するGatt Included Service の一覧を取得します
 * @details     ブロッキングAPI です。
 *              GATT Service Discovery の完了前に呼び出すと、不完全なGatt Included Service の一覧が返る可能性があります。
 *
 * @param[out]  pIncludedServices               取得したGatt Included Service を格納するための配列へのポインタ
 * @param[in]   inNum                           pIncludedServices の配列サイズ
 * @param[in]   connectionHandle                Gatt Included Service を取得するGatt Server との接続ハンドル
 * @param[in]   serviceHandle                   Gatt Included Service を取得するGatt Service のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pIncludedServices != nullptr
 *
 * @post
 *              - pIncludedServices にはGatt Service 一覧が格納されている
 *
 * @return      取得したGATT Service の数
 */
uint8_t GetGattIncludedServices(user::GattService *pIncludedServices, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したUUID を持つGatt Service を取得します
 * @details     ブロッキングAPI です。
 *               GATT Service Discovery の完了前に呼び出すと、false が返る可能性があります。
 *
 * @param[out]  pService                        取得したGatt Service を格納するためオブジェクトへのポインタ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 * @param[in]   uuid                            取得するGatt Service のUUID
 *
 * @pre
 *              - 初期化済み
 *              - pService != nullptr
 *
 * @post
 *              - 指定したUUID を持つGATT Service が存在する場合、pService にはGatt Service が格納されている
 *
 * @ret         指定したUUID を持つGATT Service が存在する場合 true、存在しない場合 false
 */
bool GetGattService(user::GattService *pService, uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したハンドルを持つGatt Attribute が所属する場合GATT Service を取得します
 * @details     ブロッキングAPI です。
 *               GATT Service Discovery の完了前に呼び出すと、false が返る可能性があります。
 *
 * @param[out]  pService                        取得したGatt Service を格納するためオブジェクトへのポインタ
 * @param[in]   connectionHandle                Gatt Service を取得するGatt Server との接続ハンドル
 * @param[in]   attributeHandle                 所属するGatt Service を取得したいGATT Attribute のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pService != nullptr
 *
 * @post
 *              - 指定したハンドルを持つGATT Attribute が所属する GATT Service が存在する場合、pService にはGatt Service が格納されている
 *
 * @ret         指定したハンドルを持つGATT Attribute が所属するGATT Service が存在する場合 true、存在しない場合 false
 */
bool GetBelongingGattService(user::GattService *pService, uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したGatt Service に属するGatt Characteristic の一覧を取得します
 * @details     ブロッキングAPI です。
 *               GATT Service Discovery の完了前に呼び出すと、不完全なGatt Characteristic の一覧が返る可能性があります。
 *
 * @param[out]  pCharacteristics                取得したGatt Characteristic を格納するための配列へのポインタ
 * @param[in]   inNum                           pCharacteristics の配列サイズ
 * @param[in]   connectionHandle                Gatt Characteristic を取得するGatt Server との接続ハンドル
 * @param[in]   serviceHandle                   Gatt Characteristic を取得するGatt Service のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pCharacteristics != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pCharacteristics にはGatt Characteristic 一覧が格納されている
 *
 * @ret         取得したGATT Characteristic の数
 */
uint8_t GetGattCharacteristics(user::GattCharacteristic *pCharacteristics, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT;

/**
 * @brief       接続されているGatt Server の指定したGatt Characteristic に属するGat Descriptor の一覧を取得します
 * @details     ブロッキングAPI です。
 *              GATT Service Discovery の完了前に呼び出すと、不完全なGatt Descriptor の一覧が返る可能性があります。
 *
 * @param[out]  pDescriptors                    取得したGatt Descriptor を格納するための配列へのポインタ
 * @param[in]   inNum                           pDescriptors の配列サイズ
 * @param[in]   connectionHandle                Gatt Descriptor を取得するGatt Server との接続ハンドル
 * @param[in]   charHandle                      Gatt Descriptor を取得するGatt Characteristic のハンドル
 *
 * @pre
 *              - 初期化済み
 *              - pDescriptors != nullptr
 *              - inNum != 0
 *
 * @post
 *              - pDescriptors にはGatt Descriptor 一覧が格納されている
 *
 * @ret         取得したGATT Descriptor の数
 */
uint8_t GetGattDescriptors(user::GattDescriptor *pDescriptors, uint8_t inNum, uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT;

//! @}

//! @name BLE ペアリング機能
//! @{

/**
 * @brief       BLE ペアリング処理が完了または処理中にエラーが発生した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBlePairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       接続されているGATT Server とペアリングします。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              最大nn::bluetooth::BLE_PAIRING_NUM_MAX 個のGATT Server とペアリングすることができます。
 *              ペアリング数が最大に達している場合、最も古いペアリング情報を削除します。
 *              すでにペアリング情報が存在する場合、本体内のペアリング情報を更新します。
 *              ペアリング処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   connectionHandle                ペアリングするGATT Server との接続ハンドル
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultBleConnectionNotFound,         接続が存在しません。}
 *  @handleresult{nn::btm::ResultBlePairingNotSupported,        ペアリング機能がサポートされていません。}
 *  @handleresult{nn::btm::ResultBlePairingOnGoing,             ペアリング処理中です。}
 * @endretresult
 */
nn::Result BlePairDevice(uint32_t connectionHandle, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       接続されているGATT Server とのペアリング情報を削除します。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              ペアリング情報が存在しないでも、GATT Server からペアリング情報を削除する処理を行います。
 *              ペアリング削除処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   connectionHandle                ペアリングを削除するGATT Server との接続ハンドル
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultBleConnectionNotFound,         接続が存在しません。}
 *  @handleresult{nn::btm::ResultBlePairingNotSupported,        ペアリング機能がサポートされていません。}
 *  @handleresult{nn::btm::ResultBlePairingOnGoing,             ペアリング処理中です。}
 * @endretresult
 */
nn::Result BleUnPairDevice(uint32_t connectionHandle, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       GATT Server とのペアリングを削除します。
 *
 * @detailes    本API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              指定したBluetooth アドレスを持つGATT Server と接続している場合、本体、及び、GATT Server からペアリング情報を削除する処理を行います。
 *              指定したBluetooth アドレスを持つGATT Server と接続していない場合は、本体からのみペアリング情報を削除します。
 *              ペアリング情報が存在しない場合でも、指定したBluetooth アドレスを持つGATT Server と接続されいる場合、そのGATT Server からペアリング情報を削除する処理を行います。
 *              ペアリング削除処理の成否を取得することはできません。
 *              BleGetPairedDevices() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *
 * @param[in]   bluetoothAddress                ペアリングを削除するGATT Server のBluetooth アドレス
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::btm::ResultPairingNotSupported,           ペアリング機能がサポートされていません。}
 *  @handleresult{nn::btm::ResultBlePairingOnGoing,             ペアリング処理中です。}
 * @endretresult
 */
nn::Result BleUnPairDevice(const nn::bluetooth::Address& address, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされているGATT Server のBluetooth アドレス一覧を取得します。
 *
 * @details     最大でBLE_PAIRING_NUM_MAX 個のBluetooth アドレスを取得します。
 *
 * @param[out]  pOutAddresses                   ペアリングされたGATT Server のBluetooth アドレスを格納するための配列へのポインタ
 * @param[in]   count                           pOutAddresses の数
 * @param[in]   BleAdvFilterForGeneral          スキャンフィルタのパラメータ
 *
 * @pre
 *              - pOutAddresses != nullptr
 *
 * @return      取得したBluetooth アドレスの数
 *
 */
uint8_t BleGetPairedDevices(nn::bluetooth::Address* pOutAddresses, uint8_t count, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT;

//! @}

//! @name その他のBLE 関連機能
//! @{

/**
 * @brief       GATT Server との間でMTU 設定が完了した際にシグナルされるシステムイベントを取得します。
 * @details     ブロッキングAPI です。
 *              サーバー内のシステムイベントのハンドルを、指定されたシステムイベントにアタッチします。
 *              イベントのクリアモードは自動です。
 *              取得可能なシステムイベントは1プロセスにつき1つです。上限を超えた場合、btmライブラリが  ABORT します。取得したシステムイベントは相互にクリアし合いません。
 *              連続でシグナルが発生した場合、古いシグナル状態は上書きされます。
 *
 * @param[out]  pSystemEvent
 *
 * @pre
 *              - 初期化済み
 *              - pSystemEvent != nullptr
 *              - *pSystemEvent は未初期化状態
 *
 * @post
 *              - *pSystemEvent は初期化済み
 */
void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT;

/**
 * @brief       GATT Server との間で、MTU 設定を行います。
 * @details     ノンブロッキングAPI です。
 *              MTU 設定が完了すると、AcquireBleMtuConfigEvent() で取得したイベントがシグナルされます。
 *              設定可能なMTU は最小で 23 Byte, 最大で 512 Byte です。
 *              同時に MTU を設定可能な接続は 1 つまでです。
 *
 * @param[in]   connectionHandle                MTU を設定するGATT Server との接続ハンドル
 * @param[in]   mtu                             設定するMTU
 *
 * @pre
 *              - 初期化済み
 */
nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT;

/**
 * @brief       GATT Server との間のMTU を取得します。
 * @details     ブロッキングAPI です。
 *              AcquireBleMtuConfigEvent() で取得したシステムイベントがシグナルされる前に呼び出すと、設定変更前のMTU が返る可能性があります。
 *              接続が見つからない場合、0 を返します。
 *
 * @param[in]   connectionHandle                MTU を取得するGATT Server との接続ハンドル
 *
 * @pre
 *              - 初期化済み
 *
 * @return      取得したMTU
 */
uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知先を登録します
 *
 * @details     指定したUUID を持つGATT Service に属するGATT Attribute を操作した際の結果に関するシステムイベントの通知先を登録します。
 *
 * @param[in]   uuid                            通知先を登録するGATT Service のUUID
 * @param[in]   path                            イベントの通知先識別子
 *
 * @pre
 *              - 初期化済み
 * @
 */
nn::Result RegisterBleGattDataPath(const user::BleDataPath& path) NN_NOEXCEPT;

/**
 * @brief       GATT Attribute に対する操作結果の通知先の登録を削除します
 *
 * @details     RegisterBleGattDataPath() を使用して登録された通知を削除します。
 *
 * @param[in]   uuid                            通知先登録を削除するGATT Service のUUID
 * @param[in]   path                            イベントの通知先識別子
 *
 * @pre
 *              - 初期化済み
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                               処理に成功しました。}
 *   @handleresult{nn::btm::ResultBusyWorking,                      ノンブロッキングAPIの指示、もしくは下層のイベント起因で動作中です。}
 * @endretresult
 * @
 */
nn::Result UnregisterBleGattDataPath(const user::BleDataPath& path) NN_NOEXCEPT;

//! @}

//! @name アプレット管理機能
//! @{

/**
 * @brief       アプレットリソースユーザ ID を登録します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象を BTM のサービス提供先として登録します。
 *
 * @pre
 *              - InitializeBtmInterface() 済み
 *
 * @param[in]   aruid                           アプレットリソースユーザ ID
 * @param[in]   appletId                        持つアプレット ID
 */
void RegisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid,
                                  nn::applet::AppletId appletId) NN_NOEXCEPT;

/**
 * @brief       アプレットリソースユーザ ID の登録を解除します。
 *
 * @details     アプレットリソースユーザ ID で指定された対象を BTM のサービス提供先から削除します。
 *
 * @pre
 *              - InitializeBtmInterface() 済み
 *              - RegisterAppletResourceUserId() でサービス提供先として登録されている
 *
 * @param[in]   aruid                           アプレットリソースユーザ ID
 */
void UnregisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

/**
 * @brief       インフォーカス状態にあるアプレットリソースユーザ ID をBTM に設定します
 *
 * @details     インフォーカス状態にあるアプレットのアプレットリソースユーザ ID を設定します。
 *              事前に RegisterAppletResourceUserId() によってサービスの提供先として登録されている必要があります。
 *              Bluetooth の接続のオーナーであるアプレット以外のアプレットがインフォーカス状態に遷移した場合、
 *              それらの接続を全て切断し、新規の接続を受け付けない状態に遷移します。
 *              ARUID の設定後、本API は即座に返ります。
 *
 * @pre
 *              - InitializeBtmInterface() 済み
 *              - RegisterAppletResourceUserId() でサービス提供先として登録されている
 *
 * @param[in]   aruid                           アプレットリソースユーザ ID
 */
void SetAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

//! @}

}} // namespace nn::btm

