﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       GATT Service クラスの宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/bluetooth/bluetooth_GattAttribute.h>

namespace nn { namespace bluetooth {

    class GattCharacteristic;

    /**
     * @brief       GATT Service を表すClass です
     *
     * @details     GATT Attribute クラスを継承します
     */
    class GattService : public GattAttribute
    {
    public:
        /**
         * @brief           コンストラクタです
         *
         * @details         全パラメータが無効な値で初期化された GATT Service オブジェクトを生成します
         */
        GattService()  NN_NOEXCEPT;

        /**
         * @brief           コンストラクタです
         *
         * @details         与えられたパラメータで初期化された GATT Service オブジェクトを生成します
         *
         * @param[in]       uuid                GATT Attribute UUID
         * @param[in]       handle              GATT Attribute Handle
         * @param[in]       connectionHandle    GATT Server との接続ハンドル
         * @param[in]       instanceId          Instance ID
         * @param[in]       endGroupHandle      End Group Handle
         * @param[in]       isPrimary           Primary Service であるかを示すフラグ
         */
        GattService(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle,
                    uint16_t instanceId, uint16_t endGroupHandle, bool isPrimary)  NN_NOEXCEPT;

        /**
         * @brief           デストラクタです
         */
        virtual ~GattService()  NN_NOEXCEPT;

        /**
         * @brief           Instance ID を取得します
         */
        uint16_t GetInstanceId() const  NN_NOEXCEPT;

        /**
         * @brief           End Group Handle を取得します
         */
        uint16_t GetEndGroupHandle() const  NN_NOEXCEPT;

        /**
         * @brief           Primary Service であるかを取得します
         */
        bool IsPrimaryService() const  NN_NOEXCEPT;

        /**
         * @brief           自身に所属する GATT Service (Included Service) の一覧を取得します
         *
         * @details         最大で @ref GattAttributeCountMaxClient 個の GATT Service を取得します。
         *
         * @param[out]      pOutServices                GATT Service の配列へのポインタ
         * @param[in]       count                       pOutServices の数
         *
         * @pre
         *                  - pOutServices != nullptr
         *
         * @return          取得したGATT Service の数
         */
        int GetIncludedServices(GattService* pOutServices, int count) const  NN_NOEXCEPT;

        /**
         * @brief           自身に所属する GATT Characteristic の一覧を取得します
         *
         * @details         最大で @ref GattAttributeCountMaxClient 個の GATT Characteristic を取得します。
         *
         * @param[out]      pOutCharacteristics         GATT Characteristic の配列へのポインタ
         * @param[in]       count                       pOutCharacteristics の数
         *
         * @pre
         *                  - pOutCharacteristics != nullptr
         *
         * @return          取得した GATT Characteristic の数
         */
        int GetCharacteristics(GattCharacteristic* pOutCharacteristics, int count) const  NN_NOEXCEPT;

        /**
         * @brief           自身に所属する GATT Characteristic のうち、指定した UUID を持つものを取得します
         *
         * @param[out]      pOutCharacteristics         GATT Characteristic オブジェクトへのポインタ
         * @param[in]       uuid                        取得したいGATT Characteristic のUUID
         *
         * @pre
         *                  - pOutCharacteristics != nullptr
         *
         * @return          指定した UUID を持つ GATT Characteristic が存在する場合は true。そうでない場合、false。
         */
        bool GetCharacteristic(GattCharacteristic* pOutCharacteristics, const GattAttributeUuid& uuid) const NN_NOEXCEPT;

    private:
        uint16_t    m_InstanceId;   //!< 同一の GATT Attribute UUID を持つ GATT Service が、同一の GATT Server 上に存在する場合に、それらを識別するために用いられるサブ識別子（Instance ID）です

        uint16_t    m_EndGroupHandle;   //!< 自身に所属する GATT Attribute が持ちうる最大のハンドルの値（End Group Handle）です

        bool        m_IsPrimary;    //!< Primary Service であるかを示すフラグです
    };
}}  // namespace nn::bluetooth
