﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/ipc/bcat_IpcService.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace bcat { namespace detail {

/*!
    @brief      シムライブラリのグローバルインスタンスです。

    @details
                シムライブラリのファイル間共通データを管理します。
*/
class ShimLibraryGlobal
{
private:
    /*!
        @brief      コンストラクタです。
    */
    ShimLibraryGlobal() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ShimLibraryGlobal& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ShimLibraryGlobal, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      インスタンスを初期化します。
    */
    void Initialize() NN_NOEXCEPT;

    /*!
        @brief      セッションを取得します。

        @return     セッション。
    */
    detail::ipc::IBcatService* GetSession() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをマウントします。

        @return     処理結果。
    */
    nn::Result MountDeliveryCacheStorage() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをマウントします。

        @param[in]  appId   アプリケーション ID。

        @return     処理結果。
    */
    nn::Result MountDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをアンマウントします。
    */
    void UnmountDeliveryCacheStorage() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージをマウントしているかどうかを確認します。
    */
    bool IsDeliveryCacheStorageMounted() NN_NOEXCEPT;

     /*!
        @brief      DeliveryCacheFileService を作成します。

        @param[out] outService  DeliveryCacheFileService。

        @return     処理結果。
    */
    nn::Result CreateFileService(detail::ipc::IDeliveryCacheFileService** outService) NN_NOEXCEPT;

    /*!
        @brief      DeliveryCacheDirectoryService を作成します。

        @param[out] outService  DeliveryCacheDirectoryService。

        @return     処理結果。
    */
    nn::Result CreateDirectoryService(detail::ipc::IDeliveryCacheDirectoryService** outService) NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュファイルのカウントをインクリメントします。
    */
    void IncrementDeliveryCacheFileCount() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュファイルのカウントをデクリメントします。
    */
    void DecrementDeliveryCacheFileCount() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュディレクトリのカウントをインクリメントします。
    */
    void IncrementDeliveryCacheDirectoryCount() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュディレクトリのカウントをデクリメントします。
    */
    void DecrementDeliveryCacheDirectoryCount() NN_NOEXCEPT;

    /*!
        @brief      データ配信キャッシュストレージのディレクトリを列挙します。

        @param[out] outCount            取得したディレクトリ名数。
        @param[out] outDirectoryNames   ディレクトリ名リスト。
        @param[in]  count               ディレクトリ名リストの要素数。

        @return     処理結果。
    */
    nn::Result EnumerateDeliveryCacheDirectory(int* outCount, DirectoryName* outDirectoryNames, int count) NN_NOEXCEPT;

private:
    //
    nn::os::SdkMutex m_Mutex;
    //
    detail::ipc::IBcatService* m_Session;
    detail::ipc::IDeliveryCacheStorageService* m_DeliveryCacheStorage;
    //
    int m_DeliveryCacheFileCount;
    int m_DeliveryCacheDirectoryCount;

private:
    //
    void CreateIpcSession() NN_NOEXCEPT;
};

}}}
