﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   アプレットマネージャ専用の型や定数の宣言
 *
 * @details
 *  本ヘッダでは、アプレットマネージャが使用する型や定数などを定義しています。
 *
 *  本ヘッダで定義している定数や列挙子の種類や値は暫定です。
 *  将来的に変更になる可能性があります。
 *
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>

namespace nn { namespace applet { namespace detail {
    struct LibraryAppletHandleObject;
}}}

namespace nn { namespace am { namespace service {
    class ILibraryAppletAccessor;
}}}

namespace nn { namespace applet {

//-----------------------------------------------------------------------------
/**
 * @brief   同時に起動可能なアプリケーションの最大数を示す定数値です。
 *
 */
const int LaunchableApplicationCountMax = 1;


//-----------------------------------------------------------------------------
/**
 * @brief   同時に起動可能なシステムアプレットの最大数を示す定数値です。
 *
 */
const int LaunchableSystemAppletCountMax = 1;


//-----------------------------------------------------------------------------
/**
 * @brief   起動時パラメータの最大サイズを示す定数値です。
 *
 */
const size_t StartupParamSizeMax = 8192;


//-----------------------------------------------------------------------------
/**
 * @brief   終了コードの最大サイズを示す定数値です。
 *
 */
const size_t ReturnValueSizeMax = 8192;


//-----------------------------------------------------------------------------
/**
 * @brief   要求コマンドの最大サイズを示す定数値です。
 *
 */
const size_t RequestCommandSizeMax = 8192;


//-----------------------------------------------------------------------------
/**
 * @brief   返答メッセージの最大サイズを示す定数値です。
 *
 */
const size_t ReplyMessageSizeMax = 8192;


/**
    @brief   アプレットやアプリケーションの識別情報を格納する構造体です。

    @details
     - appletId が AppletId_Application のとき、applicationId で表されるアプリケーションを表します
     - appletId が AppletId_Application でないとき、その appletId に対応するアプレットであることを表します
*/
struct AppletIdentityInfo
{
    AppletId appletId; //!< アプレットの ID
    union
    {
        ncm::ApplicationId applicationId; //!< アプリケーションの ID (アプリケーションの場合にのみ有効)
        ncm::SystemApplicationId systemApplicationId; //!< システムアプリケーションの ID (システムアプリケーションの場合にのみ有効)
    };

    // 以下、ユーティリティアクセッサ関数

    bool IsApplication() const NN_NOEXCEPT
    {
        return this->appletId == AppletId_Application;
    }

    bool IsSystemApplication() const NN_NOEXCEPT
    {
        return this->appletId == AppletId_SystemApplication;
    }

    static AppletIdentityInfo Make(AppletId appletId) NN_NOEXCEPT
    {
        AppletIdentityInfo ret = {};
        ret.appletId = appletId;
        return ret;
    }

    static AppletIdentityInfo Make(ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        AppletIdentityInfo ret = {};
        ret.appletId = AppletId_Application;
        ret.applicationId = applicationId;
        return ret;
    }

    static AppletIdentityInfo Make(ncm::SystemApplicationId systemApplicationId) NN_NOEXCEPT
    {
        AppletIdentityInfo ret = {};
        ret.appletId = AppletId_SystemApplication;
        ret.systemApplicationId = systemApplicationId;
        return ret;
    }
};


/**
    @brief   アプリケーションの起動要求元情報を格納する構造体です。

    @details
     - IsApplicationChain() == true の時のみ、applicationId に対象アプリケーションの ApplicationId が入ります

*/
struct ApplicationLaunchRequestInfo
{
    AppletId requestingAppletId;        //!< アプリ起動を要求しているアプレットの AppletId
    AppletId requestingMainAppletId;    //!< アプリ起動要求元のメインアプレットの AppletId
    ncm::ApplicationId applicationId;   //!< ApplicationId (IsApplicationChain() == true の場合に有効）

    //!< アプリ起動要求元がフローティング起動か否かを返します。
    bool IsFloatingApplication() const NN_NOEXCEPT
    {
        return this->requestingMainAppletId == AppletId_None;
    }

    //!< アプリ起動要求元がアプリチェインか否かを返します。
    bool IsApplicationChain() const NN_NOEXCEPT
    {
        return this->requestingMainAppletId == AppletId_Application;
    }
};


}} // namespace nn::applet

