﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/am/service/core/am_Applet.h>

#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Mutex.h>
#include <type_traits>
#include <nn/am/service/core/am_StackableApplet.h>

namespace nn { namespace am { namespace service { namespace core {

namespace detail {

class ThreadImplementedAppletImpl
{
protected:

    explicit ThreadImplementedAppletImpl(StackableApplet* pApplet) NN_NOEXCEPT;
    ~ThreadImplementedAppletImpl() NN_NOEXCEPT;

    // for ThreadImplementedApplet<typename>
    void StartImpl() NN_NOEXCEPT;
    bool TryJoinImpl() NN_NOEXCEPT;
    void InitializeMultiWaitHolderImpl(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
    Result GetResultImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

private:

    // pure
    virtual Result Run() NN_NOEXCEPT = 0;

    // for StartImpl()
    Result StartImpl2() NN_NOEXCEPT;

    StackableApplet* m_pApplet;
    os::Mutex m_Mutex{false};
    os::ThreadType m_Thread;
    void* m_Stack{nullptr};
    bool m_Started{false};
    bool m_ThreadRunning{false};
    Result m_Result;

};

/**
    @brief 独自スレッドを持つアプレットの実装ユーティリティ

    @tparam BaseApplet StackableApplet であることが必要

    @details
     Run() を実行するスレッドを作成し、このスレッドの生存をアプレットとみなす。

     - Applet への Start は、スレッドの Start
     - スレッドの終了は、Applet の Joinable
*/
template <typename BaseApplet>
class ThreadImplementedApplet
    : public BaseApplet
    , private ThreadImplementedAppletImpl
{
    static_assert(std::is_base_of<StackableApplet, BaseApplet>::value, "BaseApplet is not base of Applet.");
protected:

    ThreadImplementedApplet()
        : ThreadImplementedAppletImpl(static_cast<StackableApplet*>(this))
    {
    }

private:

    typedef ThreadImplementedAppletImpl Impl;

    // overrides Applet

    virtual void Start() NN_NOEXCEPT NN_OVERRIDE final
    {
        Impl::StartImpl();
    }

    virtual bool TryJoinOne() NN_NOEXCEPT NN_OVERRIDE final
    {
        return Impl::TryJoinImpl();
    }

    virtual void InitializeMultiWaitHolderOne(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT NN_OVERRIDE final
    {
        Impl::InitializeMultiWaitHolderImpl(pHolder);
    }

    virtual Result GetResult() NN_NOEXCEPT NN_OVERRIDE final
    {
        return Impl::GetResultImpl();
    }

    // pure
    virtual Result Run() NN_NOEXCEPT NN_OVERRIDE = 0;

};

} // detail

}}}}
