﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <memory>
#include <nn/sf/sf_Types.h>

#include <nn/am/service/am_Foundation.sfdl.h>

// for Storage Channel
#include <nn/util/util_Optional.h>
#include <list>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/am_Result.public.h>

namespace nn { namespace am { namespace service {

class StorageBody
{
public:
    virtual ~StorageBody() NN_NOEXCEPT {}

    enum class Kind
    {
        Memory,
        TransferMemory,
    };
    virtual Kind GetKind() NN_NOEXCEPT = 0; // この値を Open～ でチェックする
};

class MemoryStorageBody
    : public StorageBody
{
public:
    virtual int64_t GetSize() NN_NOEXCEPT = 0;
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT = 0;
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT = 0;
    virtual Kind GetKind() NN_NOEXCEPT NN_OVERRIDE
    {
        return Kind::Memory;
    }
};

class TransferMemoryStorageBody
    : public StorageBody
{
private:
    sf::NativeHandle m_Handle;
    size_t m_Size;

public:
    explicit TransferMemoryStorageBody(sf::NativeHandle handle, size_t size) NN_NOEXCEPT
        : m_Handle(std::move(handle))
        , m_Size(size)
    {
    }
    sf::NativeHandle GetHandle() NN_NOEXCEPT
    {
        return m_Handle.GetShared();
    }
    int64_t GetSize() NN_NOEXCEPT
    {
        return m_Size;
    }
    virtual Kind GetKind() NN_NOEXCEPT NN_OVERRIDE
    {
        return Kind::TransferMemory;
    }
};

sf::SharedPointer<IStorage> MakeTransferMemoryStorage(std::shared_ptr<TransferMemoryStorageBody> pBody) NN_NOEXCEPT;
sf::SharedPointer<IStorage> MakeMemoryStorage(std::shared_ptr<MemoryStorageBody> pBody) NN_NOEXCEPT;

Result MakeSimpleMemoryStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, size_t size) NN_NOEXCEPT;
Result MakeLargeStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle handle, size_t size, bool isWritable) NN_NOEXCEPT;
Result MakeHandleStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle handle, size_t size) NN_NOEXCEPT;

class StorageChannel
{
public:

    explicit StorageChannel(bool isFifo) NN_NOEXCEPT
        : m_IsFifo(isFifo)
    {
    }

    void Push(sf::SharedPointer<IStorage> p) NN_NOEXCEPT
    {
        return PushImpl(!m_IsFifo, std::move(p));
    }

    sf::SharedPointer<IStorage> Pop() NN_NOEXCEPT;

    void Unpop(sf::SharedPointer<IStorage> p) NN_NOEXCEPT
    {
        PushImpl(true, std::move(p));
    }

    void Clear() NN_NOEXCEPT;

    sf::NativeHandle GetPopEventHandle() NN_NOEXCEPT;

private:

    bool m_IsFifo;
    os::Mutex m_Mutex{false};
    util::optional<os::SystemEvent> m_PopEvent;
    std::list<sf::SharedPointer<IStorage>, StaticAllocator<sf::SharedPointer<IStorage>>> m_Queue;

    void PushImpl(bool toFront, sf::SharedPointer<IStorage> p) NN_NOEXCEPT;

};

}}}
