﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/vi_HotplugState.h>
#include <nn/vi/vi_PowerState.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/sf/sf_Out.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/applet/applet_Types.h>
#include <nn/am/service/am_CaptureBufferIndex.h>
#include <nn/am/service/display/am_IntegratedDisplayLayerHandle.h>
#include <nn/am/service/display/am_ScreenShotRequest.h>
#include <nn/capsrv/capsrv_ScreenShotAttribute.h>
#include <nn/oe/oe_SelfControlApis.h>

namespace nn { namespace am { namespace service {

    static const int ApplicationCopyrightTextureWidthMax  = nn::oe::CopyrightFrameBufferWidth;
    static const int ApplicationCopyrightTextureHeightMax = nn::oe::CopyrightFrameBufferHeight;

    void InitializeDisplayLayerControl() NN_NOEXCEPT;
    Result CreateIndirectLayer(vi::IndirectLayerHandleType* pOut) NN_NOEXCEPT;
    void DestroyIndirectLayer(vi::IndirectLayerHandleType layerHandle) NN_NOEXCEPT;
    Result CreateIndirectLayerConsumerEndPoint(vi::IndirectConsumerHandleType* pOut, vi::IndirectLayerHandleType layerHandle, applet::AppletResourceUserId userAruid) NN_NOEXCEPT;
    void DestroyIndirectLayerConsumerEndPoint(vi::IndirectLayerHandleType layerHandle, vi::IndirectConsumerHandleType) NN_NOEXCEPT;

// @name ディスプレイ制御モード切替
// @{

    // システムモードに変更します。
    // OMM のみ出画可能です。
    // 各アプレットの絵が出力されなくなります。
    void ChangeDisplayLayerControlModeToSystem() NN_NOEXCEPT;

    // アプレットモードに変更します。
    // 各アプレットの絵が出力されます。
    // OMM の出画は禁止されます。
    void ChangeDisplayLayerControlModeToApplet() NN_NOEXCEPT;
// @}

// @name ディスプレイレイヤ操作
// @{
    typedef display::IntegratedDisplayLayerHandle IntegratedDisplayLayerHandle;
    bool IsSystemBufferSharingEnabled() NN_NOEXCEPT;

    IntegratedDisplayLayerHandle CreateIntegratedDisplayLayer() NN_NOEXCEPT;
    void DestroyIntegratedDisplayLayer(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    bool IsIntegratedDisplayLayerUserSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    bool IsIntegratedDisplayLayerDestinationSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    bool IsIntegratedDisplayLayerBufferOwnerSpecified(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    bool SpecifyIntegratedDisplayLayerUserApplication(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerUserFullscreenApplet(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerUserPartialApplet(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerUserOverlay(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerUserNoDisplay(IntegratedDisplayLayerHandle h, applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    bool SpecifyIntegratedDisplayLayerDestinationDisplay(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerDestinationIndirect(IntegratedDisplayLayerHandle h, nn::vi::IndirectLayerHandleType indirectLayerHandle) NN_NOEXCEPT;

    bool SpecifyIntegratedDisplayLayerBufferOwnerApplet(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    bool SpecifyIntegratedDisplayLayerBufferOwnerSystemShared(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    nn::vi::fbshare::SharedBufferHandle AcquireIntegratedDisplayLayerSystemSharedBufferHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    // アプレットのスクリーンショット可否フラグを取得する関数を設定する。
    // 設定しなかった場合または func に nullptr を設定した場合、常に禁止状態として扱われる。
    void SetIntegratedDisplayLayerScreenShotPremissionGetterFunction(IntegratedDisplayLayerHandle h, bool (*func)(void*), void* userPtr) NN_NOEXCEPT;

    // Destination == Display かつ BufferOwner == Applet の場合有効なハンドルを返す。
    // それ以外の場合、 0 を返す。
    nn::vi::LayerId GetIntegratedDisplayLayerLayerId(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    // Destination == Indirect かつ BufferOwner == Applet の場合有効なハンドルを返す。
    // それ以外の場合、無効値を返す。
    nn::vi::IndirectProducerHandleType GetIntegratedDisplayLayerIndirectProducerHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    // BufferOwner == SystemShared の場合有効なハンドルを返す。
    // それ以外の場合、無効値を返す。
    nn::vi::fbshare::SharedLayerHandle GetIntegratedDisplayLayerSharedLayerHandle(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    void SetIntegratedDisplayLayerVisibility(IntegratedDisplayLayerHandle h, bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT;

    // 指定したレイヤが Foreground を失う際に遷移レイヤを表示するかを設定する。
    // Fullscreen のレイヤ以外では無視される。
    // 遷移レイヤを表示しない場合でも LastForeground のキャプチャは更新される。
    void SetIntegratedDisplayLayerTransitionLayerRequested(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT;

    // 指定したレイヤが Foreground の際に Conductor にするかを設定する。
    // この API を呼んだ時点で既に Foreground だった場合即時反映される。
    void SetIntegratedDisplayLayerAsConductorWhileForeground(IntegratedDisplayLayerHandle h, bool isConductor) NN_NOEXCEPT;

    // レイヤを動画撮影用レイヤスタックに挿入する
    // - User が Application
    // - Destination が Display
    // - BufferOwner が Applet
    // ※ Visibility に関係なくレイヤスタックに入る。
    void EnableIntegratedDisplayLayerToRecordingLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    void DisableIntegratedDisplayLayerToRecordingLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    // 以下の条件を満たした際にレイヤをデバッグ用のレイヤスタックに挿入する
    // - User が Application
    // - Destination が Display
    // - BufferOwner が Applet
    // ※ Visibility に関係なくレイヤスタックに入る。
    void EnableIntegratedDisplayLayerToDebugLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;
    void DisableIntegratedDisplayLayerToDebugLayerStackForApplication(IntegratedDisplayLayerHandle h) NN_NOEXCEPT;

    // @name システムレイヤ管理
    // @{
    void SetSystemLayerVisible(bool isVisible) NN_NOEXCEPT;
    void ClearTransitionLayerImage(Bit32 clearColor) NN_NOEXCEPT;
    // @}

    // @name OMM 用
    // @{

    // OMM からの描画の開始時に呼ぶ。
    void BeginOperationModeDisplay() NN_NOEXCEPT;

    // OMM からの描画の終了時に呼ぶ。
    // 表示中のテクスチャはすべて表示終了される。
    void EndOperationModeDisplay() NN_NOEXCEPT;

    // OMM 用のテクスチャに画像を書き込む。
    void SetOperationModeTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT;

    // OMM 用のテクスチャをディスプレイに出す。
    // 表示を終了するときは HideOperationModeTexture() を呼ぶ。
    void ShowOperationModeTexture() NN_NOEXCEPT;

    // 起動ロゴのテクスチャをディスプレイに出す。
    // 起動時以外に呼んだ場合、表示される内容は不定。
    // 表示を終了するときは HideOperationModeTexture() を呼ぶ。
    void ShowOperationModeStartupLogoTexture() NN_NOEXCEPT;

    // OMM 用のテクスチャをディスプレイから除外する。
    void HideOperationModeTexture() NN_NOEXCEPT;

    // @}

    // @name 権利表記レイヤ管理
    // @{
    nn::Result InitializeApplicationCopyrightSharedBuffer(nn::sf::NativeHandle& hTrmem, size_t trmemSize, int width, int height) NN_NOEXCEPT;
    void FinalizeApplicationCopyrightSharedBuffer() NN_NOEXCEPT;
    nn::Result SetApplicationCopyrightTexture(int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT;
    bool IsApplicationCopyrightTextureSet() NN_NOEXCEPT;
    void GetApplicationCopyrightInfoForScreenShot(nn::capsrv::ScreenShotAttribute* pOutValue) NN_NOEXCEPT;
    void SetApplicationCopyrightVisibility(bool isVisible) NN_NOEXCEPT;
    // @}

    //@name 画面写真撮影
    // @{
    void RequestScreenShotTask(const display::ScreenShotRequest& request) NN_NOEXCEPT;
    // @}

// @}

// @name キャプチャバッファ操作
// @{
    static const Bit32 CaptureBufferColorBlack = 0xFF000000;

    nn::Result AcquireCaptureBufferIndex(bool* pOutIsScreenShotEnabled, int* pOutValue, IntegratedDisplayLayerHandle h, CaptureBufferIndex index) NN_NOEXCEPT;
    nn::Result ReleaseCaptureBufferIndex(IntegratedDisplayLayerHandle h, CaptureBufferIndex index) NN_NOEXCEPT;

    nn::Result TryCaptureCurrentForegroundImage(CaptureBufferIndex index) NN_NOEXCEPT;
    nn::Result TryCaptureCurrentLayerImage(CaptureBufferIndex index, IntegratedDisplayLayerHandle handle, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT;

    nn::Result CopyImageFromCaptureBuffer(bool* pOutIsScreenShotEnabled, void* dst, size_t dstSize, CaptureBufferIndex index) NN_NOEXCEPT;
    nn::Result ClearCaptureBufferImage(CaptureBufferIndex index, Bit32 color, bool isCaptureEnabled) NN_NOEXCEPT;
    nn::Result CopyBetweenCaptureBuffersImpl(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT;

// @}

// @name サポート終了した操作
// @{

    // キャプチャバッファの TransferMemory のロックを取る。
    bool TryAcquireCaptureBufferSemaphore() NN_NOEXCEPT;
    void ReleaseCaptureBufferSemaphore() NN_NOEXCEPT;

    // キャプチャバッファを TransferMemory 化して返す。
    nn::Result AcquireCaptureBufferTransferMemory(bool* pOutIsScreenShotEnabled, nn::sf::Out<nn::sf::NativeHandle> pOutTransferMemory, CaptureBufferIndex index) NN_NOEXCEPT;
    nn::Result ReleaseCaptureBufferTransferMemory(CaptureBufferIndex index) NN_NOEXCEPT;

// @}

// for nn::idle
Result SetDisplayCmuLuma(const char* displayName, float targetLuma) NN_NOEXCEPT;

// for nn::omm
Result OpenDisplay(vi::DisplayId* pOutDisplayId, const char* displayName) NN_NOEXCEPT;
Result CloseDisplay(vi::DisplayId displayId) NN_NOEXCEPT;
Result GetDisplayHotplugState(vi::HotplugState* pOutValue, vi::DisplayId displayId) NN_NOEXCEPT;
Result GetDisplayHotplugEvent(os::SystemEventType* pOutValue, vi::DisplayId displayId) NN_NOEXCEPT;
Result SetDisplayAlpha(vi::DisplayId displayId, float alpha) NN_NOEXCEPT;
Result SetDisplayPowerState(vi::DisplayId displayId, vi::PowerState value) NN_NOEXCEPT;
Result GetDisplayResolution(int* pOutWidth, int* pOutHeight, vi::DisplayId displayId) NN_NOEXCEPT;
Result SetDefaultDisplay(vi::DisplayId displayId) NN_NOEXCEPT;


}}}
