﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    hdr.h
 * @brief   AHID HDR API
 */

#include <nn/nn_Common.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName

#include <nn/ahid/hdr/sfdl/ISession.sfdl.h>
#include <nn/ahid/hdr/hdr_Types.h>

namespace nn {
namespace ahid {
namespace hdr {

const int HdrEntriesCountMax = 32;

class Hdr
{

    NN_DISALLOW_COPY(Hdr);
    NN_DISALLOW_MOVE(Hdr);

public:

    Hdr() NN_NOEXCEPT {};
    ~Hdr() NN_NOEXCEPT {};

    /**
     * @brief Prepare Hdr instance for runtime use.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize() was called.
     * @post
     *   Hdr instance is ready for operation.
     * @details
     *   This initializes the instance of shim layer.
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Prepare Hdr instance for runtime use with DFC.
     * @param[in] session
     *  Caller specified nn::ahid::hdr::ISession.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize() was called.
     * @post
     *   Hdr instance is ready for operation.
     * @details
     *   This initializes the instance of shim layer with caller specified session. The caller and
     *   HDR server must be in the same process.
     */
    Result InitializeWith(nn::sf::SharedPointer<nn::ahid::hdr::ISession> session) NN_NOEXCEPT;

    /**
     * @brief Finalize Hdr instance for tunrime use.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has already been called.
     * @post
     *   Hdr instance can no longer be used uless Initialize is called again.
     * @details
     *   This finalizes the instance of shim layer.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Retrieves number of device entries.
     * @param[out] pOutEntries
     *  Pointer to receive the number of device entries.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   Number of attached AHID devices are returned.
     */
    Result GetDeviceEntries(size_t* pOutEntries) NN_NOEXCEPT;

    /**
     * @brief Retrieves array of device handles.
     * @param[out] pOutEntries
     *  Pointer to number of entries in array, specifies number of valid handles when function returns.
     * @param[in] handleEntries
     *  Specifies number of handles allocated in array.
     * @param[in] pHandles
     *  Pointer to stroage for receiving array of device handles.
     * @param[in] pFilter
     *  Pointer to attach filter for filtering entries to retrieve whether by VID PID or attach time.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   Array of HDR device handles is returned in pOutEntries, the handles are used to get device parameters.
     */
    Result GetDeviceList(size_t* pOutEntries, size_t handleEntries, DeviceHandle* pHandles, AttachFilter* pFilter) NN_NOEXCEPT;

    /**
     * @brief Retrieves DeviceParameters for device associated to DeviceHandle.
     * @param[out] pOutParameters
     *  Pointer to storage for DeviceParameters to retrieve.
     * @param[in] handle
     *  Device handle.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidHandle}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   Get DeviceParameters for device associated to DeviceHandle. The DeviceParameters contain information about the device. The device driver selects device to uase with AHID based on parameters.
     */
    Result GetDeviceParameters(DeviceParameters* pOutParameters, DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief Attach new device to HDR.
     * @param[out] pOutHandle
     *  HDR handle for device attached.
     * @param[in] pParameters
     *  DeviceParameters specifying parameters for device to be attached.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultMaxEntries}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   AttachDevice is only used by AHID servers to attach new device to HDR.
     */
    Result AttachDevice(DeviceHandle* pOutHandle, DeviceParameters* pParameters) NN_NOEXCEPT;

    /**
     * @brief Detach device from HDR.
     * @param[in] handle
     *  DeviceHandle associated to device to be detached from HDR.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultMaxEntries}
     *   @handleresult{ResultInvalidHandle}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   DetachDevice is only used by AHID servers to detach new device to HDR.
     */
    Result DetachDevice(DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief Set HDR device filter.
     * @param[in] pDeviceFilter
     *  Pointer to DeviceFilter array.
     * @param[in] entries
     *  Specifies number of DeviceFilter in array.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultFunctionNotSupported}
     *   @handleresult{ResultInvalidParameter}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Internal state of Hdr instance is not affected.
     * @details
     *   SetDeviceFilterForHipc sets DeviceFilter to HDR. The DeviceFilter
     *   filters attached device list per entries in pDeviceFilter. This
     *   function is only callable from DFC client. At runtime, the HDR will read the
     *   DeviceFilter from address provided by caller.
     */
    Result SetDeviceFilterForHipc(DeviceFilter *pDeviceFilter, size_t entries) NN_NOEXCEPT;

private:

    nn::sf::SharedPointer<ISession> m_Client;

};

} // end of namespace hdr
} // end of namespace ahid
} // end of namespace nn
