﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    ahid_CodeBook.h
 * @brief   AHID Code Book API
 */

namespace nn {
namespace ahid {

enum DataType               //!< Data type for Item.
{
    // Raw bits
    DataType_Bitmask8           = 0x01, //!< 1 byte simple bit mask and shift

    // Logical == Physical (Unit is 0)
    DataType_Logical8Mask       = 0x11, //!< input data encoded in 1 byte
    DataType_Logical8           = 0x12, //!< input data encoded in 1 byte
    DataType_Logical16Mask      = 0x13, //!< input data encoded in 2 bytes
    DataType_Logical16          = 0x14, //!< input data encoded in 2 bytes

    // Logical->Physical->Unit+Exponent
    DataType_Physical8Mask      = 0x21, //!< input data encoded in 1 byte
    DataType_Physical8          = 0x22, //!< input data encoded in 1 byte
};

enum CodeBook
{
    CodeBook_Size   = 4096 * 4, //!< CodeBook size
};


// All following has to be 4 byte aligned

struct Item                 //!< Generic Item.
{
    uint8_t     type;       //!< DataType
    uint8_t     length;     //!< Item lenght in bytes
    uint8_t     reportId;   //!< Item report id
    uint8_t     index;      //!< Item index

    uint16_t    usagePage;  //!< Item HID usage page
    uint16_t    usageMin;   //!< Item HID usage id min
    uint16_t    usageMax;   //!< Item HID usage id max
    uint16_t    pad2;
};


struct CodeBookHeader
{
    uint8_t     reportCount;
    uint8_t     reportId[0xff];
    uint8_t     usagePage[0xff];
    uint8_t     usage[0xff];

    uint32_t    inputSize;
    uint32_t    outputSize;
    uint32_t    featureSize;

    uint32_t    inputItems;
    uint32_t    outputItems;
    uint32_t    featureItems;

};


struct ItemBitmask8
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    uint8_t     dataBits;   // & bits
    uint8_t     dataShift;  // >> bits
};


struct ItemLogical8Mask
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    uint8_t     dataBits;   // & bits
    uint8_t     dataShift;  // >> bits
    int8_t      logicalMinimum;
    int8_t      logicalMaximum;
    uint8_t     pad[2];
};


struct ItemLogical8
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    int8_t      logicalMinimum;
    int8_t      logicalMaximum;
};


struct ItemLogical16Mask
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    uint16_t    dataBits;   // & bits
    uint16_t    dataShift;  // >> bits
    uint16_t    signBit;    // for sign extension
    int16_t     logicalMinimum;
    int16_t     logicalMaximum;
    uint16_t    pad;
};


struct ItemLogical16
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    int16_t     logicalMinimum;
    int16_t     logicalMaximum;
    uint16_t    pad;
};


struct ItemPhysical8Mask
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    uint16_t    pad;
    uint8_t     dataBits;   // & bits
    uint8_t     dataShift;  // >> bits
    int8_t      logicalMinimum;
    int8_t      logicalMaximum;
    int32_t     physicalMinimum;
    int32_t     physicalMaximum;
    int32_t     unitExponont;
    int32_t     unit;
};


struct ItemPhysical8
{
    Item        item;

    uint16_t    dataOffset; // data location (first byte) in bytes
    int8_t      logicalMinimum;
    int8_t      logicalMaximum;
    int32_t     physicalMinimum;
    int32_t     physicalMaximum;
    int32_t     unitExponont;
    int32_t     unit;
};

} // end of namespace ahid
} // end of namespace nn
