﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    ahid.h
 * @brief   AHID API
 */

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/ahid/hdr/hdr_Types.h>
#include <nn/ahid/ahid_ResultPublic.h>
#include <nn/ahid/sfdl/IServerSession.sfdl.h>
#include <nn/ahid/sfdl/ICtrlSession.sfdl.h>
#include <nn/ahid/sfdl/IReadSession.sfdl.h>
#include <nn/ahid/sfdl/IWriteSession.sfdl.h>
#include <nn/ahid/ahid_CodeBook.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn {
namespace ahid {

class Ahid
{

    NN_DISALLOW_COPY(Ahid);
    NN_DISALLOW_MOVE(Ahid);

public:

    Ahid() NN_NOEXCEPT {};
    ~Ahid() NN_NOEXCEPT {};

    /**
     * @brief Initialize AHID instance, prepare access to device.
     * @param[in] pDeviceParameters
     *   Pointer to nn::ahid::hdr::DeviceParameters with parameters obtained from HDR.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize() was called.
     * @post
     *   If successful, the device associated with pDeviceParameters is ready for use.
     * @details
     *   This initializes the shim layer and attempts to acquire the device associated with pDeviceParameters from the AHID server that attached the device to HDR.
     */
    Result Initialize(nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT;

    /**
     * @brief Initialize AHID instance, prepare access to device using specified IServerSession.
     * @param[in] pServerSession
     *   Pointer to nn::sf::SharedPointer<IServerSession>.
     * @param[in] pDeviceParameters
     *   Pointer to nn::ahid::hdr::DeviceParameters with parameters obtained from HDR.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize() was called.
     * @post
     *   If successful, the device associated with pDeviceParameters is ready for use.
     * @details
     *   This initializes the shim layer and attempts to acquire the device associated with pDeviceParameters from the AHID server that attached the device to HDR.
     *   IserverSession is specified by the caller, intended to accesss functions using DFC from the same process as AHID server.
     */
    Result InitializeWith(nn::sf::SharedPointer<nn::ahid::IServerSession> pServerSession, nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT;

    /**
     * @brief Release access to AHID device.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() has already been called.
     * @post
     *   Ahid instance can no longer be used by this process unless Initialize is called again.
     * @details
     *   This finalizes the shim layer and releases access to the AHID device, any pending read or write will be returned with error.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Get nn::ahid::hdr::DeviceParameters
     * @param[out] pDeviceParameters
     *   nn::ahid::hdr::DeviceParameters.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pDeviceParameters.
     * @details
     *   Retrieve nn::ahid::hdr::DeviceParameters.
     */
    Result GetDeviceParameters(nn::ahid::hdr::DeviceParameters *pDeviceParameters) NN_NOEXCEPT;

    /**
     * @brief Create device state change event
     * @param[out] pOutEvent
     *   Supplied system event is initialized.
     * @param[in] eventClearMode
     *   Mode of operation assumed by initialized system event, related
     *   to how set event is cleared.
     * @retresult
     *     @handleresult{ResultSuccess,           Creation succeeded}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Specified system event is now initialized.
     * @details
     *   When HID device associated with this AHID experiences a change in state, this system
     *   event will be signalled.
     */
    Result CreateStateChangeEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief Destroy device state change event
     * @param[in] pInEvent
     *   Supplied system event is destroyed.
     * @retresult
     *     @handleresult{ResultSuccess,           Creation succeeded}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     *   Specified system event should have been successfully created via prior call to CreateInterfaceStateChangeEvent().
     * @post
     *   Specified system event is now finalized.
     * @details
     *   The system event is no longer receving signals, and should not be used.
     */
    Result DestroyStateChangeEvent(nn::os::SystemEventType* pInEvent);


    /**
     * @brief Get string descriptor by index.
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which retrieved string descriptor will be copied.
     * @param[in] bufferLength
     *   Sizein bytes, of provided buffer referred to by pBuffer
     * @param[in] stringIndex
     *   Index of string descriptor to retrieve.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   String descriptor was retrieved from device.
     * @details
     *   Submit a control request where specified string descriptor is to be retrieved from the device.
     */
    Result GetString(void* pOutBuffer, size_t bufferLength, uint8_t stringIndex) NN_NOEXCEPT;

    /**
     * @brief Get report by ctrl pipe.
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which retrieved report will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @param[in] reportType
     *   Report type AHID_REPORT_TYPE_INPUT, AHID_REPORT_TYPE_OUTPUT, AHID_REPORT_TYPE_FEATURE.
     * @param[in] reportId
     *   Report id to retrieve.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Report data was retrieved from device.
     * @details
     *   Submit a control request where specified report is to be retrieved from the device.
     */
    Result GetReport(void* pOutBuffer, size_t bufferLength, uint8_t reportType, uint8_t reportId) NN_NOEXCEPT;

    /**
     * @brief Set report by ctrl pipe.
     * @param[in] pBuffer
     *   Pointer to buffer, from which set report will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @param[in] reportType
     *   Report type AHID_REPORT_TYPE_INPUT, AHID_REPORT_TYPE_OUTPUT, AHID_REPORT_TYPE_FEATURE.
     * @param[in] reportId
     *   Report id to retrieve.
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Report data was sent to device.
     * @details
     *   Submit a control request where specified report is to be sent to the device.
     */
    Result SetReport(void* pBuffer, size_t bufferLength, uint8_t reportType, uint8_t reportId) NN_NOEXCEPT;

    /**
     * @brief Get idle.
     * @param[out] pOutIdle
     *   Pointer to receive interval in milliseconds, 0 implies non-periodic transactions, specifically transactions occur when read or write take place.
     * @param[in] reportId
     *   Report ID to get idle setting for.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Specified idle parameter was retrieved from device.
     * @details
     *   Submit a control request where the specified idle parameter is to be retrieved from the device.
     */
    Result GetIdle(uint8_t *pIdle, uint8_t reportId) NN_NOEXCEPT;

    /**
     * @brief Set idle.
     * @param[in] idle
     *   Idle in milliseconds, 0 implies non-periodic transactions, specifically transactions occur when read or write take place.
     * @param[in] reportId
     *   Report ID to apply idle setting.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Specified idle parameter was sent to device.
     * @details
     *   Submit a control request where the specified idle parameter is to be sent to the device.
     */
    Result SetIdle(uint8_t idle, uint8_t reportId) NN_NOEXCEPT;

    /**
     * @brief Get interface protocol.
     * @param[out] pOutProtocol
     *   Pointer to receive protocol.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Protocol parameter was retrieved from device.
     * @details
     *   Submit a control request where the protocol is to be retrieved from the device.
     */
    Result GetProtocol(uint8_t *pOutProtocol) NN_NOEXCEPT;

    /**
     * @brief Set interface protocol.
     * @param[in] protocol
     *   Protocol to set.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Protocol parameter was sent to device.
     * @details
     *   Submit a control request where the protocol parameter is to be sent to the device.
     */
    Result SetProtocol(uint8_t protocol) NN_NOEXCEPT;

    /**
     * @brief Get descriptor.
     * @param[out] pOutBuffer
     *   Pointer to receive descriptor.
     * @param[in] bmRequestType
     *   Request type bits per USB HID specification.
     * @param[in] wValue
     *   wValue parameter per USB HID specification.
     * @param[in] wIndex
     *   wVIndex parameter per USB HID specification.
     * @param[in] wLength
     *   Length of buffer.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Specified descriptor was retrieved from device.
     * @details
     *   Submit a control request where the specified descriptor is to be retrieved from the device.
     */
    Result GetDescriptor(void* pOutBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT;

    /**
     * @brief Set descriptor.
     * @param[in] pInBuffer
     *   Pointer to send descriptor.
     * @param[in] bmRequestType
     *   Request type bits per USB HID specification.
     * @param[in] wValue
     *   wValue parameter per USB HID specification.
     * @param[in] wIndex
     *   wVIndex parameter per USB HID specification.
     * @param[in] wLength
     *   Length of buffer.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Specified descriptor was sent to device.
     * @details
     *   Submit a control request where the specified descriptor is to be sent to the device.
     */
    Result SetDescriptor(void* pInBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT;

    /**
     * @brief Get input by interrupt pipe.
     * @param[out] pOutBytesRead
     *   Pointer to value, into which the number of bytes read will be specified.
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which retrieved read will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Input data was retrieved from device.
     * @details
     *   Submit a interrupt request where input data is to be retrieved from the device.
     */
    Result Read(uint32_t *pOutBytesRead, void* pOutBuffer, size_t bufferLength) NN_NOEXCEPT;

    /**
     * @brief Get input by interrupt pipe.
     * @param[out] pOutBytesRead
     *   Pointer to value, into which the number of bytes read will be specified.
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which retrieved read will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @param[in] timeout
     *   Period of time after which transaction will be aborted and API will complete unsuccessfully, 0 = no timeout.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Input data was retrieved from device.
     * @details
     *   Submit a interrupt request where input data is to be retrieved from the device.
     */
    Result ReadWithTimeout(uint32_t *pOutBytesRead, void* pOutBuffer, size_t bufferLength, nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief Set output by interrupt pipe.
     * @param[out] pOutBytesWritten
     *   Pointer to value, into which the number of bytes written will be specified.
     * @param[in] pBuffer
     *   Pointer to buffer, from which set write will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Output data was sent to device.
     * @details
     *   Submit a interrupt request where the output data is to be sent to the device.
     */
    Result Write(uint32_t *pOutBytesWritten, void* pBuffer, size_t bufferLength) NN_NOEXCEPT;

    /**
     * @brief Set output by interrupt pipe.
     * @param[out] pOutBytesWritten
     *   Pointer to value, into which the number of bytes written will be specified.
     * @param[in] pBuffer
     *   Pointer to buffer, from which set write will be copied.
     * @param[in] bufferLength
     *   Size in bytes, of provided buffer referred to by pBuffer.
     * @param[in] timeout
     *   Period of time after which transaction will be aborted and API will complete unsuccessfully, 0 = no timeout.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     *   @handleresult{ResultFunctionNotSupported}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Output data was sent to device.
     * @details
     *   Submit a interrupt request where the output data is to be sent to the device.
     */
    Result WriteWithTimeout(uint32_t *pOutBytesWritten, void* pBuffer, size_t bufferLength, nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief Get CodebookHeader.
     * @param[out] ppOutCodeBookHeader
     *   Pointer to storage to receive pointer to CodeBookHeader.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Pointer to CodeBook is written to ppOutCodeBookHeader.
     * @details
     *   Retrieves CodeBook header.
     */
    Result GetCodeBookHeader(CodeBookHeader** ppOutCodeBookHeader) NN_NOEXCEPT;

    /**
     * @brief Get input Item for specified usage.
     * @param[out] ppOutItem
     *   Pointer to storage to receive pointer to Item for runtime input use. This value will be set to 0 if speficied control does not exist in the AHID CodeBook.
     * @param[in] reportId
     *   Report id.
     * @param[in] usagePage
     *   Usage Page as defined by HID sepcification.
     * @param[in] usageMin
     *   Usage minimum as defined by Usage Page, set to same value as usageMax if no range is desired.
     * @param[in] usageMax
     *   Usage maximum as defined by Usage Page, set to same value as usageMin if no range is desired.
     * @param[in] index
     *   Zero based index of control specified by usagePage, UsageMin, UsageMax.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultItemNotFound}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Pointer to Item is written to ppOutItem.
     * @details
     *   Retrieves input item matching parameters for runtime decoding.
     */
    Result GetInputItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin,  uint8_t usageMax, uint8_t index) NN_NOEXCEPT;

    /**
     * @brief Get output Item for specified usage.
     * @param[out] ppOutItem
     *   Pointer to storage to receive pointer to Item for runtime output use. This value will be set to 0 if speficied control does not exist in the AHID CodeBook.
     * @param[in] reportId
     *   Report id.
     * @param[in] usagePage
     *   Usage Page as defined by HID sepcification.
     * @param[in] usageMin
     *   Usage minimum as defined by Usage Page, set to same value as usageMax if no range is desired.
     * @param[in] usageMax
     *   Usage maximum as defined by Usage Page, set to same value as usageMin if no range is desired.
     * @param[in] index
     *   Zero based index of control specified by usagePage, UsageMin, UsageMax.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultItemNotFound}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Pointer to Item is written to ppOutItem.
     * @details
     *   Retrieves output item matching parameters for runtime encoding.
     */
    Result GetOutputItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin,  uint8_t usageMax, uint8_t index) NN_NOEXCEPT;

    /**
     * @brief Get feature Item for specified usage.
     * @param[out] ppOutItem
     *   Pointer to storage to receive pointer to Item for runtime feature use. This value will be set to 0 if speficied control does not exist in the AHID CodeBook.
     * @param[in] reportId
     *   Report id.
     * @param[in] usagePage
     *   Usage Page as defined by HID sepcification.
     * @param[in] usageMin
     *   Usage minimum as defined by Usage Page, set to same value as usageMax if no range is desired.
     * @param[in] usageMax
     *   Usage maximum as defined by Usage Page, set to same value as usageMin if no range is desired.
     * @param[in] index
     *   Zero based index of control specified by usagePage, UsageMin, UsageMax.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultItemNotFound}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Pointer to Item is written to ppOutItem.
     * @details
     *   Retrieves feature item matching parameters for runtime encoding or decoding.
     */
    Result GetFeatureItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin,  uint8_t usageMax, uint8_t index) NN_NOEXCEPT;

    /**
     * @brief Decode item using input data and item
     * @param[out] pOutData
     *   Ouput data for the control associated to pDecoder.
     * @param[in] pInputData
     *   Pointer to input data from device as retrieved by Read funtion.
     * @param[in] pItem
     *   Pointer to Item as retrieved by GetInputItem, GetOutputItem, GetFeatureItem function.
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pOutData.
     * @details
     *   Decode data associated with pItem from pInputData.
     */
    Result Decode(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;

    /**
     * @brief Encode item using output data and encoder
     * @param[in] pData
     *   Input data for the control associated to pEncoder.
     * @param[in] pOutputData
     *   Pointer to output data to device to be sent by Write funtion.
     * @param[in] pItem
     *   Pointer to Item as retrieved by GetInputItem, GetOutputItem, GetFeatureItem function.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pOutputData.
     * @details
     *   Encode data associated with pItem to pOutputData.
     */
    Result Encode(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;

    /**
     * @brief Get minimum, maximum, unit of measure, and exponent for item
     * @param[out] pMinimum
     *   Pointer to store minimum value.
     * @param[out] pMaximum
     *   Pointer to store maximum value.
     * @param[out] pUnit
     *   Pointer to store unit value.
     * @param[out] pExponent
     *   Pointer to store exponent value.
     * @param[in] pItem
     *   Pointer to Item for which to retrieve unit value.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pMinimum, pMaximum, pUnit, pExponent.
     * @details
     *   Retrieve unit of measure from pItem.
     */
    Result GetUnit(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;

    /**
     * @brief Get number of unique report IDs
     * @param[out] pOutReportCount
     *   Pointer to store report count.
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pOutReportCount.
     * @details
     *   Retrieve number of unique Report IDs.
     */
    Result GetReportCount(uint8_t *pOutReportCount) NN_NOEXCEPT;

    /**
     * @brief Get report ID, Usage Page, Usage, for report ID index
     * @param[out] pOutReportId
     *   Report ID.
     * @param[out] pOutUsagePage
     *   Usage Page.
     * @param[out] pOutUsage
     *   Usage.
     * @param[in] index
     *   Index in table.
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidParameter}
     * @endretresult
     * @pre
     *   Initialize() must be called once first.
     * @post
     *   Data is written to pReportId, pUsagePage, pUsage.
     * @details
     *   Retrieve report ID, Usage Page, Usage, for report ID index.
     */
    Result GetReportIdUsage(uint8_t *pOutReportId, uint8_t *pOutUsagePage, uint8_t *pOutUsage, int index) NN_NOEXCEPT;

private:

    uint8_t                                 m_CodeBook[CodeBook::CodeBook_Size];
    nn::ahid::hdr::DeviceParameters         m_DeviceParameters;
    nn::sf::SharedPointer<IServerSession>   m_ServerSession;
    nn::sf::SharedPointer<ICtrlSession>     m_CtrlSession;
    nn::sf::SharedPointer<IReadSession>     m_ReadSession;

    uint8_t* SkipItems(uint8_t* p, uint32_t items) NN_NOEXCEPT;
    Item* GetItem(uint8_t* p, uint32_t entries, uint8_t reportId, uint8_t usagePage, uint8_t usageMin, uint8_t usageMax, uint8_t index) NN_NOEXCEPT;

    // encode / decode per item type
    Result DecodeBitmask8(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodeBitmask8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodeLogical8Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodeLogical8Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodeLogical8(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodeLogical8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodeLogical16Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodeLogical16Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodeLogical16(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodeLogical16(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodePhysical8Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodePhysical8Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;
    Result DecodePhysical8(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT;
    Result EncodePhysical8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT;

    // minimum, maximum, unit, exponent, per type
    void GetBitmask8Context         (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetLogical8MaskContext     (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetLogical8Context         (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetLogical16MaskContext    (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetLogical16Context        (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetPhysical8MaskContext    (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;
    void GetPhysical8Context        (int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT;

    // read write data
    int16_t Read16BitData(uint8_t* pInputData) NN_NOEXCEPT;
    void Write16BitData(int16_t inputData, uint8_t* pOutputData) NN_NOEXCEPT;

    // get sessions on initialization
    void GetSessions(nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT;
};

} // end of namespace ahid
} // end of namespace nn
