﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   システムアプレットの制御に関する API の宣言
 *
 * @details
 *  本ヘッダでは、システムアプレットの制御に関する API を宣言しています。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/ae/ae_Types.h>
#include <nn/applet/applet_Storage.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace ae {

//-----------------------------------------------------------------------------
/**
 * @brief   システムアプレットの起動パラメータを格納する構造体です。
 *
 * @details
 *  システムアプレット起動時に呼出し元から渡された起動パラメータなどの情報を
 *  格納する構造体です。
 *
 */
struct SystemAppletParameters
{
    char*   _noParameter;
};


//-----------------------------------------------------------------------------
/**
 * @brief   システムアプレットのメイン関数のエントリを表す型です。
 */
typedef void    (*SystemAppletMainFunction)(SystemAppletParameters*);


//--------------------------------------------------------------------------

//! @name システムアプレット制御用 API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   システムアプレットとしての初期化を行ない、アプレットのメイン関数を呼び出します。
 *
 * @param[in]  appletId 自身のアプレット ID
 * @param[in]  function アプレットのメイン関数エントリ
 *
 * @pre
 *  - appletId == AppletId_SystemAppletMenu である
 *
 * @details
 *  システムアプレットとしての初期化を行ない、その後に function で指定された
 *  アプレットのメイン関数を呼び出します。function は SystemAppletMainFunction
 *  型の関数であり、引数で AppletParameter 構造体を受け取ります。
 *
 *  本関数の呼び出し中、システムアプレットとして実行中であることになります。
 *
 *  本関数は nnMain() の中から呼び出して下さい。また、
 *  本関数を呼ぶまで ae ライブラリで提供している他の機能は使用できません。
 *
 *  appletId には自身のアプレット種別を示す AppletId 型の列挙子のいずれかを
 *  指定して下さい。現時点では AppletId_SystemAppletMenu のみが指定できます。
 *
 *  本 API を発行しても、自アプレットは Foreground 権限を取得しません。
 *  自アプレットが Foreground に遷移すべき場合には、本 API 発行直後に
 *  Message_ChangeIntoForeground が通知されます。
 *
 */
void InvokeSystemAppletMain(AppletId appletId, SystemAppletMainFunction function) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   本体が起動した理由を取得します。
 *
 * @return  本体起動理由を示す BootReason 列挙子で返します。
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  本体が起動した理由を BootReason の列挙子で返します。
 *
 */
BootReason GetBootReason() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   システムアプレットが Foreground へ遷移できるように要求を出します。
 *
 * @retresult
 *   @handleresult{nn::ae::ResultAlreadyForeground}
 * @endretresult
 *
 * @pre
 *  - システムアプレットとして実行中である
 *  - システムアプレットが Background 状態である
 *
 * @details
 *  FG 状態で動作しているアプリケーション等を BG 状態へ遷移させ、
 *  システムアプレットが FG 状態へ遷移できるように要求を出します。
 *
 *  アプリケーションが LA 呼出中の場合は、その呼び出しスタックの中で
 *  先端の FG 状態にある LA に対して BG 遷移要求を出します。
 *  また、システムアプレットが LA 呼出中の場合は、その呼び出しスタックの中で
 *  先端の BG 状態にある LA に対して FG 遷移要求を出します。
 *
 *  本関数は、以下の用途で使用します。
 *  - アプリ動作中に HOME ボタン短押しされたときに、システムアプレットを
 *    FG 状態に移行させ、メニュー画面を表示させたい場合
 *
 *  本 API は要求だけを出して即座にリターンします。
 *
 */
Result RequestToGetForeground() NN_NOEXCEPT;


//! @}

//--------------------------------------------------------------------------

//! @name システムアプレット関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   強制的に FG 状態に遷移できるように要求を出します。
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  アプリケーションを含む全アプレットの中で、システムアプレットが FG 状態に
 *  遷移できるように要求を出し、その状態を維持します。
 *
 *  SA が BG 状態だった場合には、Message_ChangeIntoForeground が通知されます。
 *  SA が既に FG 状態だった場合にはメッセージは通知されません。
 *
 *  本 API は nn::ae::RequestToGetForeground() に似ていますが、
 *  SA から LA が起動している状態では以下のような違いがあります。
 *
 *  - nn::ae::RequestToGetForeground() では先端にある LA に FG 権限が移ります。
 *  - nn::ae::LockForeground() では SA に FG 権限が移ります。
 *
 *  本 API は、現状の各アプレットの状態に関わらず、SA が FG 状態となって
 *  出画やユーザインタラクションを行なう必要がある場合に使用します。
 *
 *  本 API 発行後は、基本的に SA 以外が FG 状態になることはありません。
 *  また、強制 FG 状態を解除するには nn::ae::UnlockForeground() を
 *  発行して下さい。
 *
 */
void LockForeground() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   強制 FG 状態を解放します。
 *
 * @pre
 *  - システムアプレットとして実行中である
 *
 * @details
 *  nn::ae::LockForeground() による強制 FG 状態を解除し、
 *  通常の状態遷移に戻ります。
 *
 *  本 API 発行後、本来 FG 状態になるべきアプレットが
 *  FG 状態となれるように適切なメッセージ通知が行なわれます。
 *
 */
void UnlockForeground() NN_NOEXCEPT;


/**
    @brief システムの汎用チャンネルからストレージを pop します。


    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre *pOut に書き込める

    @details
     applet::PushToSystemGeneralChannel() によって push されたストレージを pop します。
*/
bool TryPopFromSystemGeneralChannel(applet::StorageHandle* pOut) NN_NOEXCEPT;

/**
    @brief システムの汎用チャンネルからストレージを pop できるデータがあることを待機するためのイベントを取得します。

    @return イベントへのポインタを返します。

    @details
     システムの汎用チャンネルに pop できるデータがあることを待機するためのイベントを取得します。
     ここで取得されたイベントは、システムの汎用チャンネルにストレージがあるときにシグナルされます。
     このイベントは TryPopFromSystemGeneralChannel() の呼び出しによってストレージが空になると、シグナルがクリアされます。
*/
os::SystemEventType* GetPopFromSystemGeneralChannelEvent() NN_NOEXCEPT;


//! @}

//! @name HOME ボタン処理区間操作
//! @{

/**
    @brief HOME ボタン処理区間に入ります。

    @details
     HOME ボタンの禁止に関して、HOME ボタン処理区間と禁止区間が定義されます。

     HOME ボタン処理区間とは、システムアプレットのみが入ることができる区間で、
     EnterHomeButtonProcessSection() の呼び出しが返ってから(または TryEnterHomeButtonProcessSection() が true を返してから)、
     LeaveHomeButtonProcessSection() の呼び出しが呼び出されるまでの区間を言います。

     HOME ボタン処理区間には、誰も HOME ボタン処理区間にも禁止区間にも入っていないときにのみ入ることができます。

     HOME ボタン禁止区間とは、各アプレットが入ることができる区間で、
     EnterHomeButtonProhibitionSection() の呼び出しが返ってから、
     それと同数の LeaveHomeButtonProhibitionSection() が呼び出されるまでの区間を言います。

     HOME ボタン禁止区間には、誰も HOME ボタン処理区間に入っていないときにのみ入ることができます。
     複数の主体が HOME ボタン禁止区間に同時に入ることはできます。

     一般に、システムアプレットが HOME ボタンの通知を受け、それに対する処理を行うときには、
     EnterHomeButtonProcessSection() または TryEnterHomeButtonProcessSection() を呼び出し、
     処理が完了したら LeaveHomeButtonProcessSection() を呼ぶことを想定しています。
     TryEnterHomeButtonProcessSection() が false を返したような場合には、
     何も行わないこともできますし、必要に応じて禁止アイコンなどを表示することも可能です。

     その他のアプレットなどが、HOME ボタンによる遷移をされると不都合な処理を行う際には、
     EnterHomeButtonProhibitionSection() を呼んでからその処理を行い、
     終了したら LeaveHomeButtonProcessSection() を呼ぶことを想定しています。
*/
void EnterHomeButtonProcessSection() NN_NOEXCEPT;

/**
    @brief HOME ボタン処理区間に入ること試行します。

    @return HOME ボタン処理区間には入れたら true を返し、そうでない場合には false を返します。

    @see EnterHomeButtonProcessSection()
*/
bool TryEnterHomeButtonProcessSection() NN_NOEXCEPT;

/**
    @brief HOME ボタン処理区間を抜けます。

    @see EnterHomeButtonProcessSection()
*/
void LeaveHomeButtonProcessSection() NN_NOEXCEPT;

//! @}


//! @name エントランス処理区間操作
//! @{

/**
    @brief エントランス処理区間に入ります。

    @pre 自アプレットがシステムアプレットである。

    @details
     各アプレットは、エントランスの処理区間と禁止区間の排他制御を行なうために、
     これらの区間に同時に入らないようにする同期機能を使用することができます。

     本機能は、SA 上のエントランスシーンの排他制御のために用意されています。
     基本的には、HOME ボタンの処理区間・禁止区間制御と同様の機能であるため、
     使用方法は nn::ae::EnterHomeButtonProcessSection() を参照して下さい。

     エントランス処理区間の開始～終了の宣言は以下の API を使用して下さい。

     - nn::ae::EnterEntranceProcessSection()
     - nn::ae::TryEnterEntranceProcessSection()
     - nn::ae::LeaveEntranceProcessSection()

     一方、エントランス禁止区間の開始～終了の宣言は以下の API を使用して下さい。

     - nn::ae::EnterEntranceProhibitionSection()
     - nn::ae::TryEnterEntranceProhibitionSection()
     - nn::ae::LeaveEntranceProhibitionSection()

*/
void EnterEntranceProcessSection() NN_NOEXCEPT;

/**
    @brief エントランス処理区間に入ることを試行します。

    @return エントランス処理区間に入れたら true を返し、そうでない場合には false を返します。

    @see EnterEntranceProcessSection()
*/
bool TryEnterEntranceProcessSection() NN_NOEXCEPT;

/**
    @brief エントランス処理区間を抜けます。

    @see EnterEntranceProcessSection()
*/
void LeaveEntranceProcessSection() NN_NOEXCEPT;

//! @}


//! @name 割込みシーン処理区間操作
//! @{

/**
    @brief 割込みシーン処理区間に入ります。

    @pre 自アプレットがシステムアプレットである。

    @details
     各アプレットは、割込みシーンの処理区間と禁止区間の排他制御を行なうために、
     これらの区間に同時に入らないようにする同期機能を使用することができます。

     本機能は、SA 上の割込みシーンの排他制御のために用意されています。
     基本的には、HOME ボタンの処理区間・禁止区間制御と同様の機能であるため、
     使用方法は nn::ae::EnterHomeButtonProcessSection() を参照して下さい。

     割込みシーン処理区間の開始～終了の宣言は以下の API を使用して下さい。

     - nn::ae::EnterInterruptSceneProcessSection()
     - nn::ae::TryEnterInterruptSceneProcessSection()
     - nn::ae::LeaveInterruptSceneProcessSection()

     一方、割込みシーン禁止区間の開始～終了の宣言は以下の API を使用して下さい。

     - nn::ae::EnterInterruptSceneProhibitionSection()
     - nn::ae::TryEnterInterruptSceneProhibitionSection()
     - nn::ae::LeaveInterruptSceneProhibitionSection()

*/
void EnterInterruptSceneProcessSection() NN_NOEXCEPT;

/**
    @brief 割込みシーン処理区間に入ることを試行します。

    @return 割込みシーン処理区間に入れたら true を返し、そうでない場合には false を返します。

    @see EnterInterruptSceneProcessSection()
*/
bool TryEnterInterruptSceneProcessSection() NN_NOEXCEPT;

/**
    @brief 割込みシーン処理区間を抜けます。

    @see EnterInterruptSceneProcessSection()
*/
void LeaveInterruptSceneProcessSection() NN_NOEXCEPT;

//! @}

//! @name その他
//! @{

/**
    @brief CEC 設定の変更を通知し、設定を反映します。
*/
void NotifyCecSettingsChanged() NN_NOEXCEPT;

/**
    @brief HOME ボタン長押しの時間を設定します。

    @param[in] time 設定する時間
*/
void SetDefaultHomeButtonLongPressTime(TimeSpan time) NN_NOEXCEPT;

/**
    @brief 本体起動時にスリープすべきかどうかを返します。

    @details
        クレードル接続により本体が起動したとき、本 API は true を返します。

    @return スリープすべきときは true、そうでないときは false を返します。
*/
bool ShouldSleepOnBoot() NN_NOEXCEPT;


//! @}

//! @name デバッグ用
//! @{

/**
    @brief SA からのアプリケーション起動のリクエストを受け取ります。

    @param[out] pApplicationId 起動対象の ApplicationId を格納する変数へのポインタを指定します。
    @param[out] pUid 起動する際に選択するアカウントを格納する変数へのポインタを指定します。

    @return リクエストがあれば true を返し、リクエストがなければ false を返します。

    @details
     Message_RequestToLaunchApplicationForDebug を受け取った場合、本関数を呼んでリクエストを受信します。
     false が返った際には起動リクエストがないため何もしないでください。
     true が返った際には、*pApplicationId で指定されるアプリケーションをユーザー起動しようとした際と同じ振る舞いを行ってください。
     この際に、ユーザーアカウントを選択する必要がある場合には、ユーザー選択の UI を出さずに、*pUid で指定されたアカウントを使用してください。
*/
bool PopRequestLaunchApplicationForDebug(ncm::ApplicationId* pApplicationId, account::Uid* pUid) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::ae

