﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_AsyncContextImpl.h>
#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/user/account_UserReference.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn {
namespace account {
namespace nas {

template <typename Allocator>
class NintendoAccountAuthorizationRequestImpl
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(NintendoAccountAuthorizationRequestImpl);
    friend class ImplicitTask;

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    NasSessionPool<Allocator>& m_NasSessionPool;
    Executor& m_Executor;

    const user::UserRef m_User;

    detail::Uuid m_SessionId;
    NasApplicationAuthorizationProcedure* m_pProcedure;

    class ImplicitTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<NintendoAccountAuthorizationRequestImpl<Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
    public:
        explicit ImplicitTask(ParentPtr&& parent) NN_NOEXCEPT
            : m_Parent(std::move(parent))
        {
        }
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            return m_Parent->m_pProcedure->ExecuteImplicitly(resource, this);
        }
    };

public:
    NintendoAccountAuthorizationRequestImpl(
        Allocator& allocator,
        const user::UserRef& user, NasOperator& nasOp,
        NasSessionPool<Allocator>& nasSessionPool, Executor& executor) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_NasSessionPool(nasSessionPool)
        , m_Executor(executor)
        , m_User(user)
        , m_SessionId(detail::InvalidUuid)
        , m_pProcedure(nullptr)
    {
    }
    ~NintendoAccountAuthorizationRequestImpl() NN_NOEXCEPT
    {
        if (m_SessionId)
        {
            m_NasSessionPool.ReleaseNasApplicationAuthorizationSession(m_User, m_SessionId);
        }
    }

    Result Initialize(
        const NasClientInfo& nasClientInfo, const NintendoAccountAuthorizationRequestParameters& params,
        NasOperator& nasOp, sf::NativeHandle&& transferMemoryHandle, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(nasClientInfo, ResultInvalidApplication());
        NN_RESULT_THROW_UNLESS(size >= NasApplicationAuthorizationProcedure::RequiredMemorySize, ResultInsufficientBuffer());
        NN_RESULT_THROW_UNLESS(!m_SessionId, ResultInvalidProtocolAccess());

        NN_RESULT_DO(m_NasSessionPool.CreateNasApplicationAuthorizationSession(&m_SessionId, &m_pProcedure, m_User, nasClientInfo, nasOp));
        m_pProcedure->AttachTransferMemory(transferMemoryHandle.GetOsHandle(), size, transferMemoryHandle.IsManaged());
        transferMemoryHandle.Detach();

        NN_RESULT_DO(m_pProcedure->Initialize(params));
        NN_RESULT_SUCCESS;
    }

    Result Initialize(
        const detail::ApplicationInfo& appInfo, const NintendoAccountAuthorizationRequestParameters& params,
        NasOperator& nasOp, sf::NativeHandle&& transferMemoryHandle, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
        NN_RESULT_THROW_UNLESS(size >= NasApplicationAuthorizationProcedure::RequiredMemorySize, ResultInsufficientBuffer());
        NN_RESULT_THROW_UNLESS(!m_SessionId, ResultInvalidProtocolAccess());

        NN_RESULT_DO(m_NasSessionPool.CreateNasApplicationAuthorizationSession(&m_SessionId, &m_pProcedure, m_User, appInfo, nasOp));
        m_pProcedure->AttachTransferMemory(transferMemoryHandle.GetOsHandle(), size, transferMemoryHandle.IsManaged());
        transferMemoryHandle.Detach();
        NN_SDK_ASSERT(m_SessionId);

        NN_RESULT_DO(m_pProcedure->Initialize(params));
        NN_RESULT_SUCCESS;
    }

    Result GetSessionId(sf::Out<detail::Uuid> pOutSessionId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        *pOutSessionId = m_SessionId;
        NN_RESULT_SUCCESS;
    }
    Result InvokeWithoutInteractionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<ImplicitTask, ExecutionResource>>(
            &m_Allocator, typename ImplicitTask::ParentPtr(this, true));
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOutContext = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result IsAuthorized(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());
        *pOut = m_pProcedure->IsAuthorized();
        NN_RESULT_SUCCESS;
    }
    Result GetAuthorizationCode(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->GetAuthorizationCode(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize()));
        NN_SDK_ASSERT(sizeActual <= detail::NasAuthorizationCodeSizeMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
    Result GetIdToken(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->GetIdToken(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize()));
        NN_SDK_ASSERT(sizeActual <= detail::NasIdTokenSizeMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
    Result GetState(sf::Out<State> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetState(pOut.GetPointer());
    }
};

} // ~namespace nn::account::nas
}
}
