﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// BaaS との同期情報のキャッシュ

#include <nn/account/detail/account_CacheBase.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace account { namespace nas {

struct Op2MembershipHolder
{
    NetworkServiceLicense license;
    time::PosixTime expiresAt; // license が失効する時刻 (PosixTime)
    TimeSpan timeStamp; // 取得した時刻 (Uptime)
};
NN_STATIC_ASSERT(std::is_trivially_copyable<Op2MembershipHolder>::value);

class NasOp2MembershipCache
    : detail::CacheBaseN<16, NintendoAccountId, Op2MembershipHolder, detail::DefaultCacheOperator<Op2MembershipHolder>, detail::DefaultCacheResult>
{
private:
    typedef detail::CacheBaseN<16, NintendoAccountId, Op2MembershipHolder, detail::DefaultCacheOperator<Op2MembershipHolder>, detail::DefaultCacheResult> Base;

public:
    Result Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        return Base::Initialize(storage);
    }
    void Store(const NintendoAccountId& tag, NetworkServiceLicense license, time::PosixTime expiresAt) NN_NOEXCEPT
    {
        Op2MembershipHolder data;
        data.license = license;
        data.expiresAt = expiresAt;
        data.timeStamp = os::GetSystemTick().ToTimeSpan();
        Base::Store(tag, 0, data);
    }
    Result GetTimeStamp(TimeSpan* pOut, const NintendoAccountId& tag) const NN_NOEXCEPT
    {
        std::lock_guard<const Base> lock(*static_cast<const Base*>(this));
        Op2MembershipHolder data;
        NN_RESULT_DO(Base::FindUnsafe(&data, tag));
        *pOut = data.timeStamp;
        NN_RESULT_SUCCESS;
    }
    Result Get(NetworkServiceLicense* pOutLicense, time::PosixTime* pOutExpiresAt, const NintendoAccountId& tag) const NN_NOEXCEPT
    {
        std::lock_guard<const Base> lock(*static_cast<const Base*>(this));
        Op2MembershipHolder data;
        NN_RESULT_DO(Base::FindUnsafe(&data, tag));
        *pOutLicense = data.license;
        *pOutExpiresAt = data.expiresAt;
        NN_RESULT_SUCCESS;
    }
    void Invalidate(const NintendoAccountId& tag) NN_NOEXCEPT
    {
        Base::Invalidate(tag);
    }
};

}}} // ~namespace nn::account::nas
