﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <limits>
#include <utility>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace account { namespace detail {

class Counter
{
    friend class SharedCounter;
private:
    std::atomic<uint32_t> value;
    void Add() NN_NOEXCEPT
    {
        auto v = value.fetch_add(1);
        NN_ABORT_UNLESS(v < std::numeric_limits<uint32_t>::max());
    }
    void Sub() NN_NOEXCEPT
    {
        auto v = value.fetch_sub(1);
        NN_ABORT_UNLESS(v > 0);
    }
public:
    Counter() NN_NOEXCEPT
        : value(0)
    {
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return value.load() != 0;
    }
    uint32_t Get() const NN_NOEXCEPT
    {
        return value.load();
    }
};

class SharedCounter
{
private:
    Counter* m_pCounter;
    void Swap(SharedCounter& rhs) NN_NOEXCEPT
    {
        std::swap(this->m_pCounter, rhs.m_pCounter);
    }
public:
    SharedCounter() NN_NOEXCEPT
        : m_pCounter(nullptr)
    {
    }
    NN_IMPLICIT SharedCounter(Counter& counter) NN_NOEXCEPT
        : m_pCounter(&counter)
    {
        m_pCounter->Add();
    }
    SharedCounter(const SharedCounter& rhs) NN_NOEXCEPT
        : m_pCounter(rhs.m_pCounter)
    {
        if (m_pCounter)
        {
            m_pCounter->Add();
        }
    }
    SharedCounter(SharedCounter&& rhs) NN_NOEXCEPT
        : m_pCounter(rhs.m_pCounter)
    {
        rhs.m_pCounter = nullptr;
    }
    ~SharedCounter() NN_NOEXCEPT
    {
        if (m_pCounter)
        {
            m_pCounter->Sub();
            m_pCounter = nullptr;
        }
    }
    SharedCounter& operator=(Counter& rhs) NN_NOEXCEPT
    {
        SharedCounter tmp(rhs);
        tmp.Swap(*this);
        return *this;
    }
    SharedCounter& operator=(const SharedCounter& rhs) NN_NOEXCEPT
    {
        SharedCounter tmp(rhs);
        tmp.Swap(*this);
        return *this;
    }
    SharedCounter& operator=(SharedCounter&& rhs) NN_NOEXCEPT
    {
        SharedCounter tmp(std::move(rhs));
        tmp.Swap(*this);
        return *this;
    }
    void Release() NN_NOEXCEPT
    {
        SharedCounter tmp;
        tmp.Swap(*this);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_pCounter != nullptr;
    }
    uint32_t Get() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pCounter);
        return m_pCounter->Get();
    }
};

}}} // ~namespace nn::account::detail
