﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_NetworkServiceAccountAdministratorImpl.h>
#include <nn/account/detail/account_ApplicationResourceManager.h>
#include <nn/account/nas/account_NintendoAccountExtention.h>
#include <nn/account/user/account_UserStateManager.h>
#include <nn/srepo/srepo_StateNotifier.h>

namespace nn { namespace account {
class Executor;

template <typename Allocator>
class AccountServiceImplForApplication;
}} // ~namespace nn::account

namespace nn { namespace account { namespace detail {

template <typename Allocator>
class ReferenceContextImpl
    : public baas::NetworkServiceAccountAdministratorImpl<Allocator>
    , public nas::NintendoAccountExtention<Allocator, ReferenceContextImpl<Allocator>>
{
public:
    ReferenceContextImpl(
        Allocator& allocator,
        nas::NasSessionPool<Allocator>& nasSessionPool,
        baas::BaasOperator& baasOp, nas::NasOperator& nasOp,
        user::UserRef&& user,
        account::Executor& executor) NN_NOEXCEPT
        : baas::NetworkServiceAccountAdministratorImpl<Allocator>(allocator, baasOp, nasOp, std::move(user), executor)
        , nas::NintendoAccountExtention<Allocator, ReferenceContextImpl<Allocator>>(allocator, nasSessionPool, nasOp, executor)
    {
    }
};

template <typename Allocator>
class OpenContextImpl
    : public sf::IServiceObject
    , public ReferenceContextImpl<Allocator>
{
public:
    typedef sf::SharedPointer<AccountServiceImplForApplication<Allocator>> ParentPtr;

private:
    ParentPtr m_Parent;
    user::UserStateManager& m_StateManager;

    Uid const m_Uid;
    detail::ApplicationInfo const m_AppInfo;
    bool m_Initialized;

    Result Finalize() NN_NOEXCEPT;

public:
    OpenContextImpl(
        ParentPtr&& parent,
        Allocator& allocator,
        user::UserStateManager& stateManager, nas::NasSessionPool<Allocator>& nasSessionPool,
        baas::BaasOperator& baasOp, nas::NasOperator& nasOp,
        user::UserRef&& user, const detail::ApplicationInfo& appInfo,
        account::Executor& executor) NN_NOEXCEPT
        : ReferenceContextImpl<Allocator>(allocator, nasSessionPool, baasOp, nasOp, user::UserRef(user), executor)
        , m_Parent(std::move(parent))
        , m_StateManager(stateManager)
        , m_Uid(user)
        , m_AppInfo(appInfo)
        , m_Initialized(false)
    {
        NN_SDK_ASSERT(m_Uid);
        NN_SDK_ASSERT(m_AppInfo);
    }
    ~OpenContextImpl() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            auto r = Finalize();
            NN_UNUSED(r);
        }
    }
    Uid GetUid() const NN_NOEXCEPT;
    ApplicationId GetApplicationId() const NN_NOEXCEPT;

    Result Initialize() NN_NOEXCEPT;
    Result StoreOpenContext() NN_NOEXCEPT;
};

template <typename Allocator>
class OpenContextHolder
{
public:
    using Context = sf::SharedPointer<OpenContextImpl<Allocator>>;

private:
    Context m_Contexts[UserCountMax] {};

public:
    void Push(Context&& context) NN_NOEXCEPT;
    Context Pull(const Uid& uid) NN_NOEXCEPT;
    int ListUsers(Uid buffer[], int count) const NN_NOEXCEPT;
};

template <typename Allocator>
class OpenContextRetainer
    : public ApplicationResourceManager<OpenContextHolder<Allocator>>
{
public:
    bool TryPush(sf::SharedPointer<OpenContextImpl<Allocator>>&& context) NN_NOEXCEPT;
    sf::SharedPointer<OpenContextImpl<Allocator>> Pull(ApplicationId appId, const Uid& uid) NN_NOEXCEPT;
    int ListUsers(Uid buffer[], int count, ApplicationId appId) const NN_NOEXCEPT;
};

}}} // ~namespace nn::account::detail

#include <nn/nn_SdkLog.h>
#include <nn/ovln/format/ovln_AccountMessage.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace account { namespace detail {

// ---------------------------------------------------------------------------------------------
// misc

Result SendOvlnNotification(const Uid& uid, ovln::format::AccountUserStateChangeKind kind) NN_NOEXCEPT;

// ---------------------------------------------------------------------------------------------
// OpenContextImpl

template <typename Allocator>
Uid OpenContextImpl<Allocator>::GetUid() const NN_NOEXCEPT
{
    return m_Uid;
}
template <typename Allocator>
ApplicationId OpenContextImpl<Allocator>::GetApplicationId() const NN_NOEXCEPT
{
    return {m_AppInfo.launchProperty.id.value};
}

template <typename Allocator>
Result OpenContextImpl<Allocator>::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    auto user = ReferenceContextImpl<Allocator>::GetUserReference();
    NN_RESULT_DO(m_StateManager.SetUserStateClosed(GetApplicationId(), user));
    m_Initialized = false;

    auto r = SendOvlnNotification(user, ovln::format::AccountUserStateChangeKind::Closed);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN(
            "SendOvlnNotification<ovln::format::AccountUserStateChangeKind::Closed> failed with %03d-%04d(0x%08lx)\n",
            r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserClosed(static_cast<Uid>(user));
#endif

    NN_RESULT_SUCCESS;
}

template <typename Allocator>
Result OpenContextImpl<Allocator>::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Initialized);
    NN_RESULT_DO(baas::NetworkServiceAccountAdministratorImpl<Allocator>::SetApplicationInfo(m_AppInfo));

    auto user = ReferenceContextImpl<Allocator>::GetUserReference();
    auto uid = static_cast<Uid>(user);
    NN_RESULT_DO(m_StateManager.SetUserStateOpened(GetApplicationId(), std::move(user)));
    m_Initialized = true;

    auto r = SendOvlnNotification(uid, ovln::format::AccountUserStateChangeKind::Opened);
    if (!r.IsSuccess())
    {
        NN_DETAIL_ACCOUNT_WARN(
            "SendOvlnNotification<ovln::format::AccountUserStateChangeKind::Opened> failed with %03d-%04d(0x%08lx)\n",
            r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserOpened(uid);
#endif

    NN_RESULT_SUCCESS;
}

template <typename Allocator>
Result OpenContextImpl<Allocator>::StoreOpenContext() NN_NOEXCEPT
{
    return m_Parent->StoreOpenContext(sf::SharedPointer<typename std::remove_reference<decltype(*this)>::type>(this, true));
}

// ---------------------------------------------------------------------------------------------
// OpenContextRetainer
template <typename Allocator>
bool OpenContextRetainer<Allocator>::TryPush(sf::SharedPointer<OpenContextImpl<Allocator>>&& context) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(context->GetApplicationId());
    if (!p)
    {
        return false;
    }
    p->Push(std::move(context));
    return true;
}
template <typename Allocator>
sf::SharedPointer<OpenContextImpl<Allocator>> OpenContextRetainer<Allocator>::Pull(ApplicationId appId, const Uid& uid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    return p ? p->Pull(uid) : nullptr;
}
template <typename Allocator>
int OpenContextRetainer<Allocator>::ListUsers(Uid buffer[], int count, ApplicationId appId) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    return p ? p->ListUsers(buffer, count) : 0;
}

// ---------------------------------------------------------------------------------------------
// Holder

template <typename Allocator>
void OpenContextHolder<Allocator>::Push(Context&& context) NN_NOEXCEPT
{
    NN_SDK_ASSERT(context);
    auto uid = context->GetUid();
    for (const auto& c : m_Contexts)
    {
        if (c && c->GetUid() == uid)
        {
            // すでに登録済みの場合は何もしない (Release 以外では失敗)
            NN_SDK_ASSERT(NN_STATIC_CONDITION(false));
            return;
        }
    }

    for (auto& c : m_Contexts)
    {
        if (!c)
        {
#if !defined(NN_SDK_BUILD_RELEASE)
            auto u = context->GetUid();
            NN_DETAIL_ACCOUNT_TRACE("Holder::Push: %08x_%08x_%08x_%08x\n",
                static_cast<uint32_t>(u._data[0] >> 32),
                static_cast<uint32_t>(u._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(u._data[1] >> 32),
                static_cast<uint32_t>(u._data[1] & 0xFFFFFFFFull));
#endif
            c = std::move(context);
            return;
        }
    }
    NN_ABORT("Unreachable\n");
}

template <typename Allocator>
typename OpenContextHolder<Allocator>::Context OpenContextHolder<Allocator>::Pull(const Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT(uid);
    for (auto& c : m_Contexts)
    {
        if (c && c->GetUid() == uid)
        {
#if !defined(NN_SDK_BUILD_RELEASE)
            auto u = c->GetUid();
            NN_DETAIL_ACCOUNT_TRACE("Holder::Pull: %08x_%08x_%08x_%08x\n",
                static_cast<uint32_t>(u._data[0] >> 32),
                static_cast<uint32_t>(u._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(u._data[1] >> 32),
                static_cast<uint32_t>(u._data[1] & 0xFFFFFFFFull));
#endif
            Context rval {};
            swap(rval, c);
            return rval;
        }
    }
    return nullptr;
}

template <typename Allocator>
int OpenContextHolder<Allocator>::ListUsers(Uid buffer[], int count) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(count >= UserCountMax);
    int ct = 0;
    for (const auto& c : m_Contexts)
    {
        if (c)
        {
#if !defined(NN_SDK_BUILD_RELEASE)
            auto u = c->GetUid();
            NN_DETAIL_ACCOUNT_TRACE("Holder::ListUsers: %08x_%08x_%08x_%08x\n",
                static_cast<uint32_t>(u._data[0] >> 32),
                static_cast<uint32_t>(u._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(u._data[1] >> 32),
                static_cast<uint32_t>(u._data[1] & 0xFFFFFFFFull));
#endif
            buffer[ct ++] = c->GetUid();
        }
    }
    return ct;
}
}}} // ~namespace nn::account::detail
