﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_TypesForSystemServices.h>

namespace nn {
namespace account {
namespace http {
class IOAuthProcedure;
} // ~namespace nn::account::http
namespace nas {
class IOAuthProcedureForNintendoAccountLinkage;
} // ~namesapce nn::account::nas

}} // ~namespace nn::account

namespace nn {
namespace account {
namespace detail {

/**
    @brief アカウントシステムでOAuth 2.0による認証や認可の手続きを実施するためのクラスです。
    @details
        アカウントシステムでOAuth 2.0による認証や認可の手続きを実施するためのクラスです。

        このクラスでは、OAuth 2.0で必要となるユーザーの認証や認可を、Webブラウザー等のユーザーエージェント経由で要求するために必要なパラメータを提供します。
        また認証や認可の結果を含むレスポンス文字列の入力を受け付けることで、認証や認可の結果をアカウントシステムに反映します。
*/
class OAuthProcedureBase
{
    NN_DISALLOW_COPY(OAuthProcedureBase);

private:
    Uid m_User;
    http::IOAuthProcedure* m_Ptr;

protected:
    http::IOAuthProcedure* GetPtr() const NN_NOEXCEPT;

public:
    /**
        @brief 特定の認証や認可の手続きに紐づかないオブジェクトを作成します。

        @post
            - このクラスの機能を使用不可能

        @details
            特定の認証や認可の手続きに紐づかないオブジェクトを作成します。
            この状態のオブジェクトは実際の手続きのためのオブジェクトを出力するAPIへの入力として使用します。
    */
    OAuthProcedureBase() NN_NOEXCEPT;
    virtual ~OAuthProcedureBase() NN_NOEXCEPT;


    /**
        @brief 認可手続きの事前処理を行います。

        @param[out] pOutContext         非同期処理の完了を通知するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - pOutContext != nullptr
            - インターネットと通信可能
        @post
            - GetRequest() を呼び出すことができる

        @details
            認可手続きの事前処理を行います。

            この関数はインターネットとの通信を行います。

            この関数は即時返り、実際の処理は非同期的に実行されます。
            処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。
    */
    Result PrepareAsync(AsyncContext* pOut);

    /**
        @brief ユーザーの認証や認可のリクエストのためのパラメータを取得します。

        @param[out] pOutRequestUrl  認証や認可を開始するためのリクエストURL
        @param[out] pOutCallbackUri 認証や認可の完了を示すコールバックURI

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - pOutRequestUrl != nullptr
            - pOutCallbackUri != nullptr

        @details
            ユーザーの認証や認可を得るためのパラメータを取得します。
            このパラメータには次の要素があります。

            - リクエストURL
            - Webブラウザー等のユーザーエージェントでこのURLに対してリクエストを行うと、ユーザーに認証や認可の手続きを求めることができます。
            - コールバックURI
            - 認証や認可の手続きが行われると、接続先のサーバーはここで得られるURIに対するリダイレクトを行います。

            ただし実際にリダイレクトされるURIには、URIフラグメントやクエリパラメータとして認証や認可の結果が含まれることがあります。
    */
    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT;

    /**
        @brief 認証や認可の結果をアカウントシステムに入力し、認証や認可の完了に必要な処理を実施します。

        @param[in]  response            認証や認可の結果を'\0'で終端した文字列
        @param[in]  responseBufferSize  responseで指定したバッファの大きさ

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - response != nullptr
            - response[0,responseBufferSize) の範囲に1つ以上の'\0'を含む
            - response は this->GetRequest() で取得したパラメータから得られた
            - インターネットと通信可能
        @post
            - 認証や認可の手続きが完了

        @details
            認証や認可の結果をアカウントシステムに入力し、認証や認可の完了に必要な処理を実施します。
            入力すべき情報が何かは、実際の手続きを取得するAPIで指定されます。
            (参考: nn::account::NetworkServiceAccountAdministrator::CreateProcedureToLinkWithNintendoAccount())
    */
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;

    /**
        @brief 認証や認可の結果をアカウントシステムに入力し、認証や認可の完了に必要な処理を実施します。

        @param[out] pOutContext         非同期処理の完了を通知するためのオブジェクト
        @param[in]  response            認証や認可の結果を'\0'で終端した文字列
        @param[in]  responseBufferSize  responseで指定したバッファの大きさ

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - pOutContext != nullptr
            - response != nullptr
            - response[0,responseBufferSize) の範囲に1つ以上の'\0'を含む
            - response は this->GetRequest() で取得したパラメータから得られた
            - インターネットと通信可能
        @post
            - 認証や認可の手続きが完了

        @details
            認証や認可の結果をアカウントシステムに入力し、認証や認可の完了に必要な処理を実施します。
            入力すべき情報が何かは、実際の手続きを取得するAPIで指定されます。
            (参考: nn::account::NetworkServiceAccountAdministrator::CreateProcedureToLinkWithNintendoAccount())

            この関数はインターネットとの通信を行います。

            この関数は即時返り、実際の処理は非同期的に実行されます。
            処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。
    */
    Result ApplyResponseAsync(AsyncContext* pOutContext, const char* response, size_t responseBufferSize) NN_NOEXCEPT;

    /**
        @brief 認証や認可の手続きを中断します。

        @param[out] pOut 中断した手続きの識別子

        @retresult
            @handleresult{
                nn::ResultSuccess,
                再開に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - pOut != nullptr
            - ApplyResponseAsync() で開始した非同期処理が実行中でない
        @post
            - 認証や認可の手続きが中断されている

        @details
            認証や認可の結果を中断し、再開する際に指定するためのセッション識別子を取得します。
    */
    Result Suspend(SessionId* pOut) NN_NOEXCEPT;

    // 内部向け
    OAuthProcedureBase(const Uid& user, http::IOAuthProcedure* ptr) NN_NOEXCEPT;
    OAuthProcedureBase(OAuthProcedureBase&& rhs) NN_NOEXCEPT;
    OAuthProcedureBase& operator=(OAuthProcedureBase&& rhs) NN_NOEXCEPT;
    OAuthProcedureBase& Swap(OAuthProcedureBase& rhs) NN_NOEXCEPT;
};

class SimpleOAuthProcedureBase final
    : detail::OAuthProcedureBase
{
    NN_DISALLOW_COPY(SimpleOAuthProcedureBase);

public:
    SimpleOAuthProcedureBase() NN_NOEXCEPT;
    virtual ~SimpleOAuthProcedureBase() NN_NOEXCEPT {};

    using OAuthProcedureBase::GetRequest;
    using OAuthProcedureBase::ApplyResponseAsync;
    using OAuthProcedureBase::Suspend;

    // 内部向け
    SimpleOAuthProcedureBase(const Uid& user, http::IOAuthProcedure* ptr) NN_NOEXCEPT;
    SimpleOAuthProcedureBase(SimpleOAuthProcedureBase&& rhs) NN_NOEXCEPT;
    SimpleOAuthProcedureBase& operator=(SimpleOAuthProcedureBase&& rhs) NN_NOEXCEPT;
    SimpleOAuthProcedureBase& Swap(SimpleOAuthProcedureBase& rhs) NN_NOEXCEPT;
};

} // ~namespace nn::account::detail
}
}
