﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_FileSystem.h>
#include <nn/account/detail/account_SaveData.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account { namespace detail {

// 各モジュールで利用される抽象化されたストレージクラス
class AbstractLocalStorage
{
public:
    virtual const char* GetRootPath() const NN_NOEXCEPT = 0;
    virtual const detail::AbstractFileSystem& GetFileSystem() const NN_NOEXCEPT = 0;
    virtual Result Commit() const NN_NOEXCEPT = 0;
    virtual FsLockGuard AcquireWriterLock() const NN_NOEXCEPT = 0;
    virtual detail::Uuid GenerateUuidWithContext() const NN_NOEXCEPT = 0;
};

template <typename FileSystem, typename SaveDataPolicy>
class LocalStorage
    : public AbstractLocalStorage
{
    static_assert(std::is_base_of<AbstractFileSystem, FileSystem>::value, "typename FileSystem is not derived from detail::AbstractFileSystem");
    NN_DISALLOW_COPY(LocalStorage);

public:
    static const size_t VolumeNameLengthMax = 8;

private:
    FileSystem m_Fs;
    typename SaveDataPolicy::SaveData m_SaveData;

    bool m_IsMounted;
    char m_RootPath[VolumeNameLengthMax + sizeof(SaveDataDirectoryName) + 1];

public:
    LocalStorage() NN_NOEXCEPT;
    virtual ~LocalStorage() NN_NOEXCEPT;

    // AbstractLocalStorage 実装
    virtual const char* GetRootPath() const NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_RootPath;
    }
    virtual const detail::AbstractFileSystem& GetFileSystem() const NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_Fs;
    }
    virtual Result Commit() const NN_NOEXCEPT final NN_OVERRIDE;
    virtual FsLockGuard AcquireWriterLock() const NN_NOEXCEPT final NN_OVERRIDE;
    virtual detail::Uuid GenerateUuidWithContext() const NN_NOEXCEPT final NN_OVERRIDE;

    // StorageManager 実装
    Result Mount() NN_NOEXCEPT;
    Result Setup() const NN_NOEXCEPT;

    void ClearProfile() const NN_NOEXCEPT;
    void ClearCache() const NN_NOEXCEPT;
    void Clear() const NN_NOEXCEPT;
};

}}} // ~namespace nn::account::detail


/* --------------------------------------------------------------------------------------------
    実装
 */

#include <nn/nn_Abort.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace account { namespace detail {

template <typename FileSystem, typename SaveDataPolicy>
inline LocalStorage<FileSystem, SaveDataPolicy>::LocalStorage() NN_NOEXCEPT
    : m_IsMounted(false)
{
}
template <typename FileSystem, typename SaveDataPolicy>
inline LocalStorage<FileSystem, SaveDataPolicy>::~LocalStorage() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        FileSystem::Unmount(m_SaveData.VolumeName);
        m_IsMounted = false;
    }
}
template <typename FileSystem, typename SaveDataPolicy>
inline Result LocalStorage<FileSystem, SaveDataPolicy>::Commit() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    return m_SaveData.template Commit<FileSystem>();
}
template <typename FileSystem, typename SaveDataPolicy>
inline FsLockGuard LocalStorage<FileSystem, SaveDataPolicy>::AcquireWriterLock() const NN_NOEXCEPT
{
    return FileSystem::AcquireWriterLock();
}
template <typename FileSystem, typename SaveDataPolicy>
inline detail::Uuid LocalStorage<FileSystem, SaveDataPolicy>::GenerateUuidWithContext() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    return m_SaveData.template GenerateUuid<FileSystem>();
}
template <typename FileSystem, typename SaveDataPolicy>
inline Result LocalStorage<FileSystem, SaveDataPolicy>::Mount() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsMounted);
    NN_RESULT_DO(m_SaveData.template Mount<FileSystem>());

    auto l = util::SNPrintf(
        m_RootPath, sizeof(m_RootPath),
        "%s:%s", m_SaveData.VolumeName, SaveDataDirectoryName);
    NN_ABORT_UNLESS(
        l < static_cast<int>(sizeof(m_RootPath)),
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Too long volume name of SaveData (Internal Error)\n");

    m_IsMounted = true;
    NN_RESULT_SUCCESS;
}
template <typename FileSystem, typename SaveDataPolicy>
inline Result LocalStorage<FileSystem, SaveDataPolicy>::Setup() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    return FileSystem::Setup(GetRootPath());
}
template <typename FileSystem, typename SaveDataPolicy>
inline void LocalStorage<FileSystem, SaveDataPolicy>::ClearProfile() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    FileSystem::ClearProfile(GetRootPath());
}
template <typename FileSystem, typename SaveDataPolicy>
inline void LocalStorage<FileSystem, SaveDataPolicy>::ClearCache() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    FileSystem::ClearCache(GetRootPath());
}
template <typename FileSystem, typename SaveDataPolicy>
inline void LocalStorage<FileSystem, SaveDataPolicy>::Clear() const NN_NOEXCEPT
{
    auto lock = AcquireWriterLock();
    FileSystem::Clear(GetRootPath());
}

}}} // ~namespace nn::account::detail
