﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_AsyncContextImpl.h>
#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/srepo/srepo_StateNotifier.h>

namespace nn {
namespace account {
class Executor;

namespace profile {
class ProfileStorage;
} // ~namespace nn::account::profile

namespace user {
class UserRegistry;
} // ~namespace nn::account::user

} // ~namespace nn::account
}

namespace nn { namespace account { namespace baas {

class UserRegistrar
{
    NN_DISALLOW_COPY(UserRegistrar);

private:
    user::UserRegistry& m_Registry;
    profile::ProfileStorage& m_ProfileStorage;

    Result CompleteRegistrationImpl(const Uid& uid, const UserProfile& profile, const void* image, size_t imageSize) NN_NOEXCEPT;

public:
    UserRegistrar(
        user::UserRegistry& registry, profile::ProfileStorage& profileStorage) NN_NOEXCEPT
        : m_Registry(registry)
        , m_ProfileStorage(profileStorage)
    {
    }
    Result Create(Uid* pOut, const UserProfile& profile, const void* image, size_t imageSize) NN_NOEXCEPT;
    Result Import(const Uid& uid, const UserProfile& profile, const void* image, size_t imageSize) NN_NOEXCEPT;
};

template <typename Allocator>
class FloatingRegistrationRequestWithNintendoAccountImpl
    : public sf::ISharedObject
{
    NN_DISALLOW_COPY(FloatingRegistrationRequestWithNintendoAccountImpl);
    friend class IdTokenEnsuranceTask;
    friend class NsaRegistrationTask;

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    UserRegistrar m_Registrar;

    nas::NasSessionPool<Allocator>& m_NasSessionPool;
    Executor& m_Executor;

    detail::Uuid m_SessionId;
    nas::NasFloatingRegistrationProcedure* m_pProcedure;

    Uid m_Uid;
    detail::ApplicationInfo m_AppInfo;

    class IdTokenEnsuranceTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<FloatingRegistrationRequestWithNintendoAccountImpl<Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
        nas::NasFloatingRegistrationProcedure* m_pProcedure;
        detail::ApplicationInfo m_AppInfo;
    public:
        explicit IdTokenEnsuranceTask(ParentPtr&& parent, nas::NasFloatingRegistrationProcedure* pProcedure, const detail::ApplicationInfo& appInfo) NN_NOEXCEPT
            : m_Parent(std::move(parent))
            , m_pProcedure(pProcedure)
            , m_AppInfo(appInfo)
        {
        }
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidApplication());
            return m_pProcedure->EnsureNetworkServiceAccountIdToken(m_AppInfo, resource, this);
        }
    };

    class NsaRegistrationTask
        : public detail::Executable<ExecutionResource>
    {
    public:
        typedef sf::SharedPointer<FloatingRegistrationRequestWithNintendoAccountImpl<Allocator>> ParentPtr;
    private:
        ParentPtr m_Parent;
        nas::NasFloatingRegistrationProcedure* m_pProcedure;
    public:
        explicit NsaRegistrationTask(ParentPtr&& parent, nas::NasFloatingRegistrationProcedure* pProcedure) NN_NOEXCEPT
            : m_Parent(std::move(parent))
            , m_pProcedure(pProcedure)
        {
        }
        virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Parent->m_Uid);
            return m_pProcedure->RegisterNetworkServiceAccount(m_Parent->m_Uid, resource, this);
        }
    };

    Result RegisterNetworkServiceAccountAsyncImpl(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
    {
        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<NsaRegistrationTask, ExecutionResource>>(
            &m_Allocator, typename NsaRegistrationTask::ParentPtr(this, true), m_pProcedure);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOutContext = std::move(p);
        NN_RESULT_SUCCESS;
    }

public:
    FloatingRegistrationRequestWithNintendoAccountImpl(
        Allocator& allocator,
        user::UserRegistry& registry, profile::ProfileStorage& profileStorage,
        nas::NasSessionPool<Allocator>& nasSessionPool,  Executor& executor) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_Registrar(registry, profileStorage)
        , m_NasSessionPool(nasSessionPool)
        , m_Executor(executor)
        , m_SessionId(detail::InvalidUuid)
        , m_pProcedure(nullptr)
        , m_Uid(InvalidUid)
        , m_AppInfo(detail::InvalidApplicationInfo)
    {
    }
    ~FloatingRegistrationRequestWithNintendoAccountImpl() NN_NOEXCEPT
    {
        if (m_SessionId)
        {
            m_NasSessionPool.ReleaseNasFloatingRegistrationSession(m_SessionId);
        }
    }

    Result Initialize(
        ndas::NdasOperator& ndasOp, BaasOperator& baasOp, nas::NasOperator& nasOp,
        sf::NativeHandle&& transferMemoryHandle, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(size >= nas::NasGuestLoginProcedure::RequiredMemorySize, ResultInsufficientBuffer());
        NN_RESULT_THROW_UNLESS(!m_SessionId, ResultInvalidProtocolAccess());

        NN_RESULT_DO(m_NasSessionPool.CreateNasFloatingRegistrationSession(&m_SessionId, &m_pProcedure, ndasOp, baasOp, nasOp));
        m_pProcedure->AttachTransferMemory(transferMemoryHandle.GetOsHandle(), size, transferMemoryHandle.IsManaged());
        transferMemoryHandle.Detach();

        NN_RESULT_DO(m_pProcedure->Initialize());
        NN_RESULT_SUCCESS;
    }
    Result InvokeWithoutInteractionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>>) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }

    Result GetSessionId(sf::Out<detail::Uuid> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        *pOut = m_SessionId;
        NN_RESULT_SUCCESS;
    }

    Result RegisterUser(sf::Out<Uid> pOutUid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_Uid, ResultInvalidProtocolAccess());
        size_t imageSize;
        auto* image = m_pProcedure->GetProfileImagePtr(&imageSize);
        Uid uid;
        NN_RESULT_DO(m_Registrar.Create(&uid, m_pProcedure->GetUserProfileRef(), image, imageSize));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserRegistered(uid);
#endif

        *pOutUid = m_Uid = uid;
        NN_RESULT_SUCCESS;
    }
    Result RegisterUserWithUid(const Uid& uid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_Uid, ResultInvalidProtocolAccess());
        size_t imageSize;
        auto* image = m_pProcedure->GetProfileImagePtr(&imageSize);
        NN_RESULT_DO(m_Registrar.Import(uid, m_pProcedure->GetUserProfileRef(), image, imageSize));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserRegistered(uid);
#endif

        m_Uid = uid;
        NN_RESULT_SUCCESS;
    }
    Result RegisterNetworkServiceAccountAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Uid, ResultInvalidProtocolAccess());
        return RegisterNetworkServiceAccountAsyncImpl(pOutContext);
    }
    Result RegisterNetworkServiceAccountWithUidAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& uid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_Uid, ResultInvalidProtocolAccess());
        m_Uid = uid;
        return RegisterNetworkServiceAccountAsyncImpl(pOutContext);
    }

    Result SetSystemProgramIdentification(SystemProgramIdentification identification, Bit64 pid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        NN_UNUSED(pid);
        detail::ApplicationInfo appInfo = detail::ApplicationInfo::Get(identification.id.value, identification.version, detail::ApplicationMediaType::System);
        NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
        m_AppInfo = appInfo;
        NN_RESULT_SUCCESS;
    }
    Result EnsureIdTokenCacheAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::AsyncTask<IdTokenEnsuranceTask, ExecutionResource>>(
            &m_Allocator, typename IdTokenEnsuranceTask::ParentPtr(this, true), m_pProcedure, m_AppInfo);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
        *pOutContext = std::move(p);
        NN_RESULT_SUCCESS;
    }
    Result LoadIdTokenCache(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->LoadBaasIdTokenCache(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize(), m_AppInfo));
        NN_SDK_ASSERT(sizeActual <= detail::BaasIdTokenSizeMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
    Result GetAccountId(sf::Out<NetworkServiceAccountId> pOutId) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetNetworkServiceAccountId(pOutId.GetPointer());
    }
    Result GetLinkedNintendoAccountId(sf::Out<NintendoAccountId> pOut) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetLinkedNintendoAccountId(pOut.GetPointer());
    }
    Result GetNickname(const sf::OutArray<char>& nickname) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());
        return m_pProcedure->GetNickname(nickname.GetData(), nickname.GetLength());
    }
    Result GetProfileImage(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOutImage) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->GetProfileImage(&sizeActual, pOutImage.GetPointerUnsafe(), pOutImage.GetSize()));
        NN_SDK_ASSERT(sizeActual <= ProfileImageBytesMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::account::baas
