﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_NetworkServiceLicenseKind.h>
#include <nn/account/account_Types.h>
#include <nn/nn_Result.h>

namespace nn { namespace account { namespace detail {

class IAsyncNetworkServiceLicenseKindContext;

}}} // ~namespace nn::account::detail

namespace nn { namespace account {

/**
    @brief ネットワークサービスの利用資格を取得する非同期処理の状況と、取得結果を取得するためのクラス。
    @details
        このクラスは「ネットワークサービスの利用資格を取得する非同期処理」について、次を行います。
        - 非同期処理の完了の通知を受けるイベントの取得
        - 非同期処理の完了, 未完了の状態の取得
        - 非同期処理が完了している場合、その結果の取得
        - 非同期処理が未完了の場合、その中止の要求

        このクラスのオブジェクトは次の状態をもちます。
        - 無効
            - オブジェクトはネットワークサービスの利用資格を取得する非同期処理と関連づきません。
            - この状態では、デストラクタ以外を呼び出すことはできません。
        - 有効
            - このオブジェクトはネットワークサービスの利用資格を取得する非同期処理と関連づきます。
            - この状態では関連づく非同期処理に関して先述の操作が可能です。

        LoadNetworkServiceLicenseKindAsync() の引数に AsyncNetworkServiceLicenseKindContext オブジェクトを指定することで、有効なオブジェクトを取得できます。
        非同期処理に成功した場合、 @ref GetNetworkServiceLicenseKind() でネットワークサービスの利用資格を取得できます。
 */
class AsyncNetworkServiceLicenseKindContext : public AsyncContext
{
    NN_DISALLOW_COPY(AsyncNetworkServiceLicenseKindContext);
    friend Result LoadNetworkServiceLicenseKindAsync(AsyncNetworkServiceLicenseKindContext* pOut, const UserHandle& handle) NN_NOEXCEPT;

private:
    explicit AsyncNetworkServiceLicenseKindContext(detail::IAsyncNetworkServiceLicenseKindContext* ptr) NN_NOEXCEPT;

public:
    //! @copydoc AsyncContext::AsyncContext()
    AsyncNetworkServiceLicenseKindContext() NN_NOEXCEPT;
    //! @copydoc AsyncContext::AsyncContext(AsyncContext&&)
    AsyncNetworkServiceLicenseKindContext(AsyncNetworkServiceLicenseKindContext&& rhs) NN_NOEXCEPT;
    //! @copydoc AsyncContext::operator=(AsyncContext&&)
    AsyncNetworkServiceLicenseKindContext& operator=(AsyncNetworkServiceLicenseKindContext&& rhs) NN_NOEXCEPT;

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief ネットワークサービスの利用資格を取得する非同期処理の結果を取得します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultCancelled,
                Cancel() 関数の呼び出しによって処理がキャンセルされました。
                この結果をエラーとして扱わないでください。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                非同期処理中にネットワークサービスアカウントが利用不可能であると判断されました。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
            @handleresult{
                nn::account::ResultNetworkCommunicationError,
                非同期処理中のネットワーク通信において問題が発生しました。
                このエラーが発生した場合は可能な限り無視をせず、 nn::err::ShowError() 関数を使用してユーザーに通知してください。
            }
            @handleresult{
                nn::ssl::ResultSslService,
                非同期処理中のネットワーク通信において問題が発生しました。
                このエラーが発生した場合は可能な限り無視をせず、 nn::err::ShowError() 関数を使用してユーザーに通知してください。
            }
        @endretresult

        @pre
            - *this が有効
            - 非同期処理が完了している
    */
    Result GetResult() NN_NOEXCEPT;
#endif

    /**
        @brief ネットワークサービスの利用資格を取得します。

        @pre
            - *this が有効
            - GetResult().IsSuccess() == true
    */
    NetworkServiceLicenseKind GetNetworkServiceLicenseKind() NN_NOEXCEPT;
};


/**
    @brief 指定したユーザーのネットワークサービスアカウントと連携しているニンテンドーアカウントについて、
           ネットワークサービスの利用資格をサーバーに問い合わせます。

    @param[out] pOut ネットワークサービスの利用資格を取得する非同期処理の状況と、取得結果を取得するためのオブジェクトの格納先を指すポインタ
    @param[in] handle 対象のユーザーを指す UserHandle

    @retresult
        @handleresult{
            nn::ResultSuccess,
            成功しました。
        }
        @handleresult{
            nn::account::ResultNetworkServiceAccountUnavailable,
            指定したユーザーのネットワークサービスアカウントを利用可能ではありません。
            EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
        }
    @endretresult

    @pre
        - pOut != nullptr
        - pOut が無効状態

    @post
        - pOut が有効状態

    @details
        指定したユーザーのネットワークサービスアカウントと連携しているニンテンドーアカウントについて、
        ネットワークサービスの利用資格をサーバーに問い合わせます。

        この関数はインターネットとの通信を行います。

        この関数は即時返り、実際の処理は非同期的に実行されます。
        処理の完了の通知は、 *pOut から取得可能な nn::os::SystemEvent のシグナルによって行われます。

        handle に指定した対象のユーザーは、 @ref AsyncNetworkServiceLicenseKindContext オブジェクトが有効な期間はオープン状態である必要があります。
*/
Result LoadNetworkServiceLicenseKindAsync(AsyncNetworkServiceLicenseKindContext* pOut, const UserHandle& handle) NN_NOEXCEPT;

}} // ~namespace nn::account
