﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.CodeDom.Compiler;

namespace Nintendo.ServiceFramework.SfdlToCppCode
{
    internal static class SfdlCppCodeGenerator
    {
        private static void EnsureTextFile(string outputPath, string text, Encoding encoding)
        {
            try
            {
                if (System.IO.File.ReadAllText(outputPath, encoding) == text)
                {
                    return;
                }
            }
            catch
            {
                // nop
            }
            System.IO.File.WriteAllText(outputPath, text, encoding);
        }

        private static void GenerateCppCodeImpl(string outputPath, string headerPath, IEnumerable<string> typeNames)
        {
            var b = new StringBuilder();
            if (headerPath == null)
            {
                b.Append(global::SfdlToCppCode.Resource1.SfCppCodeHeaderTemplate);
            }
            using (var w = new System.IO.StringWriter(b))
            {
                var rawGen = new Nintendo.ServiceFramework.CppCode.RawCppCodeGenerator(w);
                rawGen.IsNoLintEnabled = true;
                if (headerPath != null)
                {
                    foreach (var line in System.IO.File.ReadAllLines(headerPath, Encoding.UTF8))
                    {
                        rawGen.WriteLine(line);
                    }
                }
                var entities = typeNames.Select(typeName => Type.GetType(typeName, true).GetSfEntity()).ToArray();
                new Nintendo.ServiceFramework.CppCode.CppCodeGenerator().Generate(rawGen, entities);
                rawGen.Finish();
            }
            EnsureTextFile(outputPath, b.ToString(), Encoding.UTF8);
        }

        public static void GenerateCppCode(string outputPath, string headerPath, IEnumerable<string> typeNames)
        {
            try
            {
                GenerateCppCodeImpl(outputPath, headerPath, typeNames);
            }
            catch (System.Reflection.TargetInvocationException te)
            {
                throw te.InnerException;
            }
        }
    }

    internal class Program
    {
        private class Args
        {
            public string OutputPath { get; private set; }
            public string HeaderTemplatePath { get; private set; }
            public IEnumerable<string> TypeNames { get; private set; }
            public IEnumerable<string> SourceFilePaths { get; private set; }

            public Args(string[] args)
            {
                var outputPath = default(string);
                var headerPath = default(string);
                var typeNames = new List<string>();
                var sources = new List<string>();
                var i = 0;
                while (i < args.Length)
                {
                    var s = args[i++];
                    switch (s)
                    {
                        case "-o":
                        case "--output-file":
                            outputPath = new System.IO.FileInfo(args[i++]).FullName;
                            continue;
                        case "--header-template":
                            headerPath = new System.IO.FileInfo(args[i++]).FullName;
                            continue;
                        case "-t":
                        case "--type":
                            typeNames.Add(args[i++]);
                            continue;
                        default:
                            sources.Add(new System.IO.FileInfo(s).FullName);
                            continue;
                    }
                }
                this.OutputPath = outputPath;
                this.HeaderTemplatePath = headerPath;
                this.TypeNames = typeNames;
                this.SourceFilePaths = sources;
            }
        }

        private static void LoadFiles(IEnumerable<string> sources)
        {
            var compiler = CodeDomProvider.CreateProvider("C#");
            var compilerOptions = new CompilerParameters();
            compilerOptions.GenerateInMemory = true;
            var asmName = typeof(nn.sf.IServiceObject).Assembly.Location;
            compilerOptions.ReferencedAssemblies.AddRange(new[] { asmName, "System.dll", "System.Core.dll" });
            var r = compiler.CompileAssemblyFromFile(compilerOptions, sources.ToArray());
            foreach (var s in r.Errors)
            {
                Console.Error.WriteLine(s);
            }
            if (r.Errors.HasErrors)
            {
                throw new Exception("compile error.");
            }
            AppDomain.CurrentDomain.TypeResolve += (_, args) =>
            {
                return r.CompiledAssembly;
            };
        }

        private static void Main(string[] argStrings)
        {
            Args args;
            try
            {
                args = new Args(argStrings);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.ToString());
                Environment.ExitCode = -1;
                return;
            }
            try
            {
                LoadFiles(args.SourceFilePaths);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                Environment.ExitCode = -1;
                return;
            }
            try
            {
                SfdlCppCodeGenerator.GenerateCppCode(args.OutputPath, args.HeaderTemplatePath, args.TypeNames);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.ToString());
                Environment.ExitCode = -1;
                return;
            }
        }
    }
}
