﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public class InterfaceCppCodeEmitter : EntityCppCodeEmitterBase<SfInterface>
    {
        internal InterfaceCppCodeEmitter(SfInterface entity)
            : base(entity)
        {
        }

        private static readonly string[] CommonIncludes = new[]
        {
            @"<nn/sf/detail/sf_AutogenInterfaceIncludes.h>",
        };

        private static readonly string[] CmifCommonIncludes = new[]
        {
            @"<nn/sf/cmif/detail/sf_CmifAutogenInterfaceIncludes.h>",
        };

        private static readonly string[] ImplCommonIncludes = new[]
        {
            @"<nn/sf/impl/detail/sf_AutogenImplIncludes.h>",
        };

        public override string GetCppTypeString(bool friendly)
        {
            return string.Format(@"{0}nn::sf::SharedPointer<{0}{1}>", friendly ? string.Empty : @"::", GetMyCppName().FullName);
        }

        public override IEnumerable<string> GetAdditionalIncludes()
        {
            foreach (var e in CommonIncludes)
            {
                yield return e;
            }
            if (UseCmifAttribute.GetUseCmif(Entity.InnerType))
            {
                foreach (var e in CmifCommonIncludes)
                {
                    yield return e;
                }
            }
            if (UseImplAttribute.GetUseImpl(Entity.InnerType))
            {
                foreach (var e in ImplCommonIncludes)
                {
                    yield return e;
                }
            }
        }

        public override void EmitForwardDeclarationCode(RawCppCodeGenerator rawGen)
        {
            ChangeMyNameSpace(rawGen);
            rawGen.WriteLine(@"class {0};", GetMyCppName().GetSingleName());
        }

        #region EmitDefinitionCode

        public override void EmitDefinitionCode(RawCppCodeGenerator rawGen)
        {
            var cppName = GetMyCppName();
            var name = cppName.GetSingleName();
            var fullName = cppName.FullName;
            var entity = Entity.InnerType.GetSfEntity();
            var methods = Entity.InnerType.GetMethods().SortByMethodId().ToArray();
            var baseInterface = Entity.GetBaseInterface();
            EmitServiceInterface(rawGen, name, methods, baseInterface);
            rawGen.AutoNewLine();
            EmitDefinitionGuide(rawGen, name, methods, baseInterface);
            if (UseImplAttribute.GetUseImpl(Entity.InnerType))
            {
                rawGen.AutoNewLine();
                var allMethods = Entity.InnerType.GetAllInterfaceMethods()
                    .Where((m) => !typeof(nn.sf.IServiceObject).GetMethods().Contains(m))
                    .ToArray();
                EmitImplDefinitionGuide(rawGen, allMethods.SortByMethodId().ToArray());
                rawGen.AutoNewLine();
                EmitImplTemplate(rawGen, fullName, methods, baseInterface);
            }
            if (UseCmifAttribute.GetUseCmif(Entity.InnerType))
            {
                rawGen.AutoNewLine();
                EmitMethodInfo(rawGen, entity, fullName, methods, baseInterface);
                rawGen.AutoNewLine();
                EmitCmifProxy(rawGen, fullName, methods, baseInterface);
                rawGen.AutoNewLine();
                var cmifMethods = methods.Where(m => UseCmifAttribute.GetUseCmif(m)).ToArray();
                EmitCmifDispatcher(rawGen, fullName, cmifMethods, baseInterface);
            }
        }

        #region EmitServiceInterface

        private void EmitServiceInterface(RawCppCodeGenerator rawGen, string name, MethodInfo[] methods, SfEntity baseInterface)
        {
            ChangeMyNameSpace(rawGen);
            var baseName = @"::" + new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
            rawGen.WriteLine(@"NN_SF_DETAIL_DEFINE_SERVICE_INTERFACE({0}, ({1}))", name, baseName);
            using (rawGen.Indent())
            {
                rawGen.AutoNewLine();
                EmitSyncMethods(rawGen, name, methods);
                rawGen.AutoNewLine();
                EmitMethodAccessors(rawGen, name, methods);
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_DETAIL_DEFINE_SERVICE_INTERFACE_END");
        }

        private static void EmitSyncMethods(RawCppCodeGenerator rawGen, string name, MethodInfo[] methods)
        {
            rawGen.WriteLine(@"NN_SF_DETAIL_SERVICE_INTERFACE_BEGIN_SYNC_METHOD({0})", name);
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.WriteLine(GetSyncMethodDefinitionString(method));
                }
            }
            rawGen.WriteLine(@"NN_SF_DETAIL_SERVICE_INTERFACE_END_SYNC_METHOD");
        }

        private static string GetSyncMethodDefinitionString(MethodInfo method)
        {
            return method.GetMacroFunctionString(@"NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD", true, false);
        }

        private static string GetMethodAccessorHeader(MethodInfo method, bool friendly)
        {
            var args = method.GetParameterStringList(true, friendly);
            return string.Format(@"{0} {1}({2}) NN_NOEXCEPT",
                method.GetReturnTypeString(friendly),
                method.Name,
                string.Join(@", ", args.ToArray()));
        }

        private static string GetMethodAccessorImplementation(MethodInfo method)
        {
            return method.GetMacroFunctionString(@"NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD", false, false, @";");
        }

        private static void EmitMethodAccessors(RawCppCodeGenerator rawGen, string name, MethodInfo[] methods)
        {
            rawGen.WriteLine(@"NN_SF_DETAIL_SERVICE_INTERFACE_BEGIN_ACCESSOR({0})", name);
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.AutoNewLine();
                    rawGen.WriteLine(GetMethodAccessorHeader(method, false));
                    rawGen.WriteLine(@"{");
                    using (rawGen.Indent())
                    {
                        EmitInterfacePreConditions(rawGen, method);
                        rawGen.WriteLine(GetMethodAccessorImplementation(method));
                    }
                    rawGen.WriteLine(@"}");
                }
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_DETAIL_SERVICE_INTERFACE_END_ACCESSOR");
        }

        private static void EmitInterfacePreConditions(RawCppCodeGenerator rawGen, MethodInfo method)
        {
            var inParameters = from p in method.GetParameters()
                               where TypeUtility.GetInOutType(p.ParameterType) == InOutType.In
                               select p;
            foreach (var p in inParameters)
            {
                foreach (var pre in PreBaseAttribute.GetPreAttributes(p, false))
                {
                    rawGen.WriteLine(@"NN_SF_DETAIL_CALL_INTERNAL_ASSERT_PRE(({0}), ""{1}"")", pre.MakeCppCheckString(p.Name), pre.MakeAssertionFailureString(p.Name));
                }
            }
        }

        #endregion

        #region EmitDefinitionGuide

        private void EmitDefinitionGuide(RawCppCodeGenerator rawGen, string name, MethodInfo[] methods, SfEntity baseInterface)
        {
            ChangeMyNameSpace(rawGen);
            rawGen.WriteLine(@"#if 0");
            using (rawGen.Indent())
            {
                if (baseInterface.InnerType == typeof(nn.sf.IServiceObject))
                {
                    rawGen.WriteLine(@"class {0}", name);
                }
                else
                {
                    var baseName = @"public ::" + new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
                    rawGen.WriteLine(@"class {0} : {1}", name, baseName);
                }
                rawGen.WriteLine(@"{");
                rawGen.WriteLine(@"public:");
                using (rawGen.Indent())
                {
                    foreach (var method in methods)
                    {
                        rawGen.WriteLine(@"{0};", GetMethodAccessorHeader(method, true));
                    }
                }
                rawGen.WriteLine(@"};");
            }
            rawGen.WriteLine(@"#endif");
        }

        #endregion

        #region EmitImplTemplate

        private void EmitImplTemplate(RawCppCodeGenerator rawGen, string fullName, MethodInfo[] methods)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"NN_SF_IMPL_DETAIL_DEFINE_IMPL_TEMPLATE_BASE((::{0}))", fullName);
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.AutoNewLine();
                    rawGen.WriteLine(GetBaseSyncMethodDefinitionString(method));
                    rawGen.WriteLine(@"{");
                    using (rawGen.Indent())
                    {
                        rawGen.WriteLine(GetMethodAccessorImplementationForImpl(method));
                    }
                    rawGen.WriteLine(@"}");
                }
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_IMPL_DETAIL_DEFINE_IMPL_TEMPLATE_BASE_END");
        }

        private void EmitImplTemplate(RawCppCodeGenerator rawGen, string fullName, MethodInfo[] methods, SfEntity baseInterface)
        {
            rawGen.CloseCurrentNamespace();

            if (baseInterface.InnerType == typeof(nn.sf.IServiceObject))
            {
                rawGen.WriteLine(@"NN_SF_IMPL_DETAIL_DEFINE_IMPL_TEMPLATE_BASE((::{0}))", fullName);
            }
            else
            {
                var baseName = new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
                rawGen.WriteLine(@"NN_SF_IMPL_DETAIL_DEFINE_IMPL_TEMPLATE_BASE_WITH_BASE((::{0}), (::{1}))", fullName, baseName);
            }

            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.AutoNewLine();
                    rawGen.WriteLine(GetBaseSyncMethodDefinitionString(method));
                    rawGen.WriteLine(@"{");
                    using (rawGen.Indent())
                    {
                        rawGen.WriteLine(GetMethodAccessorImplementationForImpl(method));
                    }
                    rawGen.WriteLine(@"}");
                }
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_IMPL_DETAIL_DEFINE_IMPL_TEMPLATE_BASE_END");
        }

        private static string GetBaseSyncMethodDefinitionString(MethodInfo method)
        {
            return method.GetMacroFunctionString(@"NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD", true, false);
        }

        private string GetMethodAccessorImplementationForImpl(MethodInfo method)
        {
            return method.GetMacroFunctionString(@"NN_SF_IMPL_DETAIL_RETURN_IMPL_METHOD", false, false, @";");
        }

        private void EmitImplTemplateDefinition(RawCppCodeGenerator rawGen, string name, CppName interfaceCppName)
        {
            ChangeMyNameSpace(rawGen);
            rawGen.WriteLine(@"NN_SF_IMPL_DEFINE_IMPL_TEMPLATE({0}, (::{1}))", name, interfaceCppName.FullName);
        }

        private static void EmitDefinitionGuide(RawCppCodeGenerator rawGen, MethodInfo[] methods)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"#if 0");
            rawGen.AutoNewLine();
            rawGen.WriteLine(@"// Template for implementation users");
            rawGen.WriteLine(@"class Impl");
            rawGen.WriteLine(@"{");
            rawGen.WriteLine(@"public:");
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.WriteLine(@"{0};", GetMethodAccessorHeader(method, true));
                }
            }
            rawGen.WriteLine(@"};");
            rawGen.AutoNewLine();
            rawGen.WriteLine(@"#endif");
        }

        private static void EmitImplDefinitionGuide(RawCppCodeGenerator rawGen, MethodInfo[] methods)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"#if 0");
            rawGen.AutoNewLine();
            rawGen.WriteLine(@"// Template for implementation users");
            rawGen.WriteLine(@"class Impl");
            rawGen.WriteLine(@"{");
            rawGen.WriteLine(@"public:");
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.WriteLine(@"{0};", GetMethodAccessorHeader(method, true));
                }
            }
            rawGen.WriteLine(@"};");
            rawGen.AutoNewLine();
            rawGen.WriteLine(@"#endif");
        }

        #endregion

        #region InOutInfos

        private class InOutInfo
        {
            public Type Type { get; set; }
            public string Name { get; set; }
            public ParameterInfo ParameterInfo { get; set; }
            public InOutType InOutType { get; set; }
        }

        private class InOutInfos
        {
            public bool InProcessIdEnable { get; set; }
            public List<InOutInfo> All { get; set; }
            public List<InOutInfo> InRaws { get; set; }
            public List<InOutInfo> OutRaws { get; set; }
            public List<InOutInfo> Buffers { get; set; }
            public List<InOutInfo> VariableSizeBuffers { get; set; }
            public List<InOutInfo> FixedSizeBuffers { get; set; }
            public List<InOutInfo> InObjects { get; set; }
            public List<InOutInfo> OutObjects { get; set; }
            public List<InOutInfo> InNativeHandles { get; set; }
            public List<InOutInfo> OutNativeHandles { get; set; }
        }

        private static InOutInfos MakeInOutInfos(MethodInfo method)
        {
            var inProcessIdEnable = false;
            var all = new List<InOutInfo>();
            var inRaws = new List<InOutInfo>();
            var outRaws = new List<InOutInfo>();
            var variableSizedBuffers = new List<InOutInfo>();
            var fixedSizeBuffers = new List<InOutInfo>();
            var inObjects = new List<InOutInfo>();
            var outObjects = new List<InOutInfo>();
            var inNativeHandles = new List<InOutInfo>();
            var outNativeHandles = new List<InOutInfo>();
            Func<Type, ParameterInfo, InOutType, List<InOutInfo>> selectList = (Type t, ParameterInfo p, InOutType inOutType) =>
            {
                var e = t.GetSfEntity();
                switch (inOutType)
                {
                    case InOutType.In:
                        {
                            if (e is SfNnBuffer)
                            {
                                return variableSizedBuffers;
                            }
                            else if (e is SfValueType)
                            {
                                if (p.GetCustomAttribute<Hipc.ProcessIdAttribute>() != null)
                                {
                                    inProcessIdEnable = true;
                                }
                                if (CppCode.PreBaseAttribute.GetPreAttributes(p, true).Any(a => a.UsesProcessId))
                                {
                                    inProcessIdEnable = true;
                                }
                                if (p != null && BufferParameterAttribute.IsBufferParameter(p))
                                {
                                    return fixedSizeBuffers;
                                }
                                else
                                {
                                    return inRaws;
                                }
                            }
                            else if (e is SfNativeHandle)
                            {
                                return inNativeHandles;
                            }
                            else
                            {
                                return inObjects;
                            }
                        }
                    case InOutType.Out:
                        {
                            if (e is SfValueType)
                            {
                                if (p != null && BufferParameterAttribute.IsBufferParameter(p))
                                {
                                    return fixedSizeBuffers;
                                }
                                else
                                {
                                    return outRaws;
                                }
                            }
                            else if (e is SfNativeHandle)
                            {
                                return outNativeHandles;
                            }
                            else
                            {
                                return outObjects;
                            }
                        }
                    default:
                        {
                            throw new WrongImplementationException();
                        }
                }
            };
            foreach (var p in MakeParameterInfos(method))
            {
                var name = p.Item1;
                var t = p.Item2;
                var typeAndNameWithParameterInfo = new InOutInfo { Name = p.Item1, Type = p.Item2, ParameterInfo = p.Item4, InOutType = p.Item3 };
                all.Add(typeAndNameWithParameterInfo);
                selectList(t, p.Item4, p.Item3).Add(typeAndNameWithParameterInfo);
            }
            return new InOutInfos
            {
                InProcessIdEnable = inProcessIdEnable,
                All = all,
                InRaws = inRaws.OrderBy(x => ((SfValueType)x.Type.GetSfEntity()).Alignment).ToList(),
                OutRaws = outRaws.OrderBy(x => ((SfValueType)x.Type.GetSfEntity()).Alignment).ToList(),
                Buffers = fixedSizeBuffers.Concat(variableSizedBuffers).ToList(),
                VariableSizeBuffers = variableSizedBuffers,
                FixedSizeBuffers = fixedSizeBuffers,
                InObjects = inObjects,
                OutObjects = outObjects,
                InNativeHandles = inNativeHandles,
                OutNativeHandles = outNativeHandles,
            };
        }

        private static IEnumerable<Tuple<string, Type, InOutType, ParameterInfo>> MakeParameterInfos(MethodInfo method)
        {
            var parameters = from p in method.GetParameters()
                             let t = p.ParameterType
                             select Tuple.Create(p.Name, t.GetSfParameterElementType(), t.GetInOutType(), p);
            var returnType = method.ReturnType;
            if (returnType == typeof(void) || returnType == typeof(nn.Result))
            {
                return parameters;
            }
            else
            {
                return new[] { Tuple.Create(@"NN_SF_DETAIL_OUT_RETURN_NAME", returnType, InOutType.Out, default(ParameterInfo)) }.Concat(parameters);
            }
        }

        #endregion

        #region EmitMethodInfo

        private void EmitMethodInfo(RawCppCodeGenerator rawGen, SfEntity entity, string fullName, MethodInfo[] methods, SfEntity baseInterface)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"// template <> struct nn::sf::cmif::detail::MethodInfos<{0}>", fullName);

            if (baseInterface.InnerType == typeof(nn.sf.IServiceObject))
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFOS((::{0}))", fullName);
            }
            else
            {
                var baseName = new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFOS_WITH_BASE((::{0}), (::{1}))", fullName, baseName);
            }

            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    var inOutInfo = MakeInOutInfos(method);
                    rawGen.AutoNewLine();
                    rawGen.WriteLine(@"// {0}", GetMethodAccessorHeader(method, true));
                    rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO({0})", method.Name);
                    using (rawGen.Indent())
                    {
                        rawGen.AutoNewLine();
                        EmitMethodMetaInfo(rawGen, method);
                        rawGen.AutoNewLine();
                        EmitMethodInProcIdInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        EmitMethodBufferInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        EmitMethodObjectInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        EmitMethodNativeHandleInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        EmitMethodInRawInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        EmitMethodOutRawInfo(rawGen, inOutInfo);
                        rawGen.AutoNewLine();
                        var argumentNames = inOutInfo.All.Select(e => string.Format(@"NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO({0})", e.Name));
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS(({0}))", string.Join(@", ", argumentNames));
                        rawGen.AutoNewLine();
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END({0})", method.Name);
                }
                rawGen.AutoNewLine();

                EmitReferredStructCheckCode(rawGen, entity);
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFOS_END");
        }

        private static void EmitMethodMetaInfo(RawCppCodeGenerator rawGen, MethodInfo method)
        {
            if (UseCmifAttribute.GetUseCmif(method))
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID({0})", MethodIdAttribute.GetId(method).Value.ToString());
            }
            var returnType = method.ReturnType;
            if (returnType == typeof(void))
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)");
            }
            if (returnType == typeof(nn.Result))
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_RESULT(true)");
            }
        }

        private static void EmitMethodInProcIdInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.InProcessIdEnable)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_PROCESS_ID_ENABLE()");
            }
        }

        private static void EmitMethodBufferInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.Buffers.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT({0})", inOutInfos.Buffers.Count.ToString());
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS");
                using (rawGen.Indent())
                {
                    var index = 0;
                    foreach (var tan in inOutInfos.FixedSizeBuffers)
                    {
                        var e = (SfValueType)tan.Type.GetSfEntity();
                        var parameterString = e.GetParameterString(null, tan.InOutType, false);
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO({0}, {1}, ({2}), {3}, {4})",
                            tan.Name, index.ToString(), parameterString, e.Size.ToString(), tan.ParameterInfo.GetBufferAttributeString());
                        ++index;
                    }
                    foreach (var tan in inOutInfos.VariableSizeBuffers)
                    {
                        var e = (SfNnBuffer)tan.Type.GetSfEntity();
                        var parameterString = e.GetParameterString(null, tan.InOutType, false);
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO({0}, {1}, ({2}), {3})",
                            tan.Name, index.ToString(), parameterString, tan.ParameterInfo.GetBufferAttributeString());
                        ++index;
                    }
                }
                var all = inOutInfos.Buffers.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
            }
        }

        private static void EmitMethodObjectInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.InObjects.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_COUNT({0})", inOutInfos.InObjects.Count.ToString());
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS");
                using (rawGen.Indent())
                {
                    var index = 0;
                    foreach (var tan in inOutInfos.InObjects)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFO({0}, {1}, 0)", tan.Name, index.ToString());
                        ++index;
                    }
                }
                var all = inOutInfos.InObjects.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
            }
            if (inOutInfos.OutObjects.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_COUNT({0})", inOutInfos.OutObjects.Count.ToString());
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS");
                using (rawGen.Indent())
                {
                    var index = 0;
                    foreach (var tan in inOutInfos.OutObjects)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO({0}, {1}, 0)", tan.Name, index.ToString());
                        ++index;
                    }
                }
                var all = inOutInfos.OutObjects.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
            }
        }

        private static void EmitMethodNativeHandleInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.InNativeHandles.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_COUNT({0})", inOutInfos.InNativeHandles.Count.ToString());
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS");
                using (rawGen.Indent())
                {
                    var index = 0;
                    foreach (var tan in inOutInfos.InNativeHandles)
                    {
                        var attribute = tan.ParameterInfo != null && tan.ParameterInfo.GetCustomAttribute<MoveHandleAttribute>() != null
                            ? "NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE"
                            : "NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY";
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFO({0}, {1}, {2})", tan.Name, index.ToString(), attribute);
                        ++index;
                    }
                }
                var all = inOutInfos.InNativeHandles.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
            }
            if (inOutInfos.OutNativeHandles.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_COUNT({0})", inOutInfos.OutNativeHandles.Count.ToString());
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS");
                using (rawGen.Indent())
                {
                    var index = 0;
                    foreach (var tan in inOutInfos.OutNativeHandles)
                    {
                        var attribute = tan.ParameterInfo != null && tan.ParameterInfo.GetCustomAttribute<MoveHandleAttribute>() != null
                            ? "NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE"
                            : "NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY";
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFO({0}, {1}, {2})", tan.Name, index.ToString(), attribute);
                        ++index;
                    }
                }
                var all = inOutInfos.OutNativeHandles.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
            }
        }

        private static void EmitMethodInRawInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.InRaws.Count > 0)
            {
                var saaList = new List<SizeAndAlignment>();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT");
                using (rawGen.Indent())
                {
                    foreach (var tan in inOutInfos.InRaws)
                    {
                        var entity = (SfValueType)tan.Type.GetSfEntity();
                        var typeString = entity.GetCppTypeString(false);
                        var saa = new SizeAndAlignment { Size = entity.Size, Alignment = entity.Alignment };
                        saaList.Add(saa);
                    }
                    var sizeAndAlignmentAndoffsets = BinaryFormatUtility.GetSizeAndAlignmentAndOffsets(saaList);
                    foreach (var e in inOutInfos.InRaws.Zip(sizeAndAlignmentAndoffsets.Item2, (tan, offset) => new { tan, offset }))
                    {
                        var entity = (SfValueType)e.tan.Type.GetSfEntity();
                        var typeString = entity.GetCppTypeString(false);
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER({0}, ({1}), {2})", e.tan.Name, typeString, e.offset.ToString());
                    }
                }
                var all = inOutInfos.InRaws.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE({0})", BinaryFormatUtility.GetSizeAndAlignment(saaList).Size.ToString());
            }
        }

        private static void EmitMethodOutRawInfo(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.OutRaws.Count > 0)
            {
                var saaList = new List<SizeAndAlignment>();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT");
                using (rawGen.Indent())
                {
                    foreach (var tan in inOutInfos.OutRaws)
                    {
                        var entity = (SfValueType)tan.Type.GetSfEntity();
                        var typeString = entity.GetCppTypeString(false);
                        var saa = new SizeAndAlignment { Size = entity.Size, Alignment = entity.Alignment };
                        saaList.Add(saa);
                    }
                    var sizeAndAlignmentAndoffsets = BinaryFormatUtility.GetSizeAndAlignmentAndOffsets(saaList);
                    foreach (var e in inOutInfos.OutRaws.Zip(sizeAndAlignmentAndoffsets.Item2, (tan, offset) => new { tan, offset }))
                    {
                        var entity = (SfValueType)e.tan.Type.GetSfEntity();
                        var typeString = entity.GetCppTypeString(false);
                        rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER({0}, ({1}), {2})", e.tan.Name, typeString, e.offset.ToString());
                    }
                }
                var all = inOutInfos.OutRaws.Select(x => x.Name).ToArray();
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END({0}, ({1}))", all.Length.ToString(), string.Join(", ", all));
                rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE({0})", BinaryFormatUtility.GetSizeAndAlignment(saaList).Size.ToString());
            }
        }

        private static void EmitReferredStructCheckCode(RawCppCodeGenerator rawGen, SfEntity entity)
        {
            var structs =
                from e in new HashSet<SfEntity>(entity.GetStronglyReferredEntities())
                let s = e as SfStruct
                where s != null
                select s;
            foreach (var e in structs)
            {
                if (e.IsExternal() || e.GetFields().Count() > 1)
                {
                    // 複数メンバをもつ構造体だけ検査コードを出す
                    rawGen.WriteLine(@"NN_SF_CMIF_DETAIL_METHOD_INFO_REFERRED_STRUCT_ASSERTION_CHECK(({0}), {1}, {2})",
                        e.GetCppTypeString(false), e.Size.ToString(), e.Alignment.ToString());
                }
            }
        }

        #endregion

        #region EmitCmifProxy

        private void EmitCmifProxy(RawCppCodeGenerator rawGen, string fullName, MethodInfo[] methods, SfEntity baseInterface)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"// template <...> class nn::sf::client::detail::CmifProxy<{0}, ...>", fullName);

            if (baseInterface.InnerType == typeof(nn.sf.IServiceObject))
            {
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_DEFINE_PROXY((::{0}))", fullName);
            }
            else
            {
                var baseName = new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_DEFINE_PROXY_WITH_BASE((::{0}), (::{1}))", fullName, baseName);
            }

            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.AutoNewLine();
                    EmitProxyMethodDefinition(rawGen, method);
                }
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_DEFINE_PROXY_END");
        }

        private static void EmitProxyMethodDefinition(RawCppCodeGenerator rawGen, MethodInfo method)
        {
            var inOutInfos = MakeInOutInfos(method);
            rawGen.WriteLine(@"// {0}", GetMethodAccessorHeader(method, true));
            rawGen.WriteLine(@"{0}", method.GetMacroFunctionString(@"NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD", true, false));
            rawGen.WriteLine(@"{");
            using (rawGen.Indent())
            {
                if (UseCmifAttribute.GetUseCmif(method))
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN({0})", method.Name);
                    foreach (var inOutInfo in inOutInfos.All)
                    {
                        using (rawGen.Indent())
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT({0})", inOutInfo.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_CALL_END({0})", method.Name);
                }
                else
                {
                    foreach (var p in method.GetParameters())
                    {
                        rawGen.WriteLine(@"NN_UNUSED({0});", p.Name);
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_DEFINE_SYNC_METHOD_UNSUPPORTED");
                }
            }
            rawGen.WriteLine(@"}");
        }

        private static void EmitProxyMethodDefinitionBeforeInvoke(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE");
            using (rawGen.Indent())
            {
                if (inOutInfos.Buffers.Count > 0)
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_BUFFERS");
                    using (rawGen.Indent())
                    {
                        foreach (var saa in inOutInfos.Buffers)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_BUFFER({0})", saa.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_BUFFERS_END");
                }
                if (inOutInfos.InObjects.Count > 0)
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_OBJECTS");
                    using (rawGen.Indent())
                    {
                        foreach (var saa in inOutInfos.InObjects)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_OBJECT({0})", saa.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_OBJECTS_END");
                }
                if (inOutInfos.InNativeHandles.Count > 0)
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_NATIVE_HANDLES");
                    using (rawGen.Indent())
                    {
                        foreach (var saa in inOutInfos.InNativeHandles)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_NATIVE_HANDLE({0})", saa.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_IN_NATIVE_HANDLES_END");
                }
                if (inOutInfos.OutNativeHandles.Count > 0)
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_OUT_NATIVE_HANDLES");
                    using (rawGen.Indent())
                    {
                        foreach (var saa in inOutInfos.OutNativeHandles)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_OUT_NATIVE_HANDLE({0})", saa.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_OUT_NATIVE_HANDLES_END");
                }
            }
            rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_PREPARE_MESSAGE_END");
            rawGen.AutoNewLine();
            if (inOutInfos.InRaws.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_SET_IN_RAW_MEMBERS");
                using (rawGen.Indent())
                {
                    foreach (var saa in inOutInfos.InRaws)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_SET_IN_RAW_MEMBER({0})", saa.Name);
                    }
                }
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_SET_IN_RAW_MEMBERS_END");
            }
        }

        private static void EmitProxyMethodDefinitionAfterInvoke(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.OutRaws.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_RAW_MEMBERS");
                using (rawGen.Indent())
                {
                    foreach (var saa in inOutInfos.OutRaws)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_RAW_MEMBER({0})", saa.Name);
                    }
                }
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_RAW_MEMBERS_END");
            }
            if (inOutInfos.OutObjects.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_OBJECTS");
                using (rawGen.Indent())
                {
                    foreach (var saa in inOutInfos.OutObjects)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_OBJECT({0})", saa.Name);
                    }
                }
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_OBJECTS_END");
            }
            if (inOutInfos.OutNativeHandles.Count > 0)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_NATIVE_HANDLES");
                using (rawGen.Indent())
                {
                    foreach (var saa in inOutInfos.OutNativeHandles)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_NATIVE_HANDLE({0})", saa.Name);
                    }
                }
                rawGen.WriteLine(@"NN_SF_CMIF_CLIENT_DETAIL_PROXY_GET_OUT_NATIVE_HANDLES_END");
            }
        }

        #endregion

        #region EmitCmifDispatcher

        private void EmitCmifDispatcher(RawCppCodeGenerator rawGen, string fullName, MethodInfo[] methods, SfEntity baseInterface)
        {
            rawGen.CloseCurrentNamespace();
            rawGen.WriteLine(@"// template <> class nn::sf::server::detail::CmifProcessFunctionTableGetterImpl<{0}>", fullName);

            bool hasBase = (baseInterface.InnerType != typeof(nn.sf.IServiceObject));
            if (hasBase)
            {
                var baseName = new CppName(CppFullNameAttribute.GetCppFullName(baseInterface)).ToString();
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_WITH_BASE((::{0}), (::{1}))", fullName, baseName);
            }
            else
            {
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER((::{0}))", fullName);
            }
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.AutoNewLine();
                    EmitDispatchMethodDefinition(rawGen, method);
                }
                rawGen.AutoNewLine();
                EmitDispatchMessageDefinition(rawGen, methods, hasBase);
                rawGen.AutoNewLine();
            }
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_END");
            rawGen.AutoNewLine();
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_FUNCTION_TABLE((::{0}))", fullName);
        }

        private static void EmitDispatchMethodDefinition(RawCppCodeGenerator rawGen, MethodInfo method)
        {
            var inOutInfos = MakeInOutInfos(method);
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD({0})", method.Name);
            using (rawGen.Indent())
            {
                EmitPrepareForProcess(rawGen, inOutInfos);
                EmitInOutObjectsDefinition(rawGen, inOutInfos);
                EmitInProcessIdOverwriting(rawGen, inOutInfos);
                EmitCmifServerPreConditionCheck(rawGen, inOutInfos);
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE({0})", method.Name);
                using (rawGen.Indent())
                {
                    WriteCommaSeparated(rawGen, MakeDispatchParameterStringList(inOutInfos).ToList());
                }
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END");
                EmitSetOutObjectsDefinition(rawGen, inOutInfos);
            }
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END");
        }

        private static void EmitPrepareForProcess(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE");
            if (inOutInfos.Buffers.Count > 0)
            {
                using (rawGen.Indent())
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS");
                    using (rawGen.Indent())
                    {
                        foreach (var tan in inOutInfos.Buffers)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER({0})", tan.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END");
                }
            }
            if (inOutInfos.InNativeHandles.Count > 0)
            {
                using (rawGen.Indent())
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES");
                    using (rawGen.Indent())
                    {
                        foreach (var tan in inOutInfos.InNativeHandles)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLE({0})", tan.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES_END");
                }
            }
            if (inOutInfos.OutNativeHandles.Count > 0)
            {
                using (rawGen.Indent())
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES");
                    using (rawGen.Indent())
                    {
                        foreach (var tan in inOutInfos.OutNativeHandles)
                        {
                            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE({0})", tan.Name);
                        }
                    }
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES_END");
                }
            }
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END");
        }

        private static void EmitInOutObjectsDefinition(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            for (var i = 0; i < inOutInfos.InObjects.Count; ++i)
            {
                var tan = inOutInfos.InObjects[i];
                var e = tan.Type.GetSfEntity();
                var typeName = e.GetCppTypeString(false);
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_IN_OBJECT(({0}), {1}, {2})", typeName, tan.Name, i.ToString());
            }
            for (var i = 0; i < inOutInfos.OutObjects.Count; ++i)
            {
                var tan = inOutInfos.OutObjects[i];
                var e = tan.Type.GetSfEntity();
                var typeName = e.GetCppTypeString(false);
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT(({0}), {1})", typeName, tan.Name);
            }
        }

        private static void EmitSetOutObjectsDefinition(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            for (var i = 0; i < inOutInfos.OutObjects.Count; ++i)
            {
                var tan = inOutInfos.OutObjects[i];
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT({0}, {1})", tan.Name, i.ToString());
            }
        }

        private static void EmitInProcessIdOverwriting(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            if (inOutInfos.InProcessIdEnable)
            {
                foreach (var tan in inOutInfos.InRaws)
                {
                    if (tan.ParameterInfo.GetCustomAttribute<Hipc.ProcessIdAttribute>() != null)
                    {
                        rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OVERWRITE_IN_PROCESS_ID({0})", tan.Name);
                    }
                }
            }
        }

        private static void EmitCmifServerPreConditionCheck(RawCppCodeGenerator rawGen, InOutInfos inOutInfos)
        {
            foreach (var e in inOutInfos.All)
            {
                if (!(e.InOutType == InOutType.In))
                {
                    continue;
                }
                foreach (var pre in PreBaseAttribute.GetPreAttributes(e.ParameterInfo, true))
                {
                    var checkString = pre.MakeCppCheckString(GetDispatchMemberString(e.Name, inOutInfos));
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE({0})", checkString);
                }
            }
        }

        private static void WriteCommaSeparated(RawCppCodeGenerator rawGen, List<string> list)
        {
            for (var i = 0; i < list.Count - 1; ++i)
            {
                rawGen.WriteLine(@"{0},", list[i]);
            }
            if (list.Count > 0)
            {
                rawGen.WriteLine(@"{0}", list[list.Count - 1]);
            }
        }

        private static IEnumerable<string> MakeDispatchParameterStringList(InOutInfos inOutInfos)
        {
            return from tan in inOutInfos.All
                   select GetDispatchMemberString(tan.Name, inOutInfos);
        }

        private static string GetDispatchMemberString(string name, InOutInfos inOutInfos)
        {
            {
                var inRaw = inOutInfos.InRaws.Find(tan => tan.Name == name);
                if (inRaw != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER({0})", name);
                }
            }
            {
                var outRaw = inOutInfos.OutRaws.Find(tan => tan.Name == name);
                if (outRaw != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER({0})", name);
                }
            }
            {
                var buffer = inOutInfos.Buffers.Find(tan => tan.Name == name);
                if (buffer != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER({0})", name);
                }
            }
            {
                var inObject = inOutInfos.InObjects.Find(tan => tan.Name == name);
                if (inObject != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_OBJECT({0})", name);
                }
            }
            {
                var outObject = inOutInfos.OutObjects.Find(tan => tan.Name == name);
                if (outObject != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT({0})", name);
                }
            }
            {
                var inNativeHandle = inOutInfos.InNativeHandles.Find(tan => tan.Name == name);
                if (inNativeHandle != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_NATIVE_HANDLE({0})", name);
                }
            }
            {
                var outNativeHandle = inOutInfos.OutNativeHandles.Find(tan => tan.Name == name);
                if (outNativeHandle != null)
                {
                    return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE({0})", name);
                }
            }
            throw new WrongImplementationException();
        }

        private static void EmitDispatchMessageDefinition(RawCppCodeGenerator rawGen, MethodInfo[] methods, bool hasBase)
        {
            rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE");
            using (rawGen.Indent())
            {
                foreach (var method in methods)
                {
                    rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE({0})", method.Name);
                }
            }
            if (hasBase)
            {
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END_WITH_BASE");
            }
            else
            {
                rawGen.WriteLine(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END");
            }
        }

        #endregion

        #endregion
    }
}
