﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    /// <summary>
    /// 実装用ユーティリティ
    /// </summary>
    /// <remarks>
    /// SfEntity を対応する EntityCppCodeEmitter へ変換し、そのメソッドを呼ぶような拡張メソッドを定義することで、
    /// あたかも SfEntity のインスタンスから EntityCppCodeEmitter のメソッドを呼ぶことができるかのように拡張するためのユーティリティ。
    /// このユーティリティは、Nintendo.ServiceFramework.CppCode 名前空間内か、本名前空間を using しないと拡張メソッドとしては使用できない。
    /// </remarks>
    public static class EntityCppCodeEmitterHelper
    {
        public static string GetForwardString(this SfEntity entity, string variableName, InOutType inOutType, bool friendly)
        {
            return GetEmitter(entity).GetForwardString(variableName, inOutType, friendly);
        }

        public static string GetParameterString(this SfEntity entity, string variableName, InOutType inOutType, bool friendly)
        {
            return GetEmitter(entity).GetParameterString(variableName, inOutType, friendly);
        }

        public static string GetCppTypeString(this SfEntity entity, bool friendly)
        {
            return GetEmitter(entity).GetCppTypeString(friendly);
        }

        public static string GetStructPlacedString(this SfEntity entity, string variableName, bool friendly)
        {
            return GetEmitter(entity).GetStructPlacedString(variableName, friendly);
        }

        public static IEnumerable<string> GetExternalIncludes(this SfEntity entity)
        {
            return GetEmitter(entity).GetExternalIncludes();
        }

        public static IEnumerable<string> GetAdditionalIncludes(this SfEntity entity)
        {
            return GetEmitter(entity).GetAdditionalIncludes();
        }

        public static void EmitForwardDeclarationCode(this SfEntity entity, RawCppCodeGenerator rawGen)
        {
            GetEmitter(entity).EmitForwardDeclarationCode(rawGen);
        }

        public static void EmitDefinitionCode(this SfEntity entity, RawCppCodeGenerator rawGen)
        {
            GetEmitter(entity).EmitDefinitionCode(rawGen);
        }

        public static void EmitReferenceCheckCode(this SfEntity entity, RawCppCodeGenerator rawGen)
        {
            GetEmitter(entity).EmitReferenceCheckCode(rawGen);
        }

        private static Type[] s_KnownCppCodeEmitterTypes = new[]
        {
            typeof(BuiltInCppCodeEmitter),
            typeof(InterfaceCppCodeEmitter),
            typeof(NnResultCppCodeEmitter),
            typeof(NnBufferCppCodeEmitter),
            typeof(StructCppCodeEmitter),
            typeof(NativeHandleCppCodeEmitter),
        };

        private static Dictionary<Type, Func<SfEntity, EntityCppCodeEmitter>> MakeKnownCppCodeEmitterCreator()
        {
            var ret = new Dictionary<Type, Func<SfEntity, EntityCppCodeEmitter>>();
            foreach (var emitterType in s_KnownCppCodeEmitterTypes)
            {
                // コンストラクタを取得して
                var ctor = emitterType.GetConstructors(BindingFlags.NonPublic | BindingFlags.Public | BindingFlags.Instance)[0];
                // その第一引数の型の SfEntity を取れるものとする
                var entityType = ctor.GetParameters()[0].ParameterType;
                // エンティティ型に対し、エミッタの型のコンストラクタを起動する
                ret.Add(entityType, e => (EntityCppCodeEmitter)ctor.Invoke(new object[] { e }));
            }
            return ret;
        }

        private static readonly Dictionary<Type, Func<SfEntity, EntityCppCodeEmitter>> s_KnownCppCodeEmitterCreator = MakeKnownCppCodeEmitterCreator();

        /// <summary>
        /// SfEntity から EntityCppCodeEmitter への変換
        /// </summary>
        /// <param name="entity"></param>
        /// <returns></returns>
        private static EntityCppCodeEmitter GetEmitter(this SfEntity entity)
        {
            try
            {
                return s_KnownCppCodeEmitterCreator[entity.GetType()](entity);
            }
            catch (KeyNotFoundException)
            {
                throw new WrongImplementationException();
            }
        }
    }
}
