﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>

void GetInput(int64_t* size)
{
    scanf_s("%lld", size);
    if (*size < 0)
    {
        NN_LOG("Negative size was input.\n");
        exit(1);
    }
}

extern "C" void nnMain()
{
    int64_t availableSize = 0;
    int64_t journalSize = 0;
    int64_t totalSize = 0;

    const size_t BlockSize = 16 * 1024;
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam paramDuplex;
    {
        paramDuplex.sizeBlockLevel[0] = BlockSize;
        paramDuplex.sizeBlockLevel[1] = BlockSize;
    }

    nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
    {
        paramIntegrity.sizeBlockLevel[0] = BlockSize;
        paramIntegrity.sizeBlockLevel[1] = BlockSize;
        paramIntegrity.sizeBlockLevel[2] = BlockSize;
        paramIntegrity.sizeBlockLevel[3] = BlockSize;
    }

    const int CountExpandMax = 1; // 1 指定で Expand 最大回数を自動設定

    NN_LOG("JournalSize (bytes) ? -> ");
    GetInput(&journalSize);

    NN_LOG("AvailableSize (bytes) ? # Set 0 to calculate by total size -> ");
    GetInput(&availableSize);

    if (availableSize == 0)
    {
        NN_LOG("TotalSize (bytes) ? -> ");
        GetInput(&totalSize);
    }

    uint32_t countDataBlock = static_cast<uint32_t>((availableSize + (BlockSize - 1)) / BlockSize);
    uint32_t countJournalBlock = static_cast<uint32_t>((journalSize + (BlockSize - 1)) / BlockSize);

    nn::Result result;

    if (availableSize == 0)
    {
        result = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryDataBlockCount(&countDataBlock, totalSize, journalSize, BlockSize, CountExpandMax, paramDuplex, paramIntegrity);
    }
    else
    {
        result = nn::fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(&totalSize, BlockSize, CountExpandMax, paramDuplex, paramIntegrity, countDataBlock, countJournalBlock);
    }

    if (result.IsFailure())
    {
        NN_LOG("Cannot create save data with specified parameter.\n");
        exit(1);
    }

    NN_LOG("----\n");
    NN_LOG("TotalSize:     %d\n", totalSize);
    NN_LOG("AvailableSize: %d\n", countDataBlock * BlockSize);
    NN_LOG("JournalSize:   %d\n", countJournalBlock * BlockSize);
    NN_LOG("----\n");
}
