﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Diagnostics;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;
using MakeInitialImage;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Xml.Linq;
using System.Text.RegularExpressions;

namespace RunSystemInitializer
{
    public class RunSystemInitializerArgument
    {
        [CommandLineOption('v', "verbose",
            Description = "Output trace logs.")]
        public bool Verbose { get; set; }

        [CommandLineValues(ValueName = "Arguments", Description = "Arguments for system initializer.")]
        public string[] Arguments { get; set; }
    }

    public class Program
    {
        public static void Main(string[] args)
        {
            try
            {
                RunSystemInitializerArgument parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<RunSystemInitializerArgument>(args, out parsed))
                {
                    System.Environment.Exit(1);
                }
                else
                {
                    DataflowConsole.Instance.SetOption(true);
                    ConsoleApplicationTraceListener.SetGlobal(parsed.Verbose);

                    WaitHtcService("@InitializeSystem");

                    SendArguments((new string[] { "SystemInitializer" }).Concat(parsed.Arguments).ToArray());
                }
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
        }

        private static void WaitHtcService(string serviceName)
        {
            RetryUtility.Do(() =>
            {
                using (var tma = new TargetManagerAccessor())
                {
                    if (tma.ExistService(serviceName))
                    {
                        return;
                    }

                    throw new Exception("Found no services: name = " + serviceName);
                }
            },
            (e) => {
                DataflowConsole.Instance.GetNamedTarget("WAIT").Post(string.Format("Find service: {0}\n", serviceName));
            },
            10,
            TimeSpan.FromSeconds(1));
        }

        private static void SendArguments(string[] arguments)
        {
            using (var tma = new TargetManagerAccessor())
            {
                var endPoint = tma.FindService("@InitializeSystem");

                using (var socket = NetworkUtility.CreateConnectedSocket(endPoint))
                {
                    using (var memoryStream = new MemoryStream())
                    using (var memoryWriter = new StreamWriter(memoryStream))
                    {
                        foreach (var argument in arguments)
                        {
                            memoryWriter.Write(argument);
                            memoryWriter.Write('\0');
                        }
                        memoryWriter.Flush();

                        using (var socketStream = new NetworkStream(socket))
                        {
                            var buffer = memoryStream.GetBuffer();
                            BinaryUtility.WriteBinary<int>(socketStream, buffer.Count());
                            BinaryUtility.WriteBinary<int>(socketStream, arguments.Count());
                            socketStream.Write(buffer, 0, buffer.Count());

                            var readTask = socketStream.ReadAsync(new byte[1], 0, 1);
                            readTask.Wait(TimeSpan.FromSeconds(1));
                            for (int i = 0; i < 60 && !readTask.IsCompleted; i++)
                            {
                                readTask.Wait(TimeSpan.FromSeconds(1));
                            }

                            if (!readTask.IsCompleted)
                            {
                                DataflowConsole.Instance.GetNamedTarget("ERROR").Post(string.Format("timeout.\n"));
                            }
                        }
                    }
                }
            }
        }

        public static void PrintException(Exception exception)
        {
            DataflowConsole.Instance.GetNamedTarget("ERROR").Post(string.Format("{0}\n", exception.Message));
            DataflowConsole.Instance.GetNamedTarget("ERROR").Post(string.Format("StackTrace:\n{0}\n", exception.StackTrace));
        }
    }
}
