﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading.Tasks.Dataflow;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;

namespace Nintendo.RunOnTarget
{
    public class RunCommandArgumentPrivate
    {
        public CommonRunCommandArgument Common { get; set; }
        public string TargetName { get; set; }
        public string HostBridgeName { get; set; }
        public bool MonitorSerial { get; set; }
        public bool SuppressAutoKill { get; set; }
        public bool MonitorShell { get; set; }
        public int WaitSecondsAfterReset { get; set; }
        public string ExtraTargetManagerDirectory { get; set; }

        public RunCommandArgumentPrivate(RunCommandArgumentRawPrivate rawArgs)
        {
            Common = rawArgs.MakeCommonRunCommandArgument();
            TargetName = rawArgs.Target;
            HostBridgeName = rawArgs.HostBridge == null ? rawArgs.Target : rawArgs.HostBridge;
            MonitorSerial = rawArgs.MonitorSerial;
            SuppressAutoKill = rawArgs.SuppressAutoKill;
            MonitorShell = rawArgs.MonitorShell;
            WaitSecondsAfterReset = rawArgs.WaitSecondsAfterReset;
            ExtraTargetManagerDirectory = rawArgs.ExtraTargetManagerDirectory;
        }
    }

    public class RunCommandArgumentRawPrivate : RunCommandArgumentRaw
    {
        [CommandLineOption("suppress-polling-process",
            Description = "Suppress polling process for waiting process.")]
        public bool SuppressPollingProcess { get; set; }

        [CommandLineOption("monitor-serial",
            Description = "Monitor logs from serial.")]
        public bool MonitorSerial { get; set; }

        [CommandLineOption("hostbridge",
            Description = "Set the address or the name of the HostBridge.")]
        public string HostBridge { get; set; }

        [CommandLineOption("monitor-shell",
            Description = "Monitor logs from shell process.")]
        public bool MonitorShell { get; set; }

        [CommandLineOption("wait-after-reset",
            Description = "Wait seconds after reset.")]
        public int WaitSecondsAfterReset { get; set; }

        public RunCommandArgumentRawPrivate()
        {
            IsNcaSupported = true;
        }

        public override ExitCondition MakeExitCondition()
        {
            return new ExitCondition()
            {
                NoWait = this.NoWait,
                SuccessExitPattern = this.SuccessExitPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.SuccessExitPattern)),
                FailureExitPattern = this.FailureExitPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.FailureExitPattern)),
                SuccessTimeOut = this.SuccessTimeOut == -1 ? TimeSpan.MaxValue : TimeSpan.FromSeconds(this.SuccessTimeOut),
                FailureTimeOut = this.FailureTimeOut == -1 ? TimeSpan.MaxValue : TimeSpan.FromSeconds(this.FailureTimeOut),
                SuccessTimeOutResetPattern = this.ResetSuccessTimeOutPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.ResetSuccessTimeOutPattern)),
                FailureTimeOutResetPattern = this.ResetFailureTimeOutPattern == null ? ExitCondition.NeverMatchedRegex : new Regex(Utility.MakeJoinedPattern(this.ResetFailureTimeOutPattern)),
                NotFoundFailurePatterns = this.NotFoundFailurePatterns == null ? new List<Regex>() : (from pattern in this.NotFoundFailurePatterns select new Regex(pattern)).ToList(),
                WaitExitProcess = this.SuppressPollingProcess ? false : true
            };
        }
    }

    public class NnRunCommandArgumentRaw
    {
        [CommandLineSubCommand("run", Description = "run program.")]
        public RunCommandArgumentRawPrivate Run { get; set; }
    }
}
