﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;
using System.Threading;
using System.Globalization;
using CommandUtility;

namespace Nintendo.RunOnTarget
{
    public class RunOnTargetHelpMaker : CommandLineHelpMaker
    {
        private string Example { get; set; }

        public RunOnTargetHelpMaker(string example)
        {
            Example = example;
        }

        protected override string CreateUsageText(string appName, IEnumerable<ICommandLineOptionInfo> options, IEnumerable<ICommandLineValueInfo> values)
        {
            var baseHelp = base.CreateUsageText(appName, options, values);

            var result = baseHelp.Replace("[<Arguments>...]", "[--] [<Arguments>...]");

            result += @"
Examples:" + Example;

            return result;
        }
    }

    public class CommandInterface
    {
        public static int Main(string[] rawArgs)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);
#if !DEBUG
            try
            {
#else
            {
#endif
                RunCommandArgumentRaw parsed;
                CommandLineParserSettings settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = "Run a application on a target.",
                    HelpMaker = new RunOnTargetHelpMaker(@"
  # run the application on the default target.
  RunOnTarget.exe app.nsp

  # run with application arguments (put after '--').
  RunOnTarget.exe app.nsp -- --input inputfile --output outputfile

  # run on the specified target by the serial number or the ip address.
  RunOnTarget.exe app.nsp --target XAWF0111100014
  RunOnTarget.exe app.nsp --target 192.168.0.1

  # run after resetting the target.
  RunOnTarget.exe app.nsp --reset

  # run without waiting the application.
  RunOnTarget.exe app.nsp --no-wait

  # run with timeout (3600 seconds).
  RunOnTarget.exe app.nsp --failure-timeout 3600

  # run until find the specified text (as regular expression).
  RunOnTarget.exe app.nsp
    --pattern-failure-exit ""Assert|Break|Panic|Halt|Fatal""
    --pattern-success-exit ""Success""
")
                };

                if (false == SdkCommandLineParser.ParseArgs<RunCommandArgumentRaw>(settings, rawArgs, out parsed))
                {
                    return 1;
                }

                DataflowConsole.Instance.SetOption(parsed.Verbose);
                ConsoleApplicationTraceListener.SetGlobal(parsed.Verbose);

                var args = new RunCommandArgument(parsed);
                var ret = Run(args);
                DataflowConsole.Instance.Ignore();
                return ret;
#if !DEBUG
            }
            catch (Exception exception)
            {
                PrintException(exception);
                return 1;
            }
#else
            }
#endif
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Trace.WriteLine($"StackTrace: {exception.StackTrace}");
        }

        public static int Run(RunCommandArgument args)
        {
            return new Runner().Run(args) == ExitStatus.Success ? 0 : 1;
        }
    }
}
