﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Text.RegularExpressions;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

#if SIGNED_IMAGE
#error The preprocessor symbol "SIGNED_IMAGE" is abolished. Please use "SETUP_TOOLS_USE_UNSIGNED_IMAGE" instead.
#endif

namespace Nintendo.InitializeSdev
{
    public enum FirmwareType
    {
        Firmware_SystemUpdaterProcess,
        Firmware_SafeModeUpdaterProcess,
        Firmware_SystemImage,
        Firmware_SafeModeImage,
        Firmware_QspiBootImage,
        Firmware_HostBridge,
        Firmware_HostBridgeRecovery,
        Firmware_DevKitUpdater,
        Host_TargetManager,
        Host_TargetManager_Old,
        Host_RecoveryWriterUsb,
        Host_RecoveryWriterUsb_Old,
        Host_RunOnTarget,
        Firmware_DevMenuCommand,
        Firmware_HelloWorldImage,
        Firmware_other
    };

    public enum FirmwareVersionType
    {
        Version_DefaultFirm,
        Version_OceanFirm,
        Version_ProdFirm,
        Version_ProdWithLogFirm,
        Version_Error
    };

    public class FirmwareResourceSpecifier
    {
        private string VersionString = null;
        private TargetSpecifier.SdevVersion SdevVersion = TargetSpecifier.SdevVersion.SDEV_NONE;

        FirmwareResourceAccessor resourceAccessor;

        public FirmwareResourceSpecifier()
        {
            resourceAccessor = FirmwareResourceAccessor.GetInstance();
        }

        public bool SetVersion(string version)
        {
            if (version.ToLower() == "default" || version == string.Empty)
            {
                VersionString = resourceAccessor.GetUpToDateVersion();
                return true;
            }

            // MEMO; バージョン文字列のフォーマットチェックは廃止しました (2017.04.07)
#if false
            if (!resourceAccessor.IsVersionFormat(version))
            {
                LOG.LogLine(LOG.Level.LOG_ERROR, "Invalid version string. ({0})", version);
                return false;
            }
#endif
            VersionString = version;
            return true;
        }

        public void SetSdevVersion(TargetSpecifier.SdevVersion version)
        {
            SdevVersion = version;
        }

        public string GetFileNameOfFirm(FirmwareType firmType)
        {
            return resourceAccessor.GetFileNameOfFirm(firmType, VersionString, SdevVersion);
        }

        public static string GetUpToDateVersion()
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.GetUpToDateVersion();
        }

        public static string GetUpToDateRevision()
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.GetUpToDateRevision();
        }

        public static bool EnumerateAllVersions(ref List<string> versionList)
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.EnumerateAllVersions(ref versionList);
        }

        public static bool EnumerateAllRevisions(ref List<string> revisionList)
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.EnumerateAllRevisions(ref revisionList);
        }

        public static bool ContainsOceanFirmware()
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.ContainsOceanFirmware();
        }

        public static bool ContainsProdFirmware()
        {
            FirmwareResourceAccessor resource = FirmwareResourceAccessor.GetInstance();
            return resource.ContainsProdFirmware();
        }

        public bool ValidateSetVersion()
        {
            return resourceAccessor.CheckUtilizableVersion(VersionString, SdevVersion);
        }

        public static bool CheckFilePath(string pathValueName, string pathValueValue)
        {
            // MEMO: ファイルの存在チェックは Get 時に行っている

            if(pathValueValue == null)
            {
                return false;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "{0} = {1}", pathValueName, pathValueValue);
            return true;
        }
    }

    public class FirmwareResourceAccessor
    {
        // このクラスはシングルトンです
        private static FirmwareResourceAccessor _singleInatance = null;

        private FirmwareVersion resourceFirmwareVersion;
        bool isExistOceanFirmware = false;
        bool isExistProdFirmware = false;
        // bool isExistProdWithLogFirmware = false;

        public struct FirmwareVersion
        {
            public int majorVersion;
            public int minorVersion;
            public int microVersion;
            public int majorRelStep;
            public int minorRelStep;
            public string otherString;
            public string revisionString;

            public FirmwareVersion(int major = 0, int minor = 0, int micro = 0, int majorrelstep = 0, int minorrelstep = 0, string other = "", string revision = "")
            {
                majorVersion = major;
                minorVersion = minor;
                microVersion = micro;
                majorRelStep = majorrelstep;
                minorRelStep = minorrelstep;
                otherString = other;

                revisionString = revision;
            }
        }

        private Dictionary<FirmwareType, string> firmwareExtDictionary = new Dictionary<FirmwareType, string>()
        {
#if false
            {FirmwareType.Firmware_SystemUpdaterProcess ,".kip"},
#else
            {FirmwareType.Firmware_SystemUpdaterProcess ,".nsp"},
#endif
            {FirmwareType.Firmware_SafeModeUpdaterProcess ,".nsp"},
            {FirmwareType.Firmware_SystemImage, ".initimg"},
            {FirmwareType.Firmware_SafeModeImage, ".initimg"},
            {FirmwareType.Firmware_QspiBootImage, ".qspi.img"},
            {FirmwareType.Firmware_HostBridge, ".nhf"},
            {FirmwareType.Firmware_HostBridgeRecovery, ".nhf"},
            {FirmwareType.Firmware_DevKitUpdater, ".nsp"},
            {FirmwareType.Host_TargetManager, "exe"},
            {FirmwareType.Host_TargetManager_Old, "exe"},
            {FirmwareType.Host_RunOnTarget, "exe" },
            {FirmwareType.Firmware_DevMenuCommand, "nsp"},
            {FirmwareType.Firmware_HelloWorldImage, ".nsp"},
            {FirmwareType.Firmware_other, ""}
        };

        private Dictionary<FirmwareType, string> firmwareDescriptionDictionary = new Dictionary<FirmwareType, string>()
        {
            {FirmwareType.Firmware_SystemUpdaterProcess ,"SystemUpdater process"},
            {FirmwareType.Firmware_SafeModeUpdaterProcess ,"SafeMode Updater process"},
            {FirmwareType.Firmware_SystemImage, "SystemUpdate image"},
            {FirmwareType.Firmware_SafeModeImage, "SafeMode Update image"},
            {FirmwareType.Firmware_QspiBootImage, "QSPI boot image"},
            {FirmwareType.Firmware_HostBridge, "HostBridge firmware"},
            {FirmwareType.Firmware_HostBridgeRecovery, "HostBridge recovery firmware"},
            {FirmwareType.Firmware_DevKitUpdater, "Updater process"},
            {FirmwareType.Host_TargetManager, "TargetManager execute"},
            {FirmwareType.Host_TargetManager_Old, "TargetManager execute (old)"},
            {FirmwareType.Host_RunOnTarget, "RunOnTarget command" },
            {FirmwareType.Firmware_DevMenuCommand, "DevMenuCommandSystem command"},
            {FirmwareType.Firmware_HelloWorldImage, "HelloWorld application"},
            {FirmwareType.Firmware_other, ""}
        };

        public static FirmwareResourceAccessor GetInstance()
        {
            if (_singleInatance == null)
            {
                _singleInatance = new FirmwareResourceAccessor();
            }
            return _singleInatance;
        }

        private FirmwareResourceAccessor()
        {
            ResourceDirectory = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + '\\' + ResourceDirectoryBase;
            if (!Directory.Exists(ResourceDirectory))
            {
                LOG.Log(LOG_LEVEL.LOG_WARN, "Resource directory is not found.");
            }
            if (!RetriveResourceFirmwareVersion())
            {
                LOG.Log(LOG_LEVEL.LOG_WARN, "Retrive firmware version failed.");
            }
#if !USE_LOCAL_IMAGES
            isExistOceanFirmware = CheckOceanFirmware();
            isExistProdFirmware = CheckProdFirmware();
            // isExistProdWithLogFirmware = isExistProdFirmware;
#else
            isExistOceanFirmware = false;
            isExistProdFirmware = true;
            // isExistProdWithLogFirmware = true;
#endif
        }

        public bool RetriveResourceFirmwareVersion()
        {
            string firmwareDirectory = ResourceDirectory;
            resourceFirmwareVersion = new FirmwareVersion();
            return GetResourceFirmwareVersion(ref resourceFirmwareVersion, firmwareDirectory);
        }

        private bool CheckOceanFirmware()
        {
            // 今はもう使用しない
            // string OceanSystemUpdaterFilePath = Path.Combine(ResourceDirectory, OceanDirectory, DefaultOceanSystemUpdaterName);
            // return File.Exists(OceanSystemUpdaterFilePath);
            return false;
        }

        public bool ContainsOceanFirmware()
        {
            return isExistOceanFirmware;
        }

        private bool CheckProdFirmware()
        {
            string ProdSystemUpdaterFilePath = Path.Combine(ResourceDirectory, DefaultProdSystemUpdaterName);
            return File.Exists(ProdSystemUpdaterFilePath);
        }

        public bool ContainsProdFirmware()
        {
            return isExistProdFirmware;
        }

        static bool IsProdOrProdWithLogVersion(FirmwareVersionType firmVersionType)
        {
            return firmVersionType == FirmwareVersionType.Version_ProdFirm || firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm;
        }

        static bool IsEdev(TargetSpecifier.SdevVersion sdevVersion)
        {
            switch (sdevVersion)
            {
                case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                case TargetSpecifier.SdevVersion.EDEV_MP:
                    return true;
            }
            return false;
        }

        public bool EnumerateAllVersions(ref List<string> versionList)
        {
            versionList.Clear();
            versionList.Add(GetVersionString(resourceFirmwareVersion, false));
            if (isExistOceanFirmware)
            {
                versionList.Add(GetVersionString(resourceFirmwareVersion, true));
            }
            if (isExistProdFirmware)
            {
                versionList.Add(GetVersionString(resourceFirmwareVersion, false, true));
                versionList.Add(GetVersionString(resourceFirmwareVersion, false, false, true));
            }
            return true;
        }

        public bool EnumerateAllRevisions(ref List<string> revisionList)
        {
            revisionList.Clear();
            revisionList.Add(resourceFirmwareVersion.revisionString);
            if (isExistOceanFirmware)
            {
                revisionList.Add(resourceFirmwareVersion.revisionString);
            }
            if (isExistProdFirmware)
            {
                revisionList.Add(resourceFirmwareVersion.revisionString);
                revisionList.Add(resourceFirmwareVersion.revisionString);
            }
            return true;
        }

        public bool IsNewerVersion(string checkVersionString, string originalVersionString)
        {
            FirmwareVersion checkVersion = ConvertVersionString(checkVersionString);
            FirmwareVersion originalVersion = ConvertVersionString(originalVersionString);

            FirmwareVersionComparer comp = new FirmwareVersionComparer();
            return comp.Compare(checkVersion, originalVersion) > 0;
        }

        private class FirmwareVersionComparer : IComparer<FirmwareVersion>
        {
            public int Compare(FirmwareVersion version1, FirmwareVersion version2)
            {
                if (version1.majorVersion > version2.majorVersion)
                {
                    return 1;
                }
                if (version1.majorVersion < version2.majorVersion)
                {
                    return -1;
                }
                if (version1.minorVersion > version2.minorVersion)
                {
                    return 1;
                }
                if (version1.minorVersion < version2.minorVersion)
                {
                    return -1;
                }
                if (version1.microVersion > version2.microVersion)
                {
                    return 1;
                }
                if (version1.microVersion < version2.microVersion)
                {
                    return -1;
                }
                if (version1.majorRelStep > version2.majorRelStep)
                {
                    return 1;
                }
                if (version1.majorRelStep < version2.majorRelStep)
                {
                    return -1;
                }
                if (version1.minorRelStep > version2.minorRelStep)
                {
                    return 1;
                }
                if (version1.minorRelStep < version2.minorRelStep)
                {
                    return -1;
                }

                return 0;
            }
        }

        public string GetUpToDateVersion()
        {
            return GetVersionString(resourceFirmwareVersion);
        }

        public string GetUpToDateRevision()
        {
            return resourceFirmwareVersion.revisionString;
        }

        public string GetFileNameOfFirm(FirmwareType firmType, string versionString, TargetSpecifier.SdevVersion sdevVersion = TargetSpecifier.SdevVersion.SDEV_NONE)
        {
            FirmwareVersion version = ConvertVersionString(versionString);
            bool useOceanFirmware = CheckOceanVersion(versionString);
            bool useProdFirmware = CheckProdVersion(versionString);
            bool useProdWithLogFirmware = CheckProdWithLogVersion(versionString);
            // MEMO: skip FirmwareVersion check
            FirmwareVersionType firmVersionType;
            if (useOceanFirmware)
            {
                firmVersionType = FirmwareVersionType.Version_OceanFirm;
            }
            else if(useProdFirmware)
            {
                firmVersionType = FirmwareVersionType.Version_ProdFirm;
            }
            else if (useProdWithLogFirmware)
            {
                firmVersionType = FirmwareVersionType.Version_ProdWithLogFirm;
            }
            else
            {
                firmVersionType = FirmwareVersionType.Version_DefaultFirm;
            }

            return GetFileNameOfFirm(firmType, firmVersionType, sdevVersion);
        }

        public string GetFileNameOfFirm(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion = TargetSpecifier.SdevVersion.SDEV_NONE)
        {
#if !USE_LOCAL_IMAGES
#if SETUP_TOOLS_USE_UNSIGNED_IMAGE && !SYSTEM_UPDATE_IMAGE
            if(firmType == FirmwareType.Firmware_SystemUpdaterProcess)
            {
                // Pre processor check.
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SETUP_TOOLS_USE_UNSIGNED_IMAGE can't use without setting USE_LOCAL_IMAGES and SYSTEM_UPDATE_IMAGE for SystemUpdaterXXX.nsp .");
                return null;
            }
#endif

            if (firmType == FirmwareType.Host_RecoveryWriterUsb)
            {
                return Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Tools\CommandLineTools\RecoveryWriterUsb.exe";
            }
            else if (firmType == FirmwareType.Host_RecoveryWriterUsb_Old)
            {
                return Path.Combine(ResourceDirectory, @"Compat\RecoveryWriterUsb\001\RecoveryWriterUsb.exe");
            }
            else if (firmType == FirmwareType.Firmware_DevMenuCommand)
            {
                string path = GetDevMenuCommandFileNameOfFirm();
                if(path == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "DevMenuCommand not found.");
                }
                return path;
            }
            else if (firmType == FirmwareType.Host_RunOnTarget)
            {
                return GetRunOnTargetFileName();
            }

            if (!Directory.Exists(ResourceDirectory))
            {
                goto none_quit;
            }
            if (firmVersionType == FirmwareVersionType.Version_OceanFirm)
            {
                if (!Directory.Exists(Path.Combine(ResourceDirectory, OceanDirectory)))
                {
                    goto none_quit;
                }
            }

            if (firmType == FirmwareType.Host_TargetManager)
            {
                return GetOasisDirectoryName(firmType, firmVersionType, sdevVersion);
            }
            else if (firmType == FirmwareType.Host_TargetManager_Old)
            {
                return GetOldOasisDirectoryName(firmType, firmVersionType, sdevVersion);
            }
            else if (IsDevKitUpdaterFileType(firmType))
            {
                return GetDevKitUpdaterFileNameOfFirm(firmType, firmVersionType, sdevVersion);
            }
            else if (IsSystemUpdaterFirmType(firmType))
            {
                return GetSystemUpdaterFileNameOfFirm(firmType, firmVersionType, sdevVersion);
            }
            else if (IsSafeModeUpdaterFirmType(firmType))
            {
                return GetSafeModeUpdaterFileNameOfFirm(firmType, firmVersionType, sdevVersion);
            }

            else
            {
                return GetOtherFirmFileName(firmType, firmVersionType, sdevVersion);
            }

        none_quit:
#else
            bool isOcean = firmVersionType == FirmwareVersionType.Version_OceanFirm;
            bool isProd = firmVersionType == FirmwareVersionType.Version_ProdFirm;
            bool isProdWithLog = firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm;

            switch (firmType)
            {
                case FirmwareType.Firmware_HelloWorldImage:
                    break;
                case FirmwareType.Firmware_HostBridge:
                    return FirmwareLocalAccessor.GetHostBridgeFirmwarePath(false);
                case FirmwareType.Firmware_HostBridgeRecovery:
                    return FirmwareLocalAccessor.GetHostBridgeFirmwarePath(true);
                case FirmwareType.Firmware_QspiBootImage:
                    return FirmwareLocalAccessor.GetQspiBootImagePath(sdevVersion);
#if SYSTEM_UPDATE_IMAGE
                case FirmwareType.Firmware_SystemImage:
                    return FirmwareLocalAccessor.GetSystemImagePath(sdevVersion, isOcean, isProd, isProdWithLog);
                case FirmwareType.Firmware_SafeModeImage:
                    return FirmwareLocalAccessor.GetSafeModeImagePath(sdevVersion, isOcean, isProd, isProdWithLog);
#endif
                case FirmwareType.Firmware_SystemUpdaterProcess:
#if SETUP_TOOLS_USE_UNSIGNED_IMAGE && !SYSTEM_UPDATE_IMAGE
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SETUP_TOOLS_USE_UNSIGNED_IMAGE can't use without setting SYSTEM_UPDATE_IMAGE for SystemUpdaterXXX.nsp .");
                    return null;
#else
                    return FirmwareLocalAccessor.GetSystemUpdaterPath(sdevVersion, isOcean, isProd, isProdWithLog);
#endif
                case FirmwareType.Firmware_SafeModeUpdaterProcess:
                    return FirmwareLocalAccessor.GetSafeModeUpdaterPath(sdevVersion, isOcean, isProd, isProdWithLog);
                case FirmwareType.Firmware_DevKitUpdater:
                    return FirmwareLocalAccessor.GetDevKitUpdaterPath(sdevVersion, isOcean, isProd, isProdWithLog);
                case FirmwareType.Host_TargetManager:
                    return FirmwareLocalAccessor.GetTargetManagerPath();
                case FirmwareType.Host_TargetManager_Old:
                    return FirmwareLocalAccessor.GetOldTargetManagerPath();
                case FirmwareType.Host_RecoveryWriterUsb:
                    return FirmwareLocalAccessor.GetRecoveryWriterUsbPath();
                case FirmwareType.Host_RecoveryWriterUsb_Old:
                    return FirmwareLocalAccessor.GetRecoveryWriterUsbPath(true);
            }

#endif
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private string GetOasisDirectoryName(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            string firmwareDirectory = ResourceDirectory;

            if (firmType == FirmwareType.Host_TargetManager)
            {
                IEnumerable<string> dirs = Directory.EnumerateDirectories(firmwareDirectory);
                foreach (string dir in dirs)
                {
                    if (dir.Contains("Oasis"))
                    {
                        if (Directory.Exists(dir))
                        {
                            return dir;
                        }
                        goto none_quit;
                    }
                }
                goto none_quit;
            }

        none_quit:
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private string GetOldOasisDirectoryName(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            string firmwareDirectory = ResourceDirectory;

            if (firmType == FirmwareType.Host_TargetManager_Old)
            {
                string tmDir = GetOasisDirectoryName(FirmwareType.Host_TargetManager, firmVersionType, sdevVersion);
                if (tmDir == null)
                {
                    goto none_quit;
                }
                string tmOldDir = Path.Combine(tmDir, @"OldVersions\14483");
                if (Directory.Exists(tmOldDir))
                {
                    return tmOldDir;
                }
            }

        none_quit:
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private string GetSystemUpdaterFileNameOfFirm(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            if (sdevVersion == TargetSpecifier.SdevVersion.SDEV_NONE)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "GetSystemUpdaterFileNameOfFirm : Need to set devkit version.");
                return null;
            }

            List<string> foundStringList = new List<string>();
            string firmwareDirectory = firmVersionType == FirmwareVersionType.Version_OceanFirm ? Path.Combine(ResourceDirectory, OceanDirectory) : ResourceDirectory;
            IEnumerable<string> files = Directory.EnumerateFiles(firmwareDirectory);
            foreach (string file in files)
            {
                string ext = firmwareExtDictionary[firmType];
                if (ext == string.Empty)
                {
                    continue;
                }
                int fileLen = file.Length;
                if (CheckFileNameExtension(file, ext))
                {
                    if (firmType == FirmwareType.Firmware_SystemUpdaterProcess)
                    {
                        string checkString = "SystemUpdater";
                        string fileNameWitoutExt = GetFileNameWithoutAllExtension(file, ext);
                        if (fileNameWitoutExt.Length < checkString.Length || fileNameWitoutExt.Substring(0, checkString.Length) != checkString)
                        {
                            continue;
                        }
                        if(!CheckProdSystemUpdaterName(fileNameWitoutExt, firmVersionType))
                        {
                            continue;
                        }
                        switch (sdevVersion)
                        {
                            case TargetSpecifier.SdevVersion.SDEV_1_8:
                            case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                                if (CheckKeyTypeAndDevkitType(fileNameWitoutExt, "K2", sdevVersion, firmVersionType) == false)
                                {
                                    continue;
                                }
                                break;
                            case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                            case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                            case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                                if (CheckKeyTypeAndDevkitType(fileNameWitoutExt, "K3", sdevVersion, firmVersionType) == false)
                                {
                                    continue;
                                }
                                break;
                            case TargetSpecifier.SdevVersion.SDEV_MP:
                            case TargetSpecifier.SdevVersion.EDEV_MP:
                                if (CheckKeyTypeAndDevkitType(fileNameWitoutExt, "K5", sdevVersion, firmVersionType) == false)
                                {
                                    continue;
                                }
                                break;
                            default:
                                if (fileNameWitoutExt.Contains("K2") || fileNameWitoutExt.Contains("K3") || fileNameWitoutExt.Contains("K5"))
                                {
                                    continue;
                                }
                                break;
                        }
                    }
                    foundStringList.Add(file);
                }
            }

            if (foundStringList.Count == 1)
            {
                return foundStringList.First();
            }
            else if (foundStringList.Count > 1)
            {
                return GetShortestElement(foundStringList);
            }

            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private string GetSafeModeUpdaterFileNameOfFirm(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            if (sdevVersion == TargetSpecifier.SdevVersion.SDEV_NONE)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "GetSafeModeUpdaterFileNameOfFirm : Need to set devkit version.");
                return null;
            }

            List<string> foundStringList = new List<string>();
            string firmwareDirectory = firmVersionType == FirmwareVersionType.Version_OceanFirm ? Path.Combine(ResourceDirectory, OceanDirectory) : ResourceDirectory;
            IEnumerable<string> files = Directory.EnumerateFiles(firmwareDirectory);
            foreach (string file in files)
            {
                string ext = firmwareExtDictionary[firmType];
                if (ext == string.Empty)
                {
                    continue;
                }
                int fileLen = file.Length;
                if (CheckFileNameExtension(file, ext))
                {
                    if (firmType == FirmwareType.Firmware_SafeModeUpdaterProcess)
                    {
                        string checkString = "DevSafeModeUpdater";
                        string fileNameWitoutExt = GetFileNameWithoutAllExtension(file, ext);
                        if (fileNameWitoutExt.Length < checkString.Length || fileNameWitoutExt.Substring(0, checkString.Length) != checkString)
                        {
                            continue;
                        }
                        switch (sdevVersion)
                        {
                            case TargetSpecifier.SdevVersion.EDEV_MP:
                                if (CheckKeyTypeAndDevkitType2(fileNameWitoutExt, "K5", sdevVersion, firmVersionType) == false)
                                {
                                    continue;
                                }
                                break;
                            default:
                                if (fileNameWitoutExt.Contains("K2") || fileNameWitoutExt.Contains("K3") || fileNameWitoutExt.Contains("K5"))
                                {
                                    continue;
                                }
                                break;
                        }
                    }
                    foundStringList.Add(file);
                }
            }

            if (foundStringList.Count == 1)
            {
                return foundStringList.First();
            }
            else if (foundStringList.Count > 1)
            {
                return GetShortestElement(foundStringList);
            }

            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private string GetDevKitUpdaterFileNameOfFirm(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            string returnPath = null;

            if (firmType == FirmwareType.Firmware_DevKitUpdater)
            {
                string fileName = string.Format("DevKitUpdater{0}{1}{2}.nsp", GetDevKitName(sdevVersion, firmVersionType), GetKeyTypeName(sdevVersion), GetFirmPostfixString(firmVersionType));
                returnPath = Path.Combine(ResourceDirectory, fileName);
            }

            if (!File.Exists(returnPath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                                firmwareDescriptionDictionary[firmType],
                                GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                                sdevVersion.ToString());

                returnPath = null;
            }

            return returnPath;
        }

        private string GetDevMenuCommandFileNameOfFirm()
        {
            string path = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\TargetTools\NX-NXFP2-a64\DevMenuCommand\Release\DevMenuCommand.nsp";
            if(!File.Exists(path))
            {
                // string path2 = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommand\Release\DevMenuCommand.nsp";
                string path2 = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommand\Develop\DevMenuCommand.nsp";
                if (File.Exists(path2))
                {
                    return path2;
                }
                return null;
            }
            return path;
        }

        private string GetRunOnTargetFileName()
        {
            string path = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Tools\CommandLineTools\RunOnTarget.exe";
            if (!File.Exists(path))
            {
                // string path2 = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\Release\DevMenuCommandSystem.nsp";
                string path2 = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Chris\Outputs\x86\Tools\RunnerTools\RunOnTarget\Release\RunOnTarget.exe";
                if (File.Exists(path2))
                {
                    return path2;
                }
                return null;
            }
            return path;
        }

        private string GetShortestElement(List<string> elementList)
        {
            string shortestString = null;
            int shortestStringLen = int.MaxValue;
            foreach(string element in elementList)
            {
                if(element.Length < shortestStringLen)
                {
                    shortestString = element;
                    shortestStringLen = element.Length;
                }
            }
            return shortestString;
        }

        private bool CheckKeyTypeAndDevkitType(string checkFileName, string keyName, TargetSpecifier.SdevVersion sdevVersion, FirmwareVersionType firmVersionType)
        {
            if (!checkFileName.Contains(keyName))
            {
                return false;
            }
            else if (!IsEdev(sdevVersion) && !checkFileName.Contains("Sdev") && !IsProdOrProdWithLogVersion(firmVersionType))
            {
                return false;
            }
            else if (IsEdev(sdevVersion) && !checkFileName.Contains("Edev") && !IsProdOrProdWithLogVersion(firmVersionType))
            {
                return false;
            }
            return true;
        }

        private bool CheckKeyTypeAndDevkitType2(string checkFileName, string keyName, TargetSpecifier.SdevVersion sdevVersion, FirmwareVersionType firmVersionType)
        {
            if (!checkFileName.Contains(keyName))
            {
                return false;
            }
            return true;
        }


        private bool CheckProdSystemUpdaterName(string fileNameWitoutExt, FirmwareVersionType firmVersionType)
        {
            const string ProdName = "Prod";
            const string ProdWithLogName = "ProdModeWithLog";
            switch (firmVersionType)
            {
                case FirmwareVersionType.Version_ProdFirm:
                    return fileNameWitoutExt.Contains(ProdName) && !fileNameWitoutExt.Contains(ProdWithLogName);
                case FirmwareVersionType.Version_ProdWithLogFirm:
                    return fileNameWitoutExt.Contains(ProdWithLogName);
                default:
                    return !(fileNameWitoutExt.Contains(ProdName) || fileNameWitoutExt.Contains(ProdWithLogName));
            }
        }

        private string GetOtherFirmFileName(FirmwareType firmType, FirmwareVersionType firmVersionType, TargetSpecifier.SdevVersion sdevVersion)
        {
            if (firmType == FirmwareType.Firmware_QspiBootImage && sdevVersion == TargetSpecifier.SdevVersion.SDEV_NONE)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "GetOtherFirmFileName({0}) : Need to set devkit version.", firmType);
                return null;
            }

            List<string> foundStringList = new List<string>();
            string firmwareDirectory = ResourceDirectory;
            IEnumerable<string> files = Directory.EnumerateFiles(firmwareDirectory);
            foreach (string file in files)
            {
                string ext = firmwareExtDictionary[firmType];
                if (ext == string.Empty)
                {
                    continue;
                }
                int fileLen = file.Length;
                if (CheckFileNameExtension(file, ext))
                {
                    if (firmType == FirmwareType.Firmware_QspiBootImage)
                    {
                        string fileNameWitoutExt = GetFileNameWithoutAllExtension(file, ext);
                        switch (sdevVersion)
                        {
                            case TargetSpecifier.SdevVersion.SDEV_1_8:
                            case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                                if (CheckKeyTypeAndDevkitTypeOfRecoveryImage(fileNameWitoutExt, "K2", sdevVersion) == false)
                                {
                                    continue;
                                }
                                break;
                            case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                            case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                            case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                                if (CheckKeyTypeAndDevkitTypeOfRecoveryImage(fileNameWitoutExt, "K3", sdevVersion) == false)
                                {
                                    continue;
                                }
                                break;
                            case TargetSpecifier.SdevVersion.SDEV_MP:
                            case TargetSpecifier.SdevVersion.EDEV_MP:
                                if (CheckKeyTypeAndDevkitTypeOfRecoveryImage(fileNameWitoutExt, "K5", sdevVersion) == false)
                                {
                                    continue;
                                }
                                break;
                            default:
                                if (fileNameWitoutExt.Contains("K2") || fileNameWitoutExt.Contains("K3") || fileNameWitoutExt.Contains("K5"))
                                {
                                    continue;
                                }
                                break;
                        }
                    }
                    else if (firmType == FirmwareType.Firmware_HelloWorldImage)
                    {
                        string checkString = "HelloWorld";
                        string fileNameWitoutExt = GetFileNameWithoutAllExtension(file, ext);
                        if (fileNameWitoutExt.Length < checkString.Length || fileNameWitoutExt.Substring(0, checkString.Length) != checkString)
                        {
                            continue;
                        }
                    }
                    else if (firmType == FirmwareType.Firmware_HostBridgeRecovery)
                    {
                        string fileNameWitoutExt = GetFileNameWithoutAllExtension(file, ext);
                        if (!fileNameWitoutExt.Contains("HostBridgeRecovery"))
                        {
                            continue;
                        }
                    }
                    foundStringList.Add(file);
                }
            }

            if (foundStringList.Count == 1)
            {
                return foundStringList.First();
            }
            else if (foundStringList.Count > 1)
            {
                return GetShortestElement(foundStringList);
            }

            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found (type : {0}, version : {1}, {2})",
                            firmwareDescriptionDictionary[firmType],
                            GetVersionString(resourceFirmwareVersion, firmVersionType == FirmwareVersionType.Version_OceanFirm, firmVersionType == FirmwareVersionType.Version_ProdFirm, firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm),
                            sdevVersion.ToString());

            return null;
        }

        private bool CheckFileNameExtension(string path, string extName)
        {
            return path.Substring(path.Length - extName.Length) == extName;
        }

        private string GetFileNameWithoutAllExtension(string path, string extName)
        {
            string pathWithoutExtension = path;
            if(CheckFileNameExtension(path, extName))
            {
                pathWithoutExtension = path.Substring(0, path.Length - extName.Length);
            }
            return Path.GetFileNameWithoutExtension(pathWithoutExtension);
        }

        private bool CheckKeyTypeAndDevkitTypeOfRecoveryImage(string checkFileName, string keyName, TargetSpecifier.SdevVersion sdevVersion)
        {
            if (!checkFileName.Contains(keyName))
            {
                return false;
            }
            else if (!IsEdev(sdevVersion) && !checkFileName.Contains("Hb"))
            {
                return false;
            }
            else if (IsEdev(sdevVersion) && !checkFileName.Contains("Usb"))
            {
                return false;
            }
            return true;
        }

        private bool IsSystemUpdaterFirmType(FirmwareType type)
        {
            switch (type)
            {
                case FirmwareType.Firmware_SystemUpdaterProcess:
                    return true;
                default:
                    return false;
            }
        }

        private bool IsSafeModeUpdaterFirmType(FirmwareType type)
        {
            switch (type)
            {
                case FirmwareType.Firmware_SafeModeUpdaterProcess:
                    return true;
                default:
                    return false;
            }
        }


        private bool IsDevKitUpdaterFileType(FirmwareType type)
        {
            switch(type)
            {
                case FirmwareType.Firmware_DevKitUpdater:
                    return true;
                default:
                    return false;
            }
        }

        private string GetFileNameOfSystemInitializeTool(string versionString)
        {
            return null;
        }

        public string GetVersionOfFirm(FirmwareType firmType, string versionString)
        {
            return null;
        }

        public bool IsVersionFormat(string versionString)
        {
            return Regex.IsMatch(versionString, VersionFormatRegex) || Regex.IsMatch(versionString, VersionFormatRegexOld) || Regex.IsMatch(versionString, VersionFormatRegexUnfinished);
        }

        public bool GetResourceFirmwareVersion(ref FirmwareVersion version, string path)
        {
            if (!Directory.Exists(path))
            {
                return false;
            }
            // FirmwareVersion
            string versionFilePath = path + "\\" + "FirmwareVersion.txt";
            if (!File.Exists(versionFilePath))
            {
                return false;
            }

            version = new FirmwareVersion();
            using (StreamReader reader = new StreamReader(versionFilePath))
            {
                while (reader.Peek() >= 0)
                {
                    string line = reader.ReadLine();
                    bool bRet;
                    bRet = ParseDefineMacroInt(ref version.majorVersion, "NN_FIRMWARE_VERSION_MAJOR", line);
                    if (!bRet)
                    {
                        bRet = ParseDefineMacroInt(ref version.minorVersion, "NN_FIRMWARE_VERSION_MINOR", line);
                    }
                    if (!bRet)
                    {
                        bRet = ParseDefineMacroInt(ref version.microVersion, "NN_FIRMWARE_VERSION_MICRO", line);
                    }
                    if (!bRet)
                    {
                        bRet = ParseDefineMacroInt(ref version.majorRelStep, "NN_FIRMWARE_VERSION_MAJORRELSTEP", line);
                    }
                    if (!bRet)
                    {
                        bRet = ParseDefineMacroInt(ref version.minorRelStep, "NN_FIRMWARE_VERSION_MINORRELSTEP", line);
                    }
                    if (!bRet)
                    {
                        // MEMO: obsolete
                        bRet = ParseDefineMacroInt(ref version.majorRelStep, "NN_FIRMWARE_VERSION_RELSTEP", line);
                    }
                }
            }

            //FirmwareRevision
            string revisionFilePath = path + "\\" + "FirmwareRevision.txt";
            if (File.Exists(revisionFilePath))
            {
                using (StreamReader reader = new StreamReader(revisionFilePath))
                {
                    string line = reader.ReadLine();
                    if(line != null)
                    {
                        string revisionString = line.Trim();
                        version.revisionString = revisionString.Length > 10 ? revisionString.Remove(10) : revisionString;
                    }
                }
            }

            return true;
        }

        private static bool ParseDefineMacroInt(ref int value, string macroName, string line)
        {
            const string DefineString = "#define";

            string[] words = line.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
            if(words.Length > 1 && words[0].Equals(DefineString) && words[1].Equals(macroName))
            {
                if (!int.TryParse(words[2], out value))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "firmware definition macro ({1}) parse failed ({0})", words[2], macroName);
                }
                return true;
            }

            return false;
        }

        // TODO: 第2引数以降はenum型でまとめるべき
        private string GetVersionString(FirmwareVersion version, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
            if(isOcean)
            {
                return string.Format("{0}.{1}.{2}-{3}.{4} + {5}", version.majorVersion, version.minorVersion, version.microVersion, version.majorRelStep, version.minorRelStep, OceanVersionString);
            }
            else if (isProd)
            {
                return string.Format("{0}.{1}.{2}-{3}.{4} + {5}", version.majorVersion, version.minorVersion, version.microVersion, version.majorRelStep, version.minorRelStep, ProdVersionString);
            }
            else if (isProdWithLog)
            {
                return string.Format("{0}.{1}.{2}-{3}.{4} + {5}", version.majorVersion, version.minorVersion, version.microVersion, version.majorRelStep, version.minorRelStep, ProdWithLogVersionString);
            }
            else
            {
                return string.Format("{0}.{1}.{2}-{3}.{4}", version.majorVersion, version.minorVersion, version.microVersion, version.majorRelStep, version.minorRelStep);
            }
        }

        private FirmwareVersion ConvertVersionString(string versionString)
        {
            string versionStringWithoutSpace = versionString.Replace(" ", string.Empty);
            FirmwareVersion version = new FirmwareVersion();
            Regex regex = new Regex(VersionFormatRegex);
            Match versionMatch = regex.Match(versionStringWithoutSpace);

            if (versionMatch.Success)
            {
                int.TryParse(versionMatch.Groups[1].Value, out version.majorVersion);
                int.TryParse(versionMatch.Groups[2].Value, out version.minorVersion);
                int.TryParse(versionMatch.Groups[3].Value, out version.microVersion);
                int.TryParse(versionMatch.Groups[4].Value, out version.majorRelStep);
                int.TryParse(versionMatch.Groups[5].Value, out version.minorRelStep);
            }
            else
            {
                Regex regexOld = new Regex(VersionFormatRegexOld);
                versionMatch = regexOld.Match(versionStringWithoutSpace);
                if(versionMatch.Success)
                {
                    int.TryParse(versionMatch.Groups[1].Value, out version.majorVersion);
                    int.TryParse(versionMatch.Groups[2].Value, out version.minorVersion);
                    int.TryParse(versionMatch.Groups[3].Value, out version.microVersion);
                    int.TryParse(versionMatch.Groups[4].Value, out version.majorRelStep);
                }
                else
                {
                    Regex regex2 = new Regex(VersionFormatRegexUnfinished);
                    versionMatch = regex.Match(versionStringWithoutSpace);
                    if (versionMatch.Success)
                    {
                        int.TryParse(versionMatch.Groups[1].Value, out version.majorVersion);
                        int.TryParse(versionMatch.Groups[2].Value, out version.minorVersion);
                        int.TryParse(versionMatch.Groups[3].Value, out version.microVersion);
                    }
                }
            }

            return version;
        }

        public bool CheckOceanVersion(string versionString)
        {
            string versionStringWithoutSpace = versionString.Replace(" ", string.Empty).ToLower();
            return versionStringWithoutSpace.Contains(OceanVersionString.ToLower());
        }

        public static bool CheckProdVersion(string versionString)
        {
            // MEMO: ProdWithLog ではないことも確認している
            string versionStringWithoutSpace = versionString.Replace(" ", string.Empty).ToLower();
            return versionStringWithoutSpace.Contains(ProdVersionString.ToLower()) && !versionStringWithoutSpace.Contains(ProdWithLogVersionString.ToLower());
        }

        public static bool CheckProdWithLogVersion(string versionString)
        {
            string versionStringWithoutSpace = versionString.Replace(" ", string.Empty).ToLower();
            return versionStringWithoutSpace.Contains(ProdWithLogVersionString.ToLower());
        }

        public bool CheckUtilizableVersion(string versionString, TargetSpecifier.SdevVersion sdevVersion)
        {
            bool isProd = CheckProdVersion(versionString);
            bool isProdWithLog = CheckProdWithLogVersion(versionString);

            // TODO: EDEV EP2-2 の場合はサポート外であることを警告するようにする

            if(isProd || isProdWithLog)
            {
                if(sdevVersion != TargetSpecifier.SdevVersion.SDEV_MP && sdevVersion != TargetSpecifier.SdevVersion.EDEV_MP)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "\"{0}\" type firmware supports only {1}.    (version: {2})",
                                              isProd ? "+ Prod" : "+ ProdWithLog",
                                              IsEdev(sdevVersion) ? "EDEV MP": "SDEV MP",
                                              versionString);
                    return false;
                }
            }

            return true;
        }

        // TODO: 似たような処理の関数が重複しているので整理したほうがいい
        private string GetDevKitName(TargetSpecifier.SdevVersion sdevVersion, FirmwareVersionType firmVersionType)
        {
            if(firmVersionType == FirmwareVersionType.Version_ProdFirm || firmVersionType == FirmwareVersionType.Version_ProdWithLogFirm)
            {
                return "Nx";
            }
            if(IsEdev(sdevVersion))
            {
                return "Edev";
            }
            return "Sdev";
        }

        private string GetKeyTypeName(TargetSpecifier.SdevVersion sdevVersion)
        {
            switch (sdevVersion)
            {
                case TargetSpecifier.SdevVersion.SDEV_1_8:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                    return "K2";
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                    return "K3";
                case TargetSpecifier.SdevVersion.SDEV_MP:
                case TargetSpecifier.SdevVersion.EDEV_MP:
                    return "K5";
                default:
                    return string.Empty;
            }

        }

        private string GetFirmPostfixString(FirmwareVersionType firmVersionType)
        {
#if !SETUP_TOOLS_USE_UNSIGNED_IMAGE
            string string1 = string.Empty;
#else
            string string1 = "Unsigned";
#endif
            string string2 = string.Empty;
            switch(firmVersionType)
            {
                case FirmwareVersionType.Version_ProdFirm:
                    string2 = "Prod";
                    break;
                case FirmwareVersionType.Version_ProdWithLogFirm:
                    string2 = "ProdLog";
                    break;
            }
            return string1 + string2;
        }

        private const string ResourceDirectoryBase = @"Resources\Firmwares\NX";
        private const string VersionFormatRegex = @"^([^.]+).([^.]+).([0-9]+)-([0-9]+).([0-9]+)";
        private const string VersionFormatRegexOld = @"^([^.]+).([^.]+).([0-9]+)-([0-9]+)";
        private const string VersionFormatRegexUnfinished = @"^([^.]+).([^.]+).([0-9]+)";
        private const string OceanDirectory = @"WithApplet";
        private const string DefaultOceanSystemUpdaterName = @"SystemUpdaterWithApplet.nsp";
        private const string DefaultProdSystemUpdaterName = @"SystemUpdaterK5ProdMode.nsp";
        private const string OceanVersionString = @"SystemSoftware";
        private const string ProdVersionString = @"Prod";
        private const string ProdWithLogVersionString = @"ProdWithLog";

        private string ResourceDirectory;
    }

#if USE_LOCAL_IMAGES
    // MEMO: このクラスは FirmwareResourceAccessor から呼び出しているものの
    //       デバッグ目的でファイルパスを取得するために直接呼び出すことが可能です
    public static class FirmwareLocalAccessor
    {
#if USE_USB
#if !SETUP_TOOLS_USE_UNSIGNED_IMAGE
            static string connectName = "Usb";
            static string signName = "Signed";
#else
            static string connectName = "Usb";
            static string signName = "Unsigned";
#endif
#else
#if !SETUP_TOOLS_USE_UNSIGNED_IMAGE
            static string connectName = "Hb";
            static string connectEdevName = "Usb";
            static string signName = "Signed";
#else
            static string connectName = "Hb";
            static string connectEdevName = "Usb";
            static string signName = "Unsigned";
#endif
#if SYSTEM_UPDATE_IMAGE
            static string connectForProdName = "NoConnect";
#endif
#endif

        static string GetDevkitKeyName(TargetSpecifier.SdevVersion sdevVersion)
        {
            string sdevKeyFormatName = string.Empty;
            switch(sdevVersion)
            {
                case TargetSpecifier.SdevVersion.SDEV_1_5:
                case TargetSpecifier.SdevVersion.SDEV_1_6:
                case TargetSpecifier.SdevVersion.SDEV_1_6_G:
                    sdevKeyFormatName = "K1";
                    break;
                case TargetSpecifier.SdevVersion.SDEV_1_8:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                    sdevKeyFormatName = "K2";
                    break;
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                    sdevKeyFormatName = "K3";
                    break;
                case TargetSpecifier.SdevVersion.SDEV_MP:
                case TargetSpecifier.SdevVersion.EDEV_MP:
                    sdevKeyFormatName = "K5";
                    break;
            }
            return sdevKeyFormatName;
        }

        static bool IsEdev(TargetSpecifier.SdevVersion sdevVersion)
        {
            switch (sdevVersion)
            {
                case TargetSpecifier.SdevVersion.EDEV_EP_2_1:
                case TargetSpecifier.SdevVersion.EDEV_EP_2_2:
                case TargetSpecifier.SdevVersion.EDEV_MP:
                    return true;
            }
            return false;
        }

        static string GetDevkitName(TargetSpecifier.SdevVersion sdevVersion)
        {
            return IsEdev(sdevVersion) ? "Edev" : "Sdev";
        }

        private static bool FileCheck(string filePath)
        {
            if (!System.IO.File.Exists(filePath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "file not found : {0}", filePath);
                return false;
            }
            return true;
        }

        private static bool DirectoryCheck(string directoryPath)
        {
            if (!System.IO.Directory.Exists(directoryPath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "directory not found : {0}", directoryPath);
                return false;
            }
            return true;
        }

        public static string GetQspiBootImagePath(TargetSpecifier.SdevVersion sdevVersion)
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\SystemImages\QspiBootImages\";
            string addtionalPath = @"RecoveryWriter-CONV1-CONV2-CONV3\Develop\RecoveryWriter-CONV1-CONV2-CONV3.qspi.img";
            string sdevKeyFormatName = GetDevkitKeyName(sdevVersion);
            string returnPath = basePath + addtionalPath;
            returnPath = returnPath.Replace("CONV1", sdevKeyFormatName);
            returnPath = returnPath.Replace("CONV2", IsEdev(sdevVersion) ? connectEdevName : connectName);
            returnPath = returnPath.Replace("CONV3", signName);
            return FileCheck(returnPath) ? returnPath : null;
        }

        public static string GetSystemUpdaterPath(TargetSpecifier.SdevVersion sdevVersion, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
#if !SYSTEM_UPDATE_IMAGE
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Chris\Outputs\NX-NXFP2-a64\TargetTools\SystemUpdaterCONV1CONV3\";
            string addtionalPath = @"Develop\SystemUpdaterCONV2CONV3.nsp";
            string returnPath = basePath + addtionalPath;
            string keyName = GetDevkitKeyName(sdevVersion);
            string devkitName = GetDevkitName(sdevVersion);
            string keyFormatName = keyName == "K1" ? string.Empty : keyName;
            string devkitFormatName = keyName == "K1" ? string.Empty : devkitName;
            string extraFormatName = string.Empty;
            if (isOcean)
            {
                extraFormatName = "WithApplet";
            }
            else if (isProd)
            {
                devkitFormatName = string.Empty;
                extraFormatName = "ProdMode";
            }
            else if (isProdWithLog)
            {
                devkitFormatName = string.Empty;
                extraFormatName = "ProdModeWithLog";
            }
            returnPath = returnPath.Replace("CONV1", devkitFormatName + keyFormatName);
            returnPath = returnPath.Replace("CONV2", devkitFormatName + keyFormatName);
            returnPath = returnPath.Replace("CONV3", extraFormatName);
#else
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Chris\Outputs\NX-NXFP2-a64\TargetTools\SystemUpdaterHostFs\";
            string addtionalPath = @"Develop\SystemUpdaterHostFs.nsp";
            string returnPath = basePath + addtionalPath;
#endif
            return FileCheck(returnPath) ? returnPath : null;
        }
        public static string GetSafeModeUpdaterPath(TargetSpecifier.SdevVersion sdevVersion, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Chris\Outputs\NX-NXFP2-a64\TargetTools\DevSafeModeUpdaterCONV1\";
            string addtionalPath = @"Develop\DevSafeModeUpdaterCONV2.nsp";
            string returnPath = basePath + addtionalPath;
            string keyName = GetDevkitKeyName(sdevVersion);
            string keyFormatName = keyName == "K1" ? string.Empty : keyName;
            returnPath = returnPath.Replace("CONV1", keyFormatName);
            returnPath = returnPath.Replace("CONV2", keyFormatName);
            return FileCheck(returnPath) ? returnPath : null;
        }


#if SYSTEM_UPDATE_IMAGE
        public static string GetSystemImagePath(TargetSpecifier.SdevVersion sdevVersion, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\InitialImages\";
            string addtionalPath = @"NX-CONV1-CONV2-CONV3-CONV6-CONV7-CONV4CONV5\Develop\NX-CONV1-CONV2-CONV3-CONV6-CONV7-CONV4CONV5.initimg";
            string sdevKeyFormatName = GetDevkitKeyName(sdevVersion);
            string extraFormatName = string.Empty;
            if(isOcean || isProd || isProdWithLog)
            {
                extraFormatName = "-Ocean";
            }
            string conv2String = IsEdev(sdevVersion) ? connectEdevName : connectName;
#if !SETUP_TOOLS_USE_UNSIGNED_IMAGE
            string conv4String = IsEdev(sdevVersion) ? "Internal" : "Public";
#else
            string conv4String = "Internal";
#endif
            string conv6String = "Nand";
            string conv7String = "64G";
            if (isProd)
            {
                conv2String = connectForProdName;
                conv4String = "EndUser";
                conv6String = "ProdBoot";
                conv7String = "32G";
            }
            else if(isProdWithLog)
            {
                conv2String = connectForProdName;
                conv4String = "EndUserSdLog";
                conv6String = "ProdBoot";
                conv7String = "32G";
            }
            string returnPath = basePath + addtionalPath;
            returnPath = returnPath.Replace("CONV1", sdevKeyFormatName);
            returnPath = returnPath.Replace("CONV2", conv2String);
            returnPath = returnPath.Replace("CONV3", signName);
            returnPath = returnPath.Replace("CONV4", conv4String);
            returnPath = returnPath.Replace("CONV5", extraFormatName);
            returnPath = returnPath.Replace("CONV6", conv6String);
            returnPath = returnPath.Replace("CONV7", conv7String);
            return FileCheck(returnPath) ? returnPath : null;
        }
#endif

#if SYSTEM_UPDATE_IMAGE
        public static string GetSafeModeImagePath(TargetSpecifier.SdevVersion sdevVersion, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-NXFP2-a64\InitialImages\";
            string addtionalPath = @"NX-CONV1-CONV2-CONV3-DevUpdater\Develop\NX-CONV1-CONV2-CONV3-DevUpdater.initimg";
            string sdevKeyFormatName = GetDevkitKeyName(sdevVersion);
            string extraFormatName = string.Empty;
            string conv2String = IsEdev(sdevVersion) ? connectEdevName : connectName;
            string returnPath = basePath + addtionalPath;

            returnPath = returnPath.Replace("CONV1", sdevKeyFormatName);
            returnPath = returnPath.Replace("CONV2", conv2String);
            returnPath = returnPath.Replace("CONV3", signName);
            return FileCheck(returnPath) ? returnPath : null;
        }
#endif

        public static string GetHostBridgeFirmwarePath(bool recovery = false)
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Externals\HostBridge\";
            StringBuilder addtionalPath = new StringBuilder();
            if (!recovery)
            {
                 addtionalPath.Append(@"images\package.nhf");
            }
            else
            {
                addtionalPath.Append(@"images\recovery.nhf");
            }
            string returnPath = basePath + addtionalPath;
            return FileCheck(returnPath) ? returnPath : null;
        }

        public static string GetTargetManagerPath()
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Externals\Oasis\";
            //string addtionalPath = @"bin\NintendoTargetManager.exe";
            string addtionalPath = @"bin";
            string returnPath = basePath + addtionalPath;
            return DirectoryCheck(returnPath) ? returnPath : null;
        }

        public static string GetOldTargetManagerPath()
        {
            string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Externals\Oasis\";
            string addtionalPath = @"OldVersions\14483";
            string returnPath = basePath + addtionalPath;
            return DirectoryCheck(returnPath) ? returnPath : null;
        }

        public static string GetRecoveryWriterUsbPath(bool isOld = false)
        {
            string returnPath;
            if (!isOld)
            {
                // MEMO: RecoveryWriterUsb.exe is built by specs=NX platforms==NX-Win32_VS2015
                returnPath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Programs\Eris\Outputs\NX-Win32-v140\TargetTools\RecoveryWriterUsb\Release\RecoveryWriterUsb.exe";
            }
            else
            {
                string basePath = Nintendo.ControlTarget.PathUtility.FindSigloRoot() + @"\Externals\SetupToolBinaries\bin\RecoveryWriterUsb";
                string addtionalPath = @"001\Release\RecoveryWriterUsb.exe";
                returnPath = Path.Combine(basePath, addtionalPath);
            }
            return FileCheck(returnPath) ? returnPath : null;
        }

        public static string GetVoidApplicationImagePagh()
        {
            string basePath = Path.Combine(Nintendo.ControlTarget.PathUtility.FindSigloRoot(), @"Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\VoidApplication\");
            string addtionalPath = @"Develop\VoidApplication.nsp";
            string returnPath = Path.Combine(basePath, addtionalPath);
            return FileCheck(returnPath) ? returnPath : null;
        }

        public static string GetDevKitUpdaterPath(TargetSpecifier.SdevVersion sdevVersion, bool isOcean = false, bool isProd = false, bool isProdWithLog = false)
        {
            string basePath = Path.Combine(Nintendo.ControlTarget.PathUtility.FindSigloRoot(), @"Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\");
            string addtionalPath = @"DevKitUpdaterCONV1CONV2CONV3\Develop\DevKitUpdaterCONV1CONV2CONV3";
#if !USE_DEVKITUPDATER_NSPDROOT
            addtionalPath += ".nsp";
#else
            addtionalPath += ".nspd_root";
#endif
            string returnPath = Path.Combine(basePath, addtionalPath);
            string keyName = GetDevkitKeyName(sdevVersion);
            string devkitName = GetDevkitName(sdevVersion);
            string keyFormatName = keyName;
            string devkitFormatName = devkitName;
            string extraFormatName = string.Empty;
#if SETUP_TOOLS_USE_UNSIGNED_IMAGE
            extraFormatName += "Unsigned";
#endif
            if (isProd)
            {
                devkitFormatName = "Nx";
                extraFormatName += "Prod";
            }
            else if (isProdWithLog)
            {
                devkitFormatName = "Nx";
                extraFormatName += "ProdLog";
            }
            returnPath = returnPath.Replace("CONV1", devkitFormatName);
            returnPath = returnPath.Replace("CONV2", keyFormatName);
            returnPath = returnPath.Replace("CONV3", extraFormatName);
            return FileCheck(returnPath) ? returnPath : null;
        }
    }
#endif
}
