﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.IO;
using System.Diagnostics;
using Nintendo.InitializeSdev;
using System.Text.RegularExpressions;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    // MEMO: ここでは旧セーフモード(NX Addon 0.9.0以前)の処理を扱います
    //       新セーフモードの処理については EdevInitializerSubSafemode.cs に記述してあります

    partial class EdevInitializer
    {
        private const string RecoveryWriterUsbSuccessStringOld = "[SUCCESS]";
        private const string ConvertedRecoveryWriterUsbSuccessStringOld = "[INFO](log)";

        private Dictionary<TargetSpecifier.EdevVersion, string> edevVersionDictionaryOld = new Dictionary<TargetSpecifier.EdevVersion, string>()
        {
            {TargetSpecifier.EdevVersion.EDEV_EP_2_1, "EP2-1"},
            {TargetSpecifier.EdevVersion.EDEV_EP_2_2, "EP2-2"},
            {TargetSpecifier.EdevVersion.EDEV_NONE, ""},
            {TargetSpecifier.EdevVersion.EDEV_ERROR, ""},
        };

        private ExitStatus ExecuteInSafeModeOld(out TargetSpecifier.EdevVersion edevVersion, TargetInnerClass targetKey, ref ProcessingProgress progress)
        {
            ExitStatus eStatus = ExitStatus.Failure;
            // MEMO: この関数全体での progress 増分は 35 の想定
            int safemodeStartProgress = progress.GetProgress();

            string recoveryWriterUsbPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_RecoveryWriterUsb_Old);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(recoveryWriterUsbPath), recoveryWriterUsbPath) == false)
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
                return ExitStatus.Failure;
            }

            eStatus = CheckSafeModeSerialNumberOld(targetKey.Value, recoveryWriterUsbPath);
            if (eStatus != ExitStatus.Success)
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
                return ExitStatus.Failure;
            }

            progress.SetProgress(safemodeStartProgress + 10);

#if true
            eStatus = CheckEdevVersionOld(out edevVersion, targetKey, recoveryWriterUsbPath);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }
#else
            edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_1;
#endif
            Nintendo.InitializeSdev.TargetSpecifier.SdevVersion convertedSdevVerson = ConvertEdevToSdevVersion(edevVersion);
            firmwareResource.SetSdevVersion(convertedSdevVerson);

            progress.SetProgress(safemodeStartProgress + 15);

            string qspiBootImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_QspiBootImage);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(qspiBootImagePath), qspiBootImagePath) == false)
            {
                return ExitStatus.Failure;
            }

            eStatus = UpdateRecoveryBootImageOld(targetKey, qspiBootImagePath, recoveryWriterUsbPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update Recovery boot image failed.");
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        private ExitStatus CheckSafeModeSerialNumberOld(string targetKey, string recoveryWriterUsbPath)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            string targetString;
            if (targetKey.ToUpper() != "USB")
            {
                SerialNumber serial = new SerialNumber(targetKey);
                if (serial.GetString() == null)
                {
                    return ExitStatus.Failure;
                }
                targetString = serial.GetStringWithCheckDigit();
            }
            else
            {
                targetString = "usb";
            }

            // call RecoveryWriterUsb
            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "GetSerialNumber" };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceivedOld);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceededOld())
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Target EDEV '{0}' is not found or failed to enter the safe mode.", targetKey);
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, ConstData.GetEdevErrorInfomationString());
                goto quit;
            }
            if (targetKey.ToUpper() != "USB")
            {
                if (!ParseRecoveryWriterUsbSerialNumberOutputOld(targetString))
                {
                    goto quit;
                }
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        private ExitStatus CheckEdevVersionOld(out TargetSpecifier.EdevVersion edevVersion, TargetInnerClass targetKey, string recoveryWriterUsbPath)
        {
            ExitStatus eStatus = ExitStatus.Failure;
            edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;

            // call RecoveryWriterUsb
            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "GetConfigurationId1" };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceivedOld);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceededOld())
            {
                goto quit;
            }
            if (!ParseConfigurationId1OutputOld(out edevVersion))
            {
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        private ExitStatus UpdateRecoveryBootImageOld(TargetInnerClass targetKey, string qspiBootImagePath, string recoveryWriterUsbPath)
        {
            LOG.LogLine("Update Recovery boot image.");

            ExitStatus eStatus = ExitStatus.Failure;

            // call RecoveryWriterUsb

            string qspiBootImageDirectory = Path.GetDirectoryName(qspiBootImagePath);
            string qspiBootImageFile = Path.GetFileName(qspiBootImagePath);
            string currentDirectory = Directory.GetCurrentDirectory();

            Directory.SetCurrentDirectory(qspiBootImageDirectory);

            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "SendImage", qspiBootImageFile };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceivedOld);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceededOld())
            {
                goto quit;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 30 sec.");
            Thread.Sleep(30000);

            eStatus = ExitStatus.Success;

        quit:
            Directory.SetCurrentDirectory(currentDirectory);
            return eStatus;
        }

        private bool ParseRecoveryWriterUsbSerialNumberOutputOld(string serialNumber)
        {
            if (ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessStringOld))
            {
                Regex r = new Regex(@"[0-9A-Za-z]+");
                Match m = r.Match(ProcessOutputBuffer.Substring(RecoveryWriterUsbSuccessStringOld.Length));
                if (m.Success)
                {
                    string serialString = m.Groups[0].Value;
                    if (serialString == serialNumber)
                    {
                        return true;
                    }
                    else
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Safe mode EDEV's serial number is {0}", serialString);
                    }
                }
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Target EDEV '{0}' is not found.", serialNumber);
                return false;
            }
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Target EDEV '{0}' is not found or failed to enter the safe mode.", serialNumber);
            return false;
        }

        private bool ParseConfigurationId1OutputOld(out TargetSpecifier.EdevVersion edevVersion)
        {
            if (ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessStringOld))
            {
                int stringStart = ProcessOutputBuffer.IndexOf("EDEV_");
                if (stringStart >= 0)
                {
                    if (ProcessOutputBuffer.Substring(stringStart).Contains("EDEV_00_02_00_01"))
                    {
                        edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_1;
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV version is {0}", edevVersionDictionaryOld[edevVersion]);
                        return true;
                    }
                    else if (ProcessOutputBuffer.Substring(stringStart).Contains("EDEV_00_02_00_02"))
                    {
                        edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_2;
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV version is {0}", edevVersionDictionaryOld[edevVersion]);
                        return true;
                    }
                }
            }
            edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "EDEV version check failed.");
            return false;
        }

        private void RecoveryWriterUsbOutputDataReceivedOld(object sender, DataReceivedEventArgs e)
        {
            if (Argument.Verbose)
            {
                if (e.Data != null && e.Data.Contains(RecoveryWriterUsbSuccessStringOld))
                {
                    Console.WriteLine(ConvertedRecoveryWriterUsbSuccessStringOld + e.Data.Substring(RecoveryWriterUsbSuccessStringOld.Length));
                }
                else if (e.Data != null)
                {
                    Console.WriteLine(e.Data);
                }
            }
            ProcessOutputBuffer += e.Data;
        }

        private bool CheckRecoveryWriterUsbSucceededOld()
        {
            return ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessStringOld);
        }
    }
}
