﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.ControlTarget;
using Nintendo.InitializeSdev;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    public partial class EdevInitializer
    {
        private const int RunOnTargetInterval = 4;
        private const int ResetInterval = 20;

        private string GetVersionString()
        {
            if (Argument.targetFirmwareVersion == null)
            {
                return string.Empty;
            }
            else
            {
                return Argument.targetFirmwareVersion;
            }
        }

        private void LogWarningIfProdFirmware()
        {
            string versionString = GetVersionString();

            bool isProd = FirmwareResourceAccessor.CheckProdVersion(versionString);
            bool isProdWithLog = FirmwareResourceAccessor.CheckProdWithLogVersion(versionString);
            if(isProd || isProdWithLog)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "To use \"{0}\" type firmware, you must start devkit in safe mode manually.",
                                                isProd ? "+ Prod" : "+ ProdWithLog");
            }
        }

        private ExitStatus UsbAccessCheck()
        {
            using (UsbDeviceAccessor usbAccess = new UsbDeviceAccessor())
            {
                int deviceNum = usbAccess.GetDeviceNum();
                if (deviceNum == 0)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "EDEV is not connected. Please check USB connection and EDEV is powered on.");
                    LogWarningIfProdFirmware();
                    return ExitStatus.Failure;
                }
                else if (deviceNum < 0)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "NX devkits enumeration via USB failed.");
                    return ExitStatus.Failure;
                }
            }
            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManager()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager.");
            if (ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");
            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManagerForLast()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager .");
            if (ProcessAccessor.StopTargetManagerForLast() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private ExitStatus ValidateEdevVersion(FirmwareResourceSpecifier firmwareResource)
        {
            // Prod もしくは ProdWithLog の場合はK5鍵の機種のみ利用可能
            if (!firmwareResource.ValidateSetVersion())
            {
                return ExitStatus.Failure;
            }

            return ExitStatus.Success;
        }

        private ExitStatus ExecuteBootSafeMode(ref ProcessingProgress progress, TargetInnerClass targetKey, string targetManagerPath, string oldTargetManagerPath)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Boot safemode.");
            var result = BootSafeMode.Execute(targetKey.Value, targetManagerPath, oldTargetManagerPath, Argument.Verbose);
            if (result == BootSafeModeResult.Failure)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to boot safemode.");
                return ExitStatus.Failure;
            }
            else if (result == BootSafeModeResult.NotFound)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Skip booting safemode because {0} is not found and may have been already booted in safe mode.", targetKey);
            }
            else
            {
                // セーフモードに入ったので待機する
                if(progress != null)
                {
                    int prg = progress.GetProgress() + 10;
                    progress.SetProgress(prg);
                }

                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Wait 30 sec.");
                System.Threading.Thread.Sleep(30000);
            }
            return ExitStatus.Success;
        }

        private ExitStatus GetEdevVersionAndUpdateBootImageInSafeMode(out TargetSpecifier.EdevVersion edevVersion, bool isNewSafeMode, TargetInnerClass targetKey, ref ProcessingProgress progress)
        {
            if (isNewSafeMode)
            {
                return ExecuteInSafeMode(out edevVersion, targetKey, ref progress);
            }
            else
            {
                return ExecuteInSafeModeOld(out edevVersion, targetKey, ref progress);
            }
        }

        private ExitStatus GetEdevVersionInNormalMode(out TargetSpecifier.EdevVersion edevVersion, TargetInnerClass targetKey, string targetManagerPath, ref ProcessingProgress progress)
        {
            ExitStatus eStatus;
            string devkitHardwareType;

            LOG.LogLine("The safe mode may be preparing, so try to get configuration on normal mode.");

            eStatus = TargetManagerAccess.GetDevkitHardwareType(targetManagerPath, targetKey, out devkitHardwareType);
            if (eStatus != ExitStatus.Success)
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
                return eStatus;
            }
            if (devkitHardwareType == "EDEV")   // "EDEV" とは EDEV MP のことである
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_MP;
                firmwareResource.SetSdevVersion(InitializeSdev.TargetSpecifier.SdevVersion.EDEV_MP);
                return ExitStatus.Success;
            }

            edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
            return ExitStatus.Failure;
        }

#if SYSTEM_UPDATE_IMAGE
        private ExitStatus UpdateSafeMode(string safeModeUpdaterPath, TargetInnerClass targetKey, string safeModeImagePath, string targetManagerPath)
#else
        private ExitStatus UpdateSafeMode(string safeModeUpdaterPath, TargetInnerClass targetKey, string targetManagerPath)
#endif
        {
            // MEMO: SafeModeUpdater は SystemUpdater と同様の実行手順を取る

            LOG.LogLine("Update SafeMode partition.");

            // MEMO: RunOnTarget API 呼び出しを複数回実施すると Trace も重畳して出力されてしまうため
            //       こちらがわの呼び出し時はログ出力を抑制するようにした
            //       他の対策方法としては exe 呼び出しに置き換える方法がある
#if SYSTEM_UPDATE_IMAGE
            return RunSystemUpdater(safeModeUpdaterPath, targetKey, safeModeImagePath, targetManagerPath, 5, true);
#else
            return RunSystemUpdater(safeModeUpdaterPath, targetKey, targetManagerPath, 5, true);
#endif
        }

#if SYSTEM_UPDATE_IMAGE
        private ExitStatus UpdateSystemImage(string systemUpdaterPath, TargetInnerClass targetKey, string systemImagePath, string targetManagerPath)
#else
        private ExitStatus UpdateSystemImage(string systemUpdaterPath, TargetInnerClass targetKey, string targetManagerPath)
#endif
        {
            LOG.LogLine("Update initial image.");

#if SYSTEM_UPDATE_IMAGE
            return RunSystemUpdater(systemUpdaterPath, targetKey, systemImagePath, targetManagerPath, 20);
#else
            return RunSystemUpdater(systemUpdaterPath, targetKey, targetManagerPath, 20);
#endif
        }

#if SYSTEM_UPDATE_IMAGE
        private ExitStatus RunSystemUpdater(string systemUpdaterPath, TargetInnerClass targetKey, string systemImagePath, string targetManagerPath, int waitSeconds, bool forceUnVerbose = false)
#else
        private ExitStatus RunSystemUpdater(string systemUpdaterPath, TargetInnerClass targetKey, string targetManagerPath, int waitSeconds, bool forceUnVerbose = false)
#endif
        {
            // call RunOnTarget
            List<string> args1 = new List<string>() {"--target", GetTargetStringForRunOnTarget(targetKey.Value),
                                "--failure-timeout", DefaultTimeoutSecond.ToString(),
                                "--pattern-success-exit", "Succeeded initializing the system.", "--extra-targetmanager-dir", targetManagerPath};
            List<string> args2 = new List<string>() { systemUpdaterPath };
#if SYSTEM_UPDATE_IMAGE
            List<string> args3 = new List<string>() { "--", "--input", systemImagePath };
#else
            List<string> args3 = new List<string>();
#endif
            System.Diagnostics.Stopwatch sw = new System.Diagnostics.Stopwatch();

            if (Argument.Verbose && !forceUnVerbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
            argsList.AddRange(args2);
            argsList.AddRange(args3);

            ProcessAccessor.DumpProcessArgumentList("RunOnTarget", argsList);

            sw.Start();
            int retVar = Nintendo.RunOnTarget.CommandInterface.Main(argsList.ToArray());
            sw.Stop();

            // MEMO: put dummy line feed for RunOnTarget log
            Console.WriteLine(string.Empty);

            if (retVar != 0)
            {
                // タイムアウトした場合はFuseにより起動できなくなっている可能性がある
                if ((int)sw.Elapsed.TotalSeconds > DefaultTimeoutSecond + RunOnTargetInterval)    // RunOnTarget の実行時間は timeout より長い
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "If you are about to downgrade firmware, please check downgrade is available.");
                }
            }

            if(waitSeconds > 0)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait {0} sec.", waitSeconds);
                Thread.Sleep(waitSeconds * 1000);
            }

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private string GetTargetStringForRunOnTarget(string targetKeyString)
        {
            if (targetKeyString.ToUpper() != "USB")
            {
                SerialNumber serial = new SerialNumber(targetKeyString);
                if (serial.GetString() == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "GetTargetStringForRunOnTarget: The serial number is invalid. ({0})", targetKeyString);
                    return "";
                }
                return serial.GetStringWithCheckDigit();
            }
            else
            {
                return "usb";
            }
        }

        private ExitStatus ExecutePluginsBeforeInitialize()
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePre, false);
            return bRet ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus ExecutePluginsAfterInitialize(TargetInnerClass targetKey, TargetSpecifier.EdevVersion edevVersion)
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePost, false, targetKey, TargetSpecifier.GetDevkitVersion(edevVersion));
            return bRet ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus RestartTarget(string targetKey, string targetManagerPath)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reset target.");
            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerPath);
                tmapiAccessor.EnsureStart();

                using (var target_ = tmapiAccessor.GetTarget(tmapiAccessor.FindTarget(targetKey)))
                {
                    tmapiAccessor.RebootTarget(target_);
                }

            }

            Thread.Sleep(10000);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reset target finished.");

            return ExitStatus.Success;
        }

        private ExitStatus ActivateTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return TargetManagerAccess.ActivateTarget(targetManagerPath, targetKey);
        }

        private ExitStatus DisconnectTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return TargetManagerAccess.DisconnectTarget(targetManagerPath, targetKey);
        }

        public static Nintendo.InitializeSdev.TargetSpecifier.SdevVersion ConvertEdevToSdevVersion(TargetSpecifier.EdevVersion edevVersion)
        {
            switch (edevVersion)
            {
                case TargetSpecifier.EdevVersion.EDEV_EP_2_1:
                    return InitializeSdev.TargetSpecifier.SdevVersion.EDEV_EP_2_1;
                case TargetSpecifier.EdevVersion.EDEV_EP_2_2:
                    return InitializeSdev.TargetSpecifier.SdevVersion.EDEV_EP_2_2;
                case TargetSpecifier.EdevVersion.EDEV_MP:
                    return InitializeSdev.TargetSpecifier.SdevVersion.EDEV_MP;
                default:
                    return InitializeSdev.TargetSpecifier.SdevVersion.SDEV_ERROR;
            }
        }
    }
}
