﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
//using Nintendo.ControlTarget;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    public class CommandInterface
    {
        public static int Main(string[] rawArgs)
        {
            DateTime startTime = DateTime.Now;

            // 実処理
            int returnValue = MainCore(rawArgs);

            // 統計情報の事後収集
            CollectPostData(ref rawArgs, returnValue, startTime);

            return returnValue;
        }

        private static void CollectPostData(ref string[] rawArgs, int returnValue, DateTime startTime)
        {
            try
            {
                // 統計情報の取得 (ローカルメトリー)
                if (CollectMeasurementData.Collect(ref rawArgs, returnValue, startTime) == false)
                {
                    // 失敗した場合は利用者からはわかりにくい文字列をログとして出力する
                    Console.Error.WriteLine("xxxvx");
                }
            }
            catch (Exception)
            {
                // 失敗した場合は利用者からはわかりにくい文字列をログとして出力する
                Console.Error.WriteLine("xxxvy");
                Nintendo.InitializeSdev.CollectMeasurementMutex.GetInstance().KillMutex();
            }
        }

        // ===========================================================================================
        //   事実上の Main 関数
        // ===========================================================================================
        private static int MainCore(string[] rawArgs)
        {
            try
            {
                EdevInitializeCommandArgument parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<EdevInitializeCommandArgument>(rawArgs, out parsed))
                {
                    return 1;
                }

                if (parsed.Verbose)
                {
                    LOG.SetLevel(LOG_LEVEL.LOG_INFO2);
                }

                if (parsed.ndiPipeName != null && parsed.ndiLocale != null)
                {
                    Nintendo.InitializeSdev.ProgressWriterManager.UseNdiProgressWriter(parsed.ndiPipeName);
                }

                if (parsed.displayAvailableVersion)
                {
                    return DisplayAvaialbeVersion(parsed);
                }
                else if (parsed.listAvailableTargets)
                {
                    return ListAvailableTargets(parsed);
                }
                else if (parsed.bootSaveMode)
                {
                    return ExecuteBootSafeMode(parsed);
                }

                string name = GetBaseName();
                if (name == "InitializeEdev" || name == "InitializeEdevWin")
                {
                    Console.WriteLine("========== InitializeEdev ==========");
                }

                if (parsed.targetConnectedUsb)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "--target-connected-usb is not supported already.");
                    return 1;
                }
                else if (parsed.targetOldConnectedUsb)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "--target-old-usb is not supported already.");
                    return 1;
                }

                return EdevInitialize(parsed);

            }
            catch (System.IO.FileNotFoundException exception)
            {
                PrintFileNotFoundException(exception);
                Nintendo.InitializeSdev.ProgressWriterManager.Writer.Write(-1, "File not found", exception.FileName);
                return 1;
            }
            catch (Exception exception)
            {
                PrintException(exception);
                Nintendo.InitializeSdev.ProgressWriterManager.Writer.Write(-1, "Error Occured", exception.Message);
                return 1;
            }
        }

        private static string GetBaseName()
        {
            string name = Process.GetCurrentProcess().ProcessName;
            if (name.Contains('.'))
            {
                return name.Substring(0, name.IndexOf('.'));
            }
            else
            {
                return name;
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] Caught exception.");
            Console.Error.WriteLine(string.Format("[ERROR]   Message   : {0}", exception.Message));
            Console.Error.WriteLine(string.Format("[ERROR]   Source    : {0}", exception.Source));
            Console.Error.WriteLine(string.Format("[ERROR]   TargetSite: {0}", exception.TargetSite.ToString()));
            Console.Error.WriteLine(string.Format("[ERROR]   StackTrace: {0}", exception.StackTrace));

            // Trace.WriteLine(string.Format("Message   : {0}", exception.Message));
            // Trace.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));

            Console.Error.WriteLine(string.Format("[ERROR]   {0}", ConstData.GetEdevErrorInfomationString()));
        }

        public static void PrintFileNotFoundException(System.IO.FileNotFoundException exception)
        {
            Console.Error.WriteLine(string.Format("[ERROR] Dependent file is not found ({0}).", exception.FileName));
        }

        public static int EdevInitialize(EdevInitializeCommandArgument args)
        {
            return new EdevInitializer().Execute(args) == ExitStatus.Success ? 0 : 1;
        }

        public static int DisplayAvaialbeVersion(EdevInitializeCommandArgument args)
        {
            return new DisplayInformations().AvailableFirmwareVersions(args) == ExitStatus.Success ? 0 : 1;
        }

        public static int ListAvailableTargets(EdevInitializeCommandArgument args)
        {
            return new DisplayInformations().ListAvailableTargets(args) == ExitStatus.Success ? 0 : 1;
        }

        public static int ExecuteBootSafeMode(EdevInitializeCommandArgument args)
        {
            return new EdevInitializer().ExecuteBootSafeMode(args) == ExitStatus.Success ? 0 : 1;
        }
    }
}
