﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <tm.h>
#include <tm_private.h>
#include <msclr/marshal.h>
#include <vector>
#include "TmapiException.h"

using namespace System;
using namespace System::Collections::Generic;

#define TMAPI_DO(exp) \
    do { \
        nn::tm::Error tmapi_do_temporary_error = (exp); \
        if (tmapi_do_temporary_error != nn::tm::Error_Ok) \
        { \
            nn::tm::ErrorText errorText = {0}; \
            ::nn::tm::GetErrorTextEn(&errorText, tmapi_do_temporary_error); \
            throw gcnew Nintendo::ControlTarget::Tmapi::TmapiException(static_cast<Nintendo::ControlTarget::Tmapi::TmapiError>(tmapi_do_temporary_error), String::Format("TMAPI Error: {0}", ::msclr::interop::marshal_as<String^>(errorText.text))); \
        } \
    } while(0)

namespace Nintendo {
namespace ControlTarget {
namespace Tmapi {

    struct target_info
    {
        enum
        {
            // The maximum lengths, including '\0' terminator.
            IpAddressLength    = 256,
            NameLength         = 64,
            SerialNumberLength = 24,
        };

        char                    m_IPAddress[IpAddressLength];
        char                    m_Name[NameLength];
        char                    m_TMAVersion[32];
        char                    m_ConnectionType[32];
        char                    m_HardwareType[32];
        char                    m_SerialNumber[SerialNumberLength];
        char                    m_MacAddress[32];
        char                    m_ConnectStatus[32];
        nn::tm::DiscoveryHandle m_DiscoveryHandle;
    };

    public enum class DumpType
    {
        Quick = nn::tm::DumpType_Quick,
        Full = nn::tm::DumpType_Full,
    };

    public ref class TargetHandle
    {
    private:
        nn::tm::TargetHandle* mp_Handle;
    public:

        TargetHandle(nn::tm::TargetHandle handle);
        const nn::tm::TargetHandle GetHandle() { return *mp_Handle; }
        bool Equals(TargetHandle^ rhs) { return this->mp_Handle->handle == rhs->mp_Handle->handle; }
        ~TargetHandle();
        !TargetHandle();
    };

    public ref class DiscoveryHandle
    {
    public:
        nn::tm::TargetHandle* mp_Handle;

        DiscoveryHandle(nn::tm::DiscoveryHandle handle);
        ~DiscoveryHandle();
        !DiscoveryHandle();
    };

    public ref class TargetInfo
    {
    public:
        TargetInfo(target_info *pTargetInfo);
        TargetInfo(nn::tm::DiscoveryHandle discoveryHandle);
        ~TargetInfo();
        target_info& GetRawInfo();

        String^ GetIpAddress();
        String^ GetName();
        String^ GetTmaVersion();
        String^ GetMacAddress();
        String^ GetConnectionType();
        String^ GetHardwareType();
        String^ GetSerialNumber();

        String^ ToString() override;

    private:
        !TargetInfo();
        target_info *m_pTargetInfo;
    };

    public ref class TargetAccessor
    {
    public:
        TargetAccessor(nn::tm::TargetHandle targetHandle);
        TargetAccessor::TargetAccessor(TargetHandle^ targetHandle);
        void EnsureConnect();
        void EnsureDisconnect();
        bool IsConnected();
        bool IsSleeping();
        void PowerOff();
        void PowerOn();
        void Reset();
        void Reboot();
        void RebootHard();
        void Connect();
        void Disconnect();
        void SetDefault();
        void SetName(String^ name);
        String^ TakeScreenshot(String^ directory, String^ fileName);
        void CreateTargetProgramDump(String^ filePath, DumpType dumpType);
        void SetTargetWorkingDirectory(String^ workingDirectory);
        String^ GetTargetWorkingDirectory();
        ~TargetAccessor() {};
    protected:
        !TargetAccessor() {};

    private:
        TargetHandle^ m_TargetHandle;
    };

    public ref class RegisteredTargetInfo
    {
    public:
        RegisteredTargetInfo(
            TargetHandle^ targetHandle,
            String^ name,
            String^ hardwareType,
            String^ connectionType,
            String^ tmaVersion,
            String^ serialNumber,
            String^ ipAddress,
            String^ macAddress,
            String^ status);

        TargetHandle^ GetTargetHandle();
        String^ GetName();
        String^ GetHardwareType();
        String^ GetConnectionType();
        String^ GetTmaVersion();
        String^ GetSerialNumber();
        String^ GetIpAddress();
        String^ GetMacAddress();
        String^ GetStatus();

        String^ ToString() override;

    private:
        TargetHandle^ m_TargetHandle;
        String^ m_Name;
        String^ m_HardwareType;
        String^ m_ConnectionType;
        String^ m_TmaVersion;
        String^ m_SerialNumber;
        String^ m_IpAddress;
        String^ m_MacAddress;
        String^ m_Status;
    };

    public ref class TargetManagerAccessor
    {
    public:
        TargetManagerAccessor();

        int32_t GetTargetCount();
        int32_t GetHtcsControlPort();

        bool HasDefaultTarget();
        TargetHandle^ GetDefaultTargetHandle();
        RegisteredTargetInfo^ GetDefaultTargetInfo();
        TargetAccessor^ GetDefaultTarget();
        TargetAccessor^ GetTarget(TargetHandle^ targetHandle);

        List<TargetInfo^>^ DetectTargets(TimeSpan timeOut);
        List<TargetHandle^>^ ListTargetHandles();
        List<RegisteredTargetInfo^>^ ListTargets();

        void RegisterTarget(TargetInfo^ info);
        void RegisterTarget( String^ ip );
        void UnregisterTarget(TargetHandle^ targetHandle);
        void UnregisterAllTargets();
        void EnableCradle(TargetHandle^ targetHandle);
        void DisableCradle(TargetHandle^ targetHandle);

        ~TargetManagerAccessor();

    protected:
        String^ GetTargetName(nn::tm::TargetHandle targetHandle);
        String^ GetTargetTMAVersion(nn::tm::TargetHandle targetHandle);
        String^ GetTargetIPAddress(nn::tm::TargetHandle targetHandle);
        String^ GetTargetConnectionType(nn::tm::TargetHandle targetHandle);
        String^ GetTargetHardwareType(nn::tm::TargetHandle targetHandle);
        String^ GetTargetSerialNumber(nn::tm::TargetHandle targetHandle);
        String^ GetTargetMacAddress( nn::tm::TargetHandle targetHandle );
        String^ GetTargetStatus(nn::tm::TargetHandle targetHandle);
        !TargetManagerAccessor();
    };
}
}
}
