﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Web.Script.Serialization;
using System.Diagnostics;
using Nintendo.ControlTarget.ConsoleShell;
using Nintendo.ControlTarget.Tmapi;
using Nintendo.Log;
using System.Text.RegularExpressions;

namespace Nintendo.ControlTarget
{
    public interface TargetCommunicationInterface : IDisposable
    {
        ISourceBlock<string> ProgramLog { get; }
        ISourceBlock<string> TargetLog { get; }
        ConsoleShellAccessor Shell { get; }
        CancellationTokenSource CancelTokenSource { get; }
    }

    public class TargetCommunicationOption
    {
        public string Port { get; set; }
        public IPEndPoint ShellAddress { get; set; }
        public bool SkipMonitoringSerial { get; set; }
        public bool SuppressAutoUBoot { get; set; }
        public bool SuppressAutoHtcDaemon { get; set; }
        public bool SuppressAutoKill { get; set; }
        public string UBootCommand { get; set; }
        public bool UseHostBridgeUart { get; set; }
        public IPEndPoint HostBridgeUartAddress { get; set; }
        public bool IsAllowedToUseSerial
        {
            get
            {
                return !SkipMonitoringSerial;
            }
        }
        public bool IsAllowedToUseUboot
        {
            get
            {
                return IsAllowedToUseSerial && !SuppressAutoUBoot;
            }
        }
        public bool IsAllowedToUseHtcDaemon
        {
            get
            {
                return !SuppressAutoHtcDaemon;
            }
        }
    }

    public class BasicTargetCommunication : TargetCommunicationInterface, IDisposable
    {
        public BasicTargetCommunication(TargetManagerAccessor targetManager, string targetName, string filter, int connectTimeout)
        {
            CancelTokenSource = new CancellationTokenSource();
            var targetInfo = targetManager.FindTargetInfo(targetName);

            Target = targetManager.ConnectTarget(targetInfo.GetTargetHandle(), connectTimeout);

            targetInfo = targetManager.FindRegisteredTarget(targetInfo.GetTargetHandle());

            var isLegacyPortUsed = false;
            IPEndPoint logPort;
            try
            {
                logPort = targetManager.FindService("@JsonLog", targetInfo);
            }
            catch (HtcServiceNotFound)
            {
                logPort = targetManager.FindService("iywys@$Log", targetInfo);
                isLegacyPortUsed = true;
            }

            BroadCastOutput = new BroadcastBlock<string>((s) => { return s; });
            TargetLog = BroadCastOutput;
            ProgramLog = BroadCastOutput;

            HostBridgeAccessor = new HostBridgeAccessor(logPort);
            if (isLegacyPortUsed)
            {
                Unlinker += DataflowUtility.LinkBlock("HostBridgeAccessor.ReadPort->BroadCastOutput", HostBridgeAccessor.ReadPort, BroadCastOutput).Dispose;
            }
            else
            {
                var extractJsonBlock = CreateExtractJsonBlock();
                var parseJsonLogBlock = CreateParseJsonLogBlock();
                var filterBlock = CreateFilterBlock(filter);
                var stringizeLogBlock = CreateStringizeLogBlock();

                Unlinker += DataflowUtility.LinkBlock("HostBridgeAccessor.ReadPort->extractJsonBlock", HostBridgeAccessor.RawReadPort, extractJsonBlock).Dispose;
                Unlinker += DataflowUtility.LinkBlock("extractJsonBlock->parseJsonLogBlock", extractJsonBlock, parseJsonLogBlock).Dispose;
                Unlinker += DataflowUtility.LinkBlock("parseJsonLogBlock->filterBlock", parseJsonLogBlock, filterBlock).Dispose;
                Unlinker += DataflowUtility.LinkBlock("filterBlock->stringizeLogBlock", filterBlock, stringizeLogBlock).Dispose;
                Unlinker += DataflowUtility.LinkBlock("stringizeLogBlock->BroadCastOutput", stringizeLogBlock, BroadCastOutput).Dispose;
            }

            ConnectionPollingTask = Task.Run(() => {
                try
                {
                    while (true)
                    {
                        if (CancelTokenSource.IsCancellationRequested)
                        {
                            return;
                        }

                        // ここは本来はセッションの切断を拾うべき
                        // 短時間での切断と接続が入ると処理できない
                        if (!Target.IsConnected() && !Target.IsSleeping())
                        {
                            CancelTokenSource.Cancel();
                            return;
                        }

                        Task.Delay(1000).Wait();
                    }

                }
                catch(Exception e)
                {
                    Console.Error.WriteLine($"[ERROR] Connection Error at ConnectionPollingTask. Message: '{e.Message}'");
                    CancelTokenSource.Cancel();
                }
            });

            IPEndPoint shellPort;
            try
            {
                shellPort = targetManager.FindService("iywys@$csForRunnerTools", targetInfo);
            }
            catch (HtcServiceNotFound)
            {
                try
                {
                    shellPort = targetManager.FindService("@csForRunnerTools", targetInfo);
                }
                catch (HtcServiceNotFound)
                {
                    throw new Exception("Found no shell service.");
                }
            }
            Shell = new ConsoleShellAccessor(new ConsoleShellMessenger(shellPort));
        }

        private IPropagatorBlock<string, string> CreateExtractJsonBlock()
        {
            var sb = new StringBuilder();
            var inBlock = new BufferBlock<string>();
            var outBlock = new BufferBlock<string>();
            var actionBlock = new ActionBlock<string>(input =>
            {
                sb.Append(input);
                for (;;)
                {
                    var text = sb.ToString();

                    var newLineIndex = text.IndexOf(Environment.NewLine);
                    if (newLineIndex == -1)
                    {
                        break;
                    }

                    outBlock.Post(text.Substring(0, newLineIndex));
                    sb.Remove(0, (newLineIndex + Environment.NewLine.Length));
                }
            });
            Unlinker += inBlock.LinkTo(actionBlock).Dispose;
            return DataflowBlock.Encapsulate(inBlock, outBlock);
        }

        private IPropagatorBlock<string, Dictionary<string, object>> CreateParseJsonLogBlock()
        {
            return new TransformBlock<string, Dictionary<string, object>>(input =>
            {
                var serializer = new JavaScriptSerializer();
                return serializer.Deserialize<Dictionary<string, object>>(input);
            });
        }

        private IPropagatorBlock<Dictionary<string, object>, Dictionary<string, object>> CreateFilterBlock(string filter)
        {
            var inBlock = new BufferBlock<Dictionary<string, object>>();
            var outBlock = new BufferBlock<Dictionary<string, object>>();
            var actionBlock = new ActionBlock<Dictionary<string, object>>(input =>
            {
                if (input.ContainsKey("LogSessionEnd") || LogFilter.Evaluate(filter, input))
                {
                    outBlock.Post(input);
                }
            });
            Unlinker += inBlock.LinkTo(actionBlock).Dispose;
            return DataflowBlock.Encapsulate(inBlock, outBlock);
        }

        private IPropagatorBlock<Dictionary<string, object>, string> CreateStringizeLogBlock()
        {
            var inBlock = new BufferBlock<Dictionary<string, object>>();
            var outBlock = new BufferBlock<string>();
            var actionBlock = new ActionBlock<Dictionary<string, object>>(input =>
            {
                if (input.ContainsKey("TextLog"))
                {
                    outBlock.Post(StripSgrCommand((string)input["TextLog"]));
                }
                if (input.ContainsKey("LogSessionEnd"))
                {
                    outBlock.Post("\0\0\0\0");
                }
            });
            Unlinker += inBlock.LinkTo(actionBlock).Dispose;
            return DataflowBlock.Encapsulate(inBlock, outBlock);
        }

        private static string StripSgrCommand(string text)
        {
            return SgrCommandPattern.Replace(text, string.Empty);
        }

        public ISourceBlock<string> ProgramLog { get; private set; }
        public ISourceBlock<string> TargetLog { get; private set; }
        public ConsoleShellAccessor Shell { get; private set; }
        public CancellationTokenSource CancelTokenSource { get; private set; }
        public TargetAccessor Target { get; private set; }

        private Action Unlinker;
        private BroadcastBlock<string> BroadCastOutput;
        private HostBridgeAccessor HostBridgeAccessor;
        private Task ConnectionPollingTask;
        private static readonly Regex SgrCommandPattern = new Regex(@"\u001b\[(\d+(;\d+)*)?m", RegexOptions.Compiled);

        public void Dispose()
        {
            Trace.WriteLine("Dispose connection to target.");
            CancelTokenSource.Cancel();
            ConnectionPollingTask.Wait();
            ConnectionPollingTask.Dispose();
            Trace.WriteLine("Dispose connection monitor task.");
            Target.Dispose();
            Trace.WriteLine("Dispose target accessor.");
            Shell.Dispose();
            Trace.WriteLine("Dispose console shell port.");
            Unlinker?.Invoke();
            Trace.WriteLine("Dispose data stream.");
            HostBridgeAccessor.Dispose();
            Trace.WriteLine("Dispose host bridge accessor.");
        }
    }
}
