﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Xml.Linq;
using System.Threading.Tasks;

namespace Nintendo.ControlTarget
{
    public class HtcServiceNotFound : Exception
    {
        public HtcServiceNotFound(string message)
            : base(message)
        {
        }
    }

    public class HtcService
    {
        public HtcService(string targetName, string serviceName, IPEndPoint endPoint)
        {
            TargetName = targetName;
            ServiceName = serviceName;
            EndPoint = endPoint;
        }

        public string TargetName { get; set; }
        public string TargetId { get; set; }
        public string ServiceName { get; set; }
        public IPEndPoint EndPoint { get; set; }
    }

    public class HtcsTarget
    {
        public HtcsTarget(string peerType, string peerName, string id)
        {
            PeerType = peerType;
            PeerName = peerName;
            Id = id;
        }

        public string PeerType { get; set; }
        public string PeerName { get; set; }
        public string Id { get; set; }
    }

    public interface HtcsAccessor
    {
        bool IsRunning();
        void EnsureStart();
        void Start();
        string ReadPortMappingXml();
        List<HtcsTarget> ListTarget();
        List<HtcService> ListService();
        IPEndPoint FindService(string serviceName);
        IPEndPoint FindServiceWithRetry(string serviceName);
        bool ExistService(string serviceName);
    }

    public abstract class HtcsAccessorBase : HtcsAccessor
    {
        protected static readonly Int32 DefaultHtcsControlPort = 8003;
        public bool IsHtcsControlPort(Int32 port)
        {
            try
            {
                using (var socket = NetworkUtility.CreateConnectedSocket(new IPEndPoint(IPAddress.Loopback, port)))
                {
                    using (var socketStream = new NetworkStream(socket))
                    {
                        using (var socketReader = new StreamReader(socketStream))
                        {
                            Task<string> str = socketReader.ReadLineAsync();
                            if (str.Wait(3000))
                            {
                                return (str.Result.IndexOf("<HtcsInfo><TargetList>") == 0);
                            }
                        }
                    }
                }
            }
            catch
            {
                return false;
            }
            return false;
        }

        public virtual int GetHtcsControlPort()
        {
            // レジストリ HKEY_CURRENT_USER\Software\Nintendo\NX\Oasis\Ports\HtcsControl を見て、現在開かれているポートを返すことも可能
            return DefaultHtcsControlPort;
        }

        public virtual bool IsRunning()
        {
            foreach (System.Diagnostics.Process p in System.Diagnostics.Process.GetProcesses())
            {
                if (p.MainWindowTitle == "Nintendo Target Manager")
                {
                    return true;
                }
            }
            return false;
        }

        public virtual void EnsureStart()
        {
            if (!IsRunning())
            {
                Start();
            }
            else
            {
            }
        }

        public virtual void Start()
        {
            Trace.WriteLine("Start HtcsDaemon.");
            string args = "";

            if (StartTrayIcon)
            {
                args += "-tasktray ";
            }
            if (DisableTelemetryDialog)
            {
                args += "-NoDialog ";
            }
            if (StartForInitialize)
            {
                args += "-ForInitialize ";
            }

            if (args == "")
            {
                Process.Start(HtcsDaemonPath.FullName);
            }
            else
            {
                Process.Start(HtcsDaemonPath.FullName, args);
            }
        }

        public string ReadPortMappingXml()
        {
            var endPoint = this.PortOfHtcs;

            Trace.WriteLine($"Connect: {endPoint}");
            using (var socket = NetworkUtility.CreateConnectedSocket(endPoint))
            {
                using (var socketStream = new NetworkStream(socket))
                {
                    Trace.WriteLine($"Connected: {endPoint}");
                    using (var socketWriter = new StreamWriter(socketStream))
                    using (var socketReader = new StreamReader(socketStream))
                    {
                        // システムポートがないものが取得されるため読み捨てる
                        socketReader.ReadLine();

                        socketWriter.AutoFlush = true;
                        string xml = new XElement("RequestSystemPortMapping").ToString(SaveOptions.DisableFormatting);
                        socketWriter.WriteLine(xml);

                        return socketReader.ReadLine();
                    }
                }
            }
        }

        public List<HtcsTarget> ListTarget()
        {
            var xml = ReadPortMappingXml();
            using (var reader = new StringReader(xml))
            {
                var element = XElement.Load(reader);

                var services =
                    from x in element.Descendants("Target")
                    select
                        new HtcsTarget(
                            x.Element("PeerType").Value,
                            x.Element("HtcsPeerName").Value,
                            x.Element("Id").Value);

                return services.ToList();
            }
        }

        public List<HtcService> ListService()
        {
            var xml = ReadPortMappingXml();
            using (var reader = new StringReader(xml))
            {
                var element = XElement.Load(reader);

                var services =
                    from x in element.Descendants("PortMapItem")
                    select
                        new HtcService(
                            x.Element("HtcsPeerName").Value,
                            x.Element("HtcsPortName").Value,
                            new IPEndPoint(
                                IPAddress.Parse(x.Element("IPAddress").Value),
                                int.Parse(x.Element("TcpPortNumber").Value)));

                return services.ToList();
            }
        }

        public IPEndPoint FindService(string serviceName)
        {
            var foundService = (
                from service in ListService()
                where service.ServiceName == serviceName
                select service.EndPoint).FirstOrDefault();

            if (foundService == null)
            {
                throw new HtcServiceNotFound(string.Format("service(name={0}) is not found", serviceName));
            }

            return foundService;
        }

        public IPEndPoint FindService(string serviceName, string targetName)
        {
            var foundService = (
                from service in ListService()
                where service.ServiceName == serviceName && service.TargetName == targetName
                select service.EndPoint).FirstOrDefault();

            if (foundService == null)
            {
                throw new HtcServiceNotFound(string.Format("service(name={0}) is not found", serviceName));
            }

            return foundService;
        }

        public IPEndPoint FindServiceWithRetry(string serviceName)
        {
            return RetryUtility.Do(
                () =>
                {
                    return FindService(serviceName);
                },
                (e) =>
                {
                    if (e is Tmapi.TmapiException)
                    {
                        Tmapi.TmapiException te = (Tmapi.TmapiException)e;
                        if (te.GetTmapiError() == Tmapi.TmapiError.Error_IncompatibleTargetManager)
                        {
                            throw new RetryUtility.NoRetryException("TMAPI is imcompatible");
                        }
                    }
                    Trace.WriteLine($"FindServiceError: {e.Message}");
                },
                (es) =>
                {
                    throw new HtcServiceNotFound(string.Format("Found no shell service(name={0}).", serviceName.Replace(Utility.HtcsSystemPortPrefix, "Nintendo System ")));
                },
                15,
                TimeSpan.FromSeconds(1));
        }

        public IPEndPoint FindServiceWithRetry(string serviceName, string targetName)
        {
            return RetryUtility.Do(
                () =>
                {
                    return FindService(serviceName, targetName);
                },
                (e) =>
                {
                    if (e is Tmapi.TmapiException)
                    {
                        Tmapi.TmapiException te = (Tmapi.TmapiException)e;
                        if (te.GetTmapiError() == Tmapi.TmapiError.Error_IncompatibleTargetManager)
                        {
                            throw new RetryUtility.NoRetryException("TMAPI is imcompatible");
                        }
                    }
                    Trace.WriteLine($"FindServiceError: {e.Message}");
                },
                (es) =>
                {
                    throw new HtcServiceNotFound($"Found no shell service(name={serviceName.Replace(Utility.HtcsSystemPortPrefix, "Nintendo System ")}).");
                },
                15,
                TimeSpan.FromSeconds(1));
        }

        public bool ExistService(string serviceName)
        {
            var foundService = (
                from service in ListService()
                where service.ServiceName == serviceName
                select service.EndPoint).FirstOrDefault();

            if (foundService == null)
            {
                return false;
            }

            return true;
        }

        public virtual void SetStartTrayIcon(bool startTrayIcon)
        {
            StartTrayIcon = startTrayIcon;
            // トレイアイコン起動にする時はテレメトリー確認ダイアログも表示しないようにする
            DisableTelemetryDialog = startTrayIcon;
        }

        public virtual void SetStartForInitialize(bool startForInitialize)
        {
            StartForInitialize = startForInitialize;
        }

        public abstract FileInfo HtcsDaemonPath { get; }
        public abstract IPEndPoint PortOfHtcs { get; }

        protected bool StartTrayIcon = false;
        protected bool DisableTelemetryDialog = false;
        protected bool StartForInitialize = false;
    }
}
