﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading.Tasks;

namespace Nintendo.ControlTarget.ConsoleShell
{
    public abstract class IConsoleShellCommand
    {
        public string CommandName { get; protected set; }

        public IConsoleShellCommand(string name)
        {
            CommandName = name;
        }

        public byte[] GetCommand(ulong id)
        {
            using (var memoryWriter = new MemoryStream())
            {
                using (var binaryWriter = new BinaryWriter(memoryWriter))
                {
                    var body = GetCommandBody();

                    var header = GetCommandHeader(id, GetCommandId(), body.Count());

                    binaryWriter.Write(header);
                    binaryWriter.Write(body);
                }

                return memoryWriter.ToArray();
            }
        }

        public abstract int GetCommandId();

        public ConsoleShellRequest MakeRequest()
        {
            return new ConsoleShellRequest(this);
        }

        protected abstract void MakeCommandBody(BinaryWriter writer);

        private byte[] GetCommandBody()
        {
            using (var memoryWriter = new MemoryStream())
            {
                using (var binaryWriter = new BinaryWriter(memoryWriter))
                {
                    MakeCommandBody(binaryWriter);
                }

                return memoryWriter.ToArray();
            }
        }

        private byte[] GetCommandHeader(ulong id, int commandId, int bodySize)
        {
            using (var memoryWriter = new MemoryStream(0))
            {
                using (var binaryWriter = new BinaryWriter(memoryWriter))
                {
                    binaryWriter.Write(id);
                    binaryWriter.Write(commandId);
                    binaryWriter.Write(bodySize);
                }

                return memoryWriter.ToArray();
            }
        }

        public abstract string CommandExpression { get; }
    }

    public class ConsoleShellRequest
    {
        public static Random IdGenerator = new Random();
        public ulong RequestId { get; private set; }
        public byte[] RequestData { get; private set; }
        public IConsoleShellCommand Command { get; private set; }

        public ConsoleShellRequest(IConsoleShellCommand command)
        {
            RequestId = (ulong)IdGenerator.Next();
            Command = command;
            RequestData = Command.GetCommand(RequestId);
        }

        public string RequestExpression
        {
            get
            {
                return string.Format("Request(ReuestId: 0x{0,0:X8}, {1})", RequestId, Command.CommandExpression);
            }
        }
    }

    public class LaunchApplicationOption
    {
        public bool EnableJit { get; set; } = true;
        public bool EnableAslr { get; set; } = true;
        public string EnvPath { get; set; } = string.Empty;

        public UInt32 MakeFlag()
        {
            return (
                (EnableJit ? LaunchApplicationFlag.EnableJitDebug : 0) |
                (EnableAslr ? 0 : LaunchApplicationFlag.DisableAslr)
                );
        }
    }

    public class LaunchApplicationCommand : IConsoleShellCommand
    {
        public UInt64 ProgramId = 0xFF00000000000000;
        public UInt32 Flags = 0;
        public string Path;
        public string[] Arguments;
        public string EnvPath;

        public LaunchApplicationCommand(string path, string[] arguments, LaunchApplicationOption option) : base("LaunchApplication")
        {
            Flags = option.MakeFlag();
            Path = path;
            Arguments = arguments;
            EnvPath = string.IsNullOrEmpty(option.EnvPath) ? string.Empty : option.EnvPath;
        }

        private string Escape(string arg)
        {
            return String.Format("\"{0}\"", arg);
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
            var pathBytes = System.Text.Encoding.UTF8.GetBytes(Path).ToArray();
            Arguments = Arguments == null ? new string[0]: Arguments;
            var argumentBytes = (from argument in this.Arguments select System.Text.Encoding.UTF8.GetBytes(Escape(argument)).Concat(new byte[] { 0x20 })).SelectMany(v => v).Reverse().Skip(1).Reverse().ToArray();
            var envPathBytes = System.Text.Encoding.UTF8.GetBytes(EnvPath).ToArray();
            UInt32 pathSize = (UInt32)pathBytes.Count();
            UInt32 argumentSize = (UInt32)argumentBytes.Count();
            UInt32 envPathSize = (UInt32)envPathBytes.Count();

            writer.Write(ProgramId);
            writer.Write(pathSize);
            writer.Write(argumentSize);
            writer.Write(Flags);
            writer.Write(envPathSize);
            writer.Write(pathBytes);
            writer.Write(argumentBytes);
            writer.Write(envPathBytes);
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.LaunchApplication;
        }

        public override string CommandExpression
        {
            get
            {
                return $"LaunchApplication(path={Path}, flags={Flags:X8})";
            }
        }
    }

    public class LaunchInstalledApplicationCommand : IConsoleShellCommand
    {
        public UInt64 ApplicationId;
        public UInt32 Flags = 0;
        public string[] Arguments;
        public string EnvPath;

        public LaunchInstalledApplicationCommand(UInt64 applicationId, string[] arguments, LaunchApplicationOption option)
            : base("LaunchInstalledApplication")
        {
            Flags = option.MakeFlag();
            ApplicationId = applicationId;
            Arguments = arguments;
            EnvPath = string.IsNullOrEmpty(option.EnvPath) ? string.Empty : option.EnvPath;
        }

        private string Escape(string arg)
        {
            return String.Format("\"{0}\"", arg);
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
            Arguments = Arguments == null ? new string[0] : Arguments;
            var argumentBytes = (from argument in this.Arguments select System.Text.Encoding.UTF8.GetBytes(Escape(argument)).Concat(new byte[] { 0x20 })).SelectMany(v => v).Reverse().Skip(1).Reverse().ToArray();
            var envPathBytes = System.Text.Encoding.UTF8.GetBytes(EnvPath).ToArray();
            UInt32 argumentSize = (UInt32)argumentBytes.Count();
            UInt32 envPathSize = (UInt32)envPathBytes.Count();
            UInt32 reserved = 0;

            writer.Write(ApplicationId);
            writer.Write(argumentSize);
            writer.Write(Flags);
            writer.Write(envPathSize);
            writer.Write(reserved);
            writer.Write(argumentBytes);
            writer.Write(envPathBytes);
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.LaunchInstalledApplication;
        }

        public override string CommandExpression
        {
            get
            {
                return $"LaunchInstalledApplication(applicationId=0x{ApplicationId:X16}, flag=0x{Flags:X8})";
            }
        }
    }

    public class LaunchInstalledSystemApplicationCommand : IConsoleShellCommand
    {
        public UInt64 ApplicationId;
        public string[] Arguments;

        public LaunchInstalledSystemApplicationCommand(UInt64 applicationId, string[] arguments)
            : base("LaunchInstalledSystemApplicationCommand")
        {
            ApplicationId = applicationId;
            Arguments = arguments;
        }

        private string Escape(string arg)
        {
            return String.Format("\"{0}\"", arg);
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
            Arguments = Arguments == null ? new string[0] : Arguments;
            var argumentBytes = (from argument in this.Arguments select System.Text.Encoding.UTF8.GetBytes(Escape(argument)).Concat(new byte[] { 0x20 })).SelectMany(v => v).Reverse().Skip(1).Reverse().ToArray();
            UInt32 argumentSize = (UInt32)argumentBytes.Count();
            UInt32 reserved = 0;

            writer.Write(ApplicationId);
            writer.Write(argumentSize);
            writer.Write(reserved);

            writer.Write(argumentBytes);
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.LaunchInstalledSystemApplication;
        }

        public override string CommandExpression
        {
            get
            {
                return $"LaunchInstalledSystemApplication(systemApplicationId=0x{ApplicationId:X16})";
            }
        }
    }

    public class LaunchGameCardApplicationCommand : IConsoleShellCommand
    {
        public UInt64 ApplicationId;
        public UInt32 Flags = 0;
        public string[] Arguments;
        public string EnvPath;

        public LaunchGameCardApplicationCommand(string[] arguments, LaunchApplicationOption option)
            : base("LaunchGameCardApplication")
        {
            Flags = option.MakeFlag();
            ApplicationId = 0;
            Arguments = arguments;
            EnvPath = string.IsNullOrEmpty(option.EnvPath) ? string.Empty : option.EnvPath;
        }

        private string Escape(string arg)
        {
            return String.Format("\"{0}\"", arg);
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
            Arguments = Arguments == null ? new string[0] : Arguments;
            var argumentBytes = (from argument in this.Arguments select System.Text.Encoding.UTF8.GetBytes(Escape(argument)).Concat(new byte[] { 0x20 })).SelectMany(v => v).Reverse().Skip(1).Reverse().ToArray();
            var envPathBytes = System.Text.Encoding.UTF8.GetBytes(EnvPath).ToArray();
            UInt32 argumentSize = (UInt32)argumentBytes.Count();
            UInt32 envPathSize = (UInt32)envPathBytes.Count();
            UInt32 reserved = 0;

            writer.Write(ApplicationId);
            writer.Write(argumentSize);
            writer.Write(Flags);
            writer.Write(envPathSize);
            writer.Write(reserved);
            writer.Write(argumentBytes);
            writer.Write(envPathBytes);
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.LaunchGameCardApplication;
        }

        public override string CommandExpression
        {
            get
            {
                return $"LaunchGameCardApplication(flags={Flags:X8})";
            }
        }
    }

    public class TerminateAppicationCommand : IConsoleShellCommand
    {
        public TerminateAppicationCommand()
            : base("TerminateApplication")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.TerminateApplication;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "TerminateApplication";
            }
        }
    }

    public class TerminateProcessCommand : IConsoleShellCommand
    {
        public TerminateProcessCommand()
            : base("TerminateProcess")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.TerminateProcess;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "TerminateProcess";
            }
        }
    }

    public class RebootCommand : IConsoleShellCommand
    {
        public RebootCommand()
            : base("RebootCommand")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.RebootCommand;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "RebootCommand";
            }
        }
    }

    public class ShutdownCommand : IConsoleShellCommand
    {
        public ShutdownCommand()
            : base("ShutdownCommand")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.ShutdownCommand;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "ShutdownCommand";
            }
        }
    }

    public class SetSafeModeCommand : IConsoleShellCommand
    {
        public SetSafeModeCommand()
            : base("SetSafeModeCommand")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.SetSafeModeCommand;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "SetSafeModeCommand";
            }
        }
    }

    public class ControlVirtualTemperatureCommand : IConsoleShellCommand
    {
        public bool IsEnable;
        public int Temperature;

        public ControlVirtualTemperatureCommand(bool isEnable, int temperature)
            : base("ControlVirtualTemperatureCommand")
        {
            IsEnable = isEnable;
            Temperature = temperature;
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.ControlVirtualTemperature;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
            writer.Write(IsEnable);
            writer.Write((byte)0);
            writer.Write((byte)0);
            writer.Write((byte)0);
            writer.Write(Temperature);
        }

        public override string CommandExpression
        {
            get
            {
                return "ControlVirtualTemperatureCommand";
            }
        }

    }

    public class SaveScreenShotCommand : IConsoleShellCommand
    {
        public SaveScreenShotCommand()
            : base("SaveScreenShotCommand")
        {
        }

        public override int GetCommandId()
        {
            return ConsoleShellCommandId.SaveScreenShot;
        }

        protected override void MakeCommandBody(BinaryWriter writer)
        {
        }

        public override string CommandExpression
        {
            get
            {
                return "SaveScreenShotCommand";
            }
        }

    }

    public class ConsoleShellCommandId
    {
        public const int LaunchApplication = 1;
        public const int TerminateProcess = 2;
        public const int GetFirmwareVersion = 3;
        public const int RebootCommand = 4;
        public const int SetSafeModeCommand = 5;
        public const int TerminateApplication = 7;
        public const int ShutdownCommand = 8;
        public const int ControlVirtualTemperature = 11;
        public const int LaunchInstalledApplication = 12;
        public const int LaunchGameCardApplication = 13;
        public const int LaunchInstalledSystemApplication = 14;
        public const int SaveScreenShot = 15;
    }

    public class LaunchApplicationFlag
    {
        public const uint EnableJitDebug = (1 << 2);
        public const uint DisableAslr    = (1 << 3);
    }
}
