﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nn.Adl.Semantics
{
    [Flags]
    public enum ModificationSelectionOption
    {
        None = 0,
        Recursively = 1 << 0,
    }

    public interface ISemanticInfo
    {
        Syntax.SyntaxElement SyntaxElement { get; }
    }

    public class Document : ISemanticInfo
    {
        public IEnumerable<Definition> Children { get; set; }
        public Syntax.Document SyntaxElement { get; set; }

        Syntax.SyntaxElement ISemanticInfo.SyntaxElement
        {
            get { return this.SyntaxElement; }
        }
    }

    public class Definition : ISemanticInfo
    {
        public SemanticEnvironment Environment { get; set; }
        public string Name { get; set; }
        public Expression Value { get; set; }
        public IEnumerable<Definition> Children { get; set; }
        public Syntax.Definition SyntaxElement { get; set; }

        Syntax.SyntaxElement ISemanticInfo.SyntaxElement
        {
            get { return this.SyntaxElement; }
        }

        public Modification GetModification(string category, string name, ModificationSelectionOption selectionOption)
        {
            return Environment.LookupModification(category, name, selectionOption);
        }

        public Modification GetModification(string category, string name)
        {
            return Environment.LookupModification(category, name, ModificationSelectionOption.None);
        }
    }

    public abstract class Expression : ISemanticInfo
    {
        public Syntax.Expression SyntaxElement { get; internal set; }
        Syntax.SyntaxElement ISemanticInfo.SyntaxElement
        {
            get { return this.SyntaxElement; }
        }
    }

    public class ImmediateExpression : Expression
    {
        public object Value { get; set; }
    }

    public class Modification : ISemanticInfo
    {
        public string Name { get; set; }
        public string Category { get; set; }
        public Expression Expression { get; set; }
        public bool? InheritsToChildren { get; set; }
        public Syntax.Modification SyntaxElement { get; set; }
        Syntax.SyntaxElement ISemanticInfo.SyntaxElement
        {
            get { return this.SyntaxElement; }
        }
    }

    internal static class Utility
    {
        public static Tuple<IEnumerable<T>, IEnumerable<R0>> Filter<T, R0>(IEnumerable<T> e)
            where R0 : class
        {
            var t = new List<T>();
            var r0 = new List<R0>();
            foreach (var i in e)
            {
                var x0 = i as R0;
                if (x0 != null)
                {
                    r0.Add(x0);
                    continue;
                }
                t.Add(i);
            }
            return Tuple.Create((IEnumerable<T>)t, (IEnumerable<R0>)r0);
        }

        public static Tuple<IEnumerable<T>, IEnumerable<R0>, IEnumerable<R1>> Filter<T, R0, R1>(IEnumerable<T> e)
            where R0 : class
            where R1 : class
        {
            var t = new List<T>();
            var r0 = new List<R0>();
            var r1 = new List<R1>();
            foreach (var i in e)
            {
                var x0 = i as R0;
                if (x0 != null)
                {
                    r0.Add(x0);
                    continue;
                }
                var x1 = i as R1;
                if (x1 != null)
                {
                    r1.Add(x1);
                    continue;
                }
                t.Add(i);
            }
            return Tuple.Create((IEnumerable<T>)t, (IEnumerable<R0>)r0, (IEnumerable<R1>)r1);
        }
    }

    public class SemanticEnvironment
    {
        private SemanticEnvironment m_Parent;
        private List<Modification> m_Modifications = new List<Modification>();

        internal SemanticEnvironment(SemanticEnvironment parent)
        {
            this.m_Parent = parent;
        }

        internal void AddModification(SyntaxToSemanticsConverter converter, Syntax.Modification modification)
        {
            var categories = modification.Categories != null
                ? modification.Categories.Elements.Select(x => x.Name)
                : new string[] { null };
            foreach (var category in categories)
            {
                if (m_Modifications.Any(m => m.Category == category && m.Name == modification.DefineeName.Name))
                {
                    throw new ErrorException(string.Format(
                        @"duplicated attribute modification: {0} line {1} pos {2}",
                        modification.DefineeName.Name, 0, 1));
                }
                AddModification(converter, modification, category);
            }
        }

        private void AddModification(SyntaxToSemanticsConverter converter, Syntax.Modification modification, string category)
        {
            var ret = converter.ModificationConverter.ConvertModification(this, modification);
            ret.Category = category;
            ret.SyntaxElement = modification;
            if (!ret.InheritsToChildren.HasValue)
            {
                ret.InheritsToChildren = modification.ModificationPosition != Syntax.ModificationPosition.PostModify;
            }
            m_Modifications.Add(ret);
        }

        private Expression ConvertModificationValue(SyntaxToSemanticsConverter converter, Syntax.Expression expression)
        {
            return converter.ModificationExpressionConverter.ConvertExpression(m_Parent, expression);
        }

        private Modification LookupModificationOne(string category, string name)
        {
            return m_Modifications.Find(e => e.Category == category && e.Name == name)
                ?? m_Modifications.Find(e => e.Category == null && e.Name == name);
        }

        private Modification LookupModificationRecursively(string category, string name, ModificationSelectionOption selectionOption)
        {
            var first = true;
            for (var e = this; e != null; e = e.m_Parent)
            {
                var ret = e.LookupModificationOne(category, name);
                if (ret != null)
                {
                    if (first || ret.InheritsToChildren.Value)
                    {
                        return ret;
                    }
                }
                first = false;
            }
            return null;
        }

        public Modification LookupModification(string category, string name, ModificationSelectionOption selectionOption)
        {
            if (selectionOption.HasFlag(ModificationSelectionOption.Recursively))
            {
                return LookupModificationRecursively(category, name, selectionOption);
            }
            else
            {
                return LookupModificationOne(category, name);
            }
        }
    }

    public class SyntaxToSemanticsConverter
    {
        public ISyntaxToSemanticsModificationConverter ModificationConverter { get; set; }
        public ISyntaxToSemanticsExpressionConverter DefinitionExpressionConverter { get; set; }
        public ISyntaxToSemanticsExpressionConverter ModificationExpressionConverter { get; set; }

        public SyntaxToSemanticsConverter()
        {
            this.ModificationConverter = new DefaultSyntaxToSemanticsModificationConverter();
            this.DefinitionExpressionConverter = new DefaultSyntaxToSemanticsExpressionConverter();
            this.ModificationExpressionConverter = new DefaultSyntaxToSemanticsExpressionConverter();
        }

        public Document Convert(Syntax.Document syntaxDocument)
        {
            return new Document
            {
                Children = ConvertStatements(null, syntaxDocument.Statements.Elements).ToArray(),
                SyntaxElement = syntaxDocument,
            };
        }

        private IEnumerable<Definition> ConvertStatements(SemanticEnvironment parent, IEnumerable<Syntax.Statement> statements)
        {
            var p = Utility.Filter<Syntax.Statement, Syntax.ScopedModification>(statements);
            var scopedModifications = p.Item2;
            var environment = MakeEnvironment(parent, scopedModifications.SelectMany(x => x.Modifications));
            foreach (var e in p.Item1)
            {
                var definition = e as Syntax.Definition;
                if (definition != null)
                {
                    yield return ConvertDefinition(environment, definition);
                    continue;
                }
                var group = e as Syntax.Group;
                if (group != null)
                {
                    foreach (var f in ConvertGroup(environment, group))
                    {
                        yield return f;
                    }
                    continue;
                }
                throw null;
            }
        }

        private IEnumerable<Definition> ConvertGroup(SemanticEnvironment parent, Syntax.Group group)
        {
            var environment = MakeEnvironment(parent, group.Modifications);
            foreach (var e2 in ConvertStatements(environment, group.Statements))
            {
                yield return e2;
            }
        }

        private SemanticEnvironment MakeEnvironment(SemanticEnvironment parent, IEnumerable<Syntax.Modification> modifications)
        {
            var environment = new SemanticEnvironment(parent);
            foreach (var e in modifications)
            {
                environment.AddModification(this, e);
            }
            return environment;
        }

        private Definition ConvertDefinition(SemanticEnvironment parent, Syntax.Definition definition)
        {
            var environment = MakeEnvironment(parent, definition.GetModifications());
            return new Definition
            {
                Environment = environment,
                Name = definition.DefineeName.Name,
                Value = ConvertDefinitionValue(parent, definition.ValueExpression),
                Children = definition.Children == null ? null : ConvertStatements(environment, definition.Children.Elements).ToArray(),
                SyntaxElement = definition,
            };
        }

        private Expression ConvertDefinitionValue(SemanticEnvironment parent, Syntax.Expression expression)
        {
            return DefinitionExpressionConverter.ConvertExpression(parent, expression);
        }
    }

    public interface ISyntaxToSemanticsModificationConverter
    {
    }

    public class DefaultSyntaxToSemanticsModificationConverter : ISyntaxToSemanticsModificationConverter
    {
        protected ISyntaxToSemanticsExpressionConverter SyntaxToSemanticsExpressionConverter { get; set; }

        public DefaultSyntaxToSemanticsModificationConverter()
        {
            this.SyntaxToSemanticsExpressionConverter = new DefaultSyntaxToSemanticsExpressionConverter();
        }

        protected internal Modification ConvertImpl(SemanticEnvironment environment, Syntax.ValueModification modification)
        {
            return new Modification
            {
                Name = modification.DefineeName.Name,
                Expression = SyntaxToSemanticsExpressionConverter.ConvertExpression(environment, modification.ValueExpression),
            };
        }

        protected internal Modification ConvertImpl(SemanticEnvironment environment, Syntax.NonValueModification modification)
        {
            return new Modification
            {
                Name = modification.DefineeName.Name,
                Expression = null,
            };
        }
    }

    public interface ISyntaxToSemanticsExpressionConverter
    {
    }

    public class DefaultSyntaxToSemanticsExpressionConverter : ISyntaxToSemanticsExpressionConverter
    {
        protected internal Expression ConvertImpl(SemanticEnvironment environment, Syntax.Expression expression)
        {
            throw new ErrorException(string.Format("unknown expression: {0}", expression.GetType()));
        }

        protected internal Expression ConvertImpl(SemanticEnvironment environment, Syntax.LiteralExpression expression)
        {
            return new ImmediateExpression
            {
                Value = expression.Literal.GetValue(),
            };
        }
    }

    public static class SyntaxToSemanticsConverterExtension
    {
        public static Expression ConvertExpression(this ISyntaxToSemanticsExpressionConverter converter, SemanticEnvironment environment, Syntax.Expression source)
        {
            var ret = source != null
                ? (Expression)((dynamic)converter).ConvertImpl(environment, (dynamic)source)
                : new ImmediateExpression() { Value = true };
            ret.SyntaxElement = source;
            return ret;
        }

        public static Modification ConvertModification(this ISyntaxToSemanticsModificationConverter converter, SemanticEnvironment environment, Syntax.Modification source)
        {
            var ret = (Modification)((dynamic)converter).ConvertImpl(environment, (dynamic)source);
            ret.SyntaxElement = source;
            return ret;
        }
    }
}
