﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ManuHostTools.UsbLibrary;

namespace Nintendo.ManuHostTools
{
    public class UsbFileIoDaemon
    {
        public static readonly Guid UsbDeviceGuid = new Guid("97FFFD48-2D1D-47A0-85A3-07FDE6FA0143");
        public static readonly int DefaultInterfaceNumber = (int)UsbSession.InterfaceNumber.UsbFileIo;

        private UsbStream stream = null;
        private bool IsServerRun = false;
        public bool IsVerbose = false;

        private enum RequestId : uint
        {
            Nop = 0,
            GetFileSize = 1,
            ReadFromHost = 2,
            WriteToHost = 3,
        };

        private enum ResultId : uint
        {
            Success       = 0,
            ArgumentNullException,
            SecurityException,
            ArgumentException,
            UnauthorizedAccessException,
            PathTooLongException,
            NotSupportedException,
            FileNotFoundException,
            Undefined     = 0xFFFFFFFF,
        };

        private static ResultId FsExceptionToResultId(Exception e)
        {
            switch(e.GetType().Name)
            {
                case "ArgumentNullException":
                    {
                        return ResultId.ArgumentNullException;
                    }
                case "SecurityException":
                    {
                        return ResultId.SecurityException;
                    }
                case "ArgumentException":
                    {
                        return ResultId.ArgumentException;
                    }
                case "UnauthorizedAccessException":
                    {
                        return ResultId.UnauthorizedAccessException;
                    }
                case "PathTooLongException":
                    {
                        return ResultId.PathTooLongException;
                    }
                case "NotSupportedException":
                    {
                        return ResultId.NotSupportedException;
                    }
                case "FileNotFoundException":
                    {
                        return ResultId.FileNotFoundException;
                    }
                default :
                    {
                        return ResultId.Undefined;
                    }
            }
        }

        public void CreateSession(UsbDevice usbDevice, int port)
        {
            var usbSession = new UsbSession();

            usbSession.CreateSession(usbDevice, port);

            this.stream = new UsbStream(usbSession);
        }

        public Task RunAsync()
        {
            var task = new Task(() =>
            {
                this.Run();
            });
            task.Start();
            return task;
        }

        public void Run()
        {
            IsServerRun = true;
            this.ServerLoop();
        }

        public void Stop()
        {
            IsServerRun = false;
        }

        private void ServerLoop()
        {
            while (IsServerRun)
            {
                try
                {
                    var requestHeader = RequestHeader.ReceiveRequestHeader(this.stream);

                    if (requestHeader.MagicNumber != RequestHeader.CorrectMagicNumber)
                    {
                        VerboseWriteLine("Invalid Magic Number (Expect : {0:X}, Actual : {1:X})", RequestHeader.CorrectMagicNumber, requestHeader.MagicNumber);
                        continue;
                    }

                    switch ((RequestId)requestHeader.Request)
                    {
                        case RequestId.Nop:
                            {
                                VerboseWriteLine("Nop");
                                break;
                            }
                        case RequestId.GetFileSize:
                            {
                                ProcessGetFileSize(requestHeader);
                                break;
                            }
                        case RequestId.ReadFromHost:
                            {
                                ProcessReadFromHost(requestHeader);
                                break;
                            }
                        case RequestId.WriteToHost:
                            {
                                ProcessWriteToHost(requestHeader);
                                break;
                            }
                        default:
                            {
                                VerboseWriteLine("Unknown Command ID : {0}", requestHeader.Request);
                                break;
                            }
                    }
                }
                catch (OperationCanceledException)
                {
                    this.Stop();
                    VerboseWriteLine("UfioDaemon stop because device disconnected.");
                }
            }
        }

        private void ProcessGetFileSize(RequestHeader header)
        {
            var pathBytes = new byte[header.BodySize];
            stream.Read(pathBytes, 0, pathBytes.Length);
            var pathString = System.Text.Encoding.ASCII.GetString(pathBytes);
            var requestResult = new RequestResult();

            VerboseWriteLine("GetFileSize, File:{0}", Path.GetFileName(pathString));

            try
            {
                var fileInfo = new System.IO.FileInfo(pathString);
                requestResult.ReturnValue = (UInt64)fileInfo.Length;
                requestResult.Id = ResultId.Success;
            }
            catch(Exception e)
            {
                VerboseWriteLine("Exception: {0}", e.Message);
                requestResult.Id = FsExceptionToResultId(e);
            }
            finally
            {
                requestResult.Send(this.stream);
            }
        }

        private void ProcessWriteToHost(RequestHeader header)
        {
            var readWriteBody = ReadWriteBody.ReceiveReadWriteBody(header, this.stream);
            var requestResult = new RequestResult();
            requestResult.Id = ResultId.Success;
            requestResult.Send(this.stream);

            VerboseWriteLine("WriteToHost, File:{0}, Offset:{1}, Length:{2}", Path.GetFileName(readWriteBody.Path), readWriteBody.Offset, readWriteBody.Length);

            try
            {
                var receiveBuffer = new Byte[readWriteBody.Length];
                var fileMode = File.Exists(readWriteBody.Path) ? FileMode.Open : FileMode.Create;
                stream.Read(receiveBuffer, 0, (int)readWriteBody.Length);

                using (var fs = new FileStream(readWriteBody.Path, fileMode, FileAccess.Write))
                {
                    fs.Seek((int)readWriteBody.Offset, SeekOrigin.Begin);
                    fs.Write(receiveBuffer, 0, (int)readWriteBody.Length);
                }

                requestResult.Id = ResultId.Success;
            }
            catch (Exception e)
            {
                VerboseWriteLine("Exception: {0}", e.Message);
                requestResult.Id = FsExceptionToResultId(e);
            }
            finally
            {
                requestResult.Send(this.stream);
            }
        }

        private void ProcessReadFromHost(RequestHeader header)
        {
            var readWriteBody = ReadWriteBody.ReceiveReadWriteBody(header, this.stream);
            var requestResult = new RequestResult();

            VerboseWriteLine("ReadFromHost, File:{0}, Offset:{1}, Length:{2}", Path.GetFileName(readWriteBody.Path), readWriteBody.Offset, readWriteBody.Length);

            try
            {
                var fileInfo = new System.IO.FileInfo(readWriteBody.Path);

                var sendBuffer = new Byte[readWriteBody.Length];

                using (var fs = new System.IO.FileStream(readWriteBody.Path, FileMode.Open, FileAccess.Read))
                {
                    fs.Seek((int)readWriteBody.Offset, SeekOrigin.Begin);
                    fs.Read(sendBuffer, 0, (int)sendBuffer.Length);
                }

                requestResult.ReturnValue = (UInt64)fileInfo.Length;
                requestResult.Id = ResultId.Success;
                requestResult.Send(this.stream);

                if(readWriteBody.Offset + readWriteBody.Length > (UInt64)fileInfo.Length)
                {
                    VerboseWriteLine("Too large readSize. File ({0}) is {1} bytes. read area is {2} to {3}",
                        Path.GetFileName(readWriteBody.Path), fileInfo.Length, readWriteBody.Offset, readWriteBody.Offset+ readWriteBody.Length);
                    return;
                }

                stream.Write(sendBuffer, 0, (int)sendBuffer.Length);

                requestResult.Id = ResultId.Success;
                requestResult.Send(this.stream);
            }
            catch (Exception e)
            {
                VerboseWriteLine("Exception: {0}", e.Message);
                requestResult.Id = FsExceptionToResultId(e);
                requestResult.Send(this.stream);
            }
        }

        private class RequestResult
        {
            public static readonly UInt64 CorrectMagicNumber = 0xDEADCAFEDEADCAFE;
            public ResultId Id = ResultId.Undefined;
            public UInt64 ReturnValue = 0;

            public void Send(UsbStream stream)
            {
                RequestResult.SendRequestResult(this, stream);
            }

            public static void SendRequestResult(RequestResult requestResult, UsbStream stream)
            {
                var requestResultBytes = new byte[24];

                using (var ms = new MemoryStream(requestResultBytes))
                using (var bw = new BinaryWriter(ms))
                {
                    bw.Write(RequestResult.CorrectMagicNumber);
                    bw.Write((UInt32)requestResult.Id);
                    bw.Write((UInt32)0x0);
                    bw.Write(requestResult.ReturnValue);

                }
                stream.Write(requestResultBytes, 0, requestResultBytes.Length);
            }
        }

        private class RequestHeader
        {
            public static readonly UInt64 CorrectMagicNumber = 0xDEADCAFEDEADCAFE;
            public UInt64 MagicNumber = 0;
            public UInt32 Request = 0;
            public UInt64 BodySize = 0;

            public static RequestHeader ReceiveRequestHeader(UsbStream stream)
            {
                var receiveBuffer = new byte[8];
                var requestHeader = new RequestHeader();

                // MagicNumber (uint64_t)
                stream.Read(receiveBuffer, 0, sizeof(UInt64));
                requestHeader.MagicNumber = BitConverter.ToUInt64(receiveBuffer, 0);

                // RequestId (uint32_t)
                stream.Read(receiveBuffer, 0, sizeof(UInt32));
                requestHeader.Request = BitConverter.ToUInt32(receiveBuffer, 0);

                // Reserved (uint32_t) 読み捨て
                stream.Read(receiveBuffer, 0, sizeof(UInt32));

                // BodySize (uint64_t)
                stream.Read(receiveBuffer, 0, sizeof(UInt64));
                requestHeader.BodySize = BitConverter.ToUInt32(receiveBuffer, 0);

                return requestHeader;
            }
        }

        private class ReadWriteBody
        {
            public UInt64 Length = 0;
            public UInt64 Offset = 0;
            public string Path = String.Empty;

            public static ReadWriteBody ReceiveReadWriteBody(RequestHeader requestHeader, UsbStream stream)
            {
                var readWriteBody = new ReadWriteBody();
                var receiveBuffer = new byte[requestHeader.BodySize];

                // Length (uint64_t)
                stream.Read(receiveBuffer, 0, sizeof(UInt64));
                readWriteBody.Length = BitConverter.ToUInt64(receiveBuffer, 0);

                // Offset (uint32_t)
                stream.Read(receiveBuffer, 0, sizeof(UInt64));
                readWriteBody.Offset = BitConverter.ToUInt32(receiveBuffer, 0);

                // Path (uint32_t) 読み捨て
                var pathLength = receiveBuffer.Length - sizeof(UInt64) - sizeof(UInt64);
                stream.Read(receiveBuffer, 0, pathLength);
                readWriteBody.Path = Encoding.UTF8.GetString(receiveBuffer, 0, pathLength);

                return readWriteBody;
            }
        }

        private void VerboseWriteLine(string format, params Object[] objs)
        {
            if (this.IsVerbose)
            {
                Console.WriteLine(format, objs);
            }
        }

        private void VerboseWrite(string format, params Object[] objs)
        {
            if (this.IsVerbose)
            {
                Console.Write(format, objs);
            }
        }
    }
}
