﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Threading;
using System.Xml.Serialization;
using Nintendo.ManuHostTools.UsbLibrary;
using Nintendo.ManuHostTools.Commands;

namespace Nintendo.ManuHostTools
{
    internal class RunOnTargetFromNand
    {
        private static readonly Guid UsbDeviceGuid = new Guid("97FFFD48-2D1D-47A0-85A3-07FDE6FA0143");
        private static readonly int DefaultInterfaceNumber = (int)UsbSession.InterfaceNumber.CommandRequest;
        private static readonly int OldDefaultInterfaceNumber = (int)UsbSession.InterfaceNumber.OldCommandRequest;
        private static readonly int HostCommunicatorInterfaceNumber = (int)UsbSession.InterfaceNumber.UsbHostCommunicator;
        private ProgramOption option;

        public bool IsVerbose = false;

        public RunOnTargetFromNand(ProgramOption option)
        {
            this.option = option;
        }

        public void Run()
        {
            Log("Please connect target with USB");
            UsbDevice targetUsbDevice = DetectSingleUsbDevice(TimeSpan.FromSeconds(this.option.TargetDetectTimeOutSec));
            if (targetUsbDevice == null)
            {
                throw new Exception("Unable to find USB Device.");
            }

            Log("Detect target.");

            var task = new Task(() =>
            {
                var ufioDaemon = new UsbFileIoDaemon();
                ufioDaemon.IsVerbose = option.IsVerbose;
                var usbSession = new UsbSession();
                var logSession = new UsbSession();

                while (true)
                {
                    try
                    {
                        try
                        {
                            usbSession.CreateSession(targetUsbDevice, RunOnTargetFromNand.DefaultInterfaceNumber);
                        }
                        catch
                        {
                            Log("Use old usb interface(number: {0})", RunOnTargetFromNand.OldDefaultInterfaceNumber);
                            usbSession.CreateSession(targetUsbDevice, RunOnTargetFromNand.OldDefaultInterfaceNumber);
                        }

                        logSession.CreateSession(targetUsbDevice, RunOnTargetFromNand.HostCommunicatorInterfaceNumber);
                        ufioDaemon.CreateSession(targetUsbDevice, UsbFileIoDaemon.DefaultInterfaceNumber);
                    }
                    catch (OperationCanceledException e)
                    {
                        Log("Failed to establish USB session. Exception message \"{0}\"  Try reconnecting...", e.Message);
                        targetUsbDevice = DetectSingleUsbDevice(TimeSpan.FromSeconds(this.option.TargetDetectTimeOutSec));
                        if (targetUsbDevice == null)
                        {
                            Log("Unable to reconnect USB Device.");
                            throw e;
                        }
                        continue;
                    }
                    break;
                }

                try
                {
                    var logTask = new Task(() =>
                    {
                        while (true)
                        {
                            var readBuf = new byte[1024];
                            var readSize = logSession.Read(readBuf, 0, readBuf.Length);
                            var text = System.Text.Encoding.ASCII.GetString(readBuf, 0, readSize);
                            Console.WriteLine("[target] {0}", text);
                        }
                    });
                    logTask.Start();

                    var stream = new UsbStream(usbSession);
                    ufioDaemon.RunAsync();
                    Command command;
                    if (this.option.Shutdown)
                    {
                        command = new ShutdownCommand();
                    }
                    else if (this.option.Reboot)
                    {
                        command = new RebootCommand();
                    }
                    else
                    {
                        command = new ApplicationLaunchCommand(this.option.ProgramId, this.option.Arguments);
                    }
                    var result = command.Send(stream);
                    if (result.Id != Command.ResultId.Success)
                    {
                        Log("Receive error result from Target (ResultId : {0}, InnerValue 0x{1:X8})", Enum.GetName(typeof(Command.ResultId), result.Id), result.NnResult);
                    }
                    else
                    {
                        Log("Target program exit.");
                    }
                }
                catch (OperationCanceledException)
                {
                    Log("RunOnTargetFromNand is exit because device disconnected.");
                }
                finally
                {
                    ufioDaemon.Stop();
                }
            });

            task.Start();

            task.Wait();
        }

        private UsbDevice DetectSingleUsbDevice(TimeSpan timeout)
        {
            if (String.IsNullOrEmpty(this.option.UsbDevicePath))
            {
                var deviceList = UsbDevice.WaitForDetectUsbDeviceByGuid(UsbDeviceGuid, timeout);
                return (deviceList.Count == 0) ? null : deviceList[this.option.UsbDeviceIndex];
            }
            else
            {
                return UsbDevice.DetectUsbDeviceByDevicePath(option.UsbDevicePath, UsbDeviceGuid, timeout);
            }
        }


        private void Log(string message, params object[] args)
        {
            Console.WriteLine(message, args);
        }
    }
}
