﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SvcProfilerSourceGenerator : SourceGenerator
    {
        public SvcProfilerSourceGenerator(CodeGenNames p) : base(p) { }

        public void Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string path)
        {
            this.Generate(ss, abi, null, conv, templatePath, path);
        }

        protected string Namespace
        {
            get
            {
                return string.Format("{0}::{1}", CodeGenNames.UserCommonNamespace, Params.StubNamespace);
            }
        }

        protected string ProfilerNamespace
        {
            get
            {
                return string.Format("{0}::{1}::profiler", CodeGenNames.UserCommonNamespace, Params.StubNamespace);
            }
        }

        protected override string Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        {
            var fnMap = ss.Operations.ToDictionary(x => x.FunctionNumber, x => x);
            var includes = ss.Categories.Where(c => c.Includes != null && c.Operations != null)
                                        .SelectMany(c => c.Includes)
                                        .ToLookup(x => x)
                                        .Select(x => x.Key)
                                        .OrderBy(x => x)
                                        .ToArray();

            var sb = new StringBuilder();

            foreach (var inc in includes)
            {
                sb.Append(SourceGenerator.MakeIncludeLine(inc));
            }
            sb.AppendLine();

            sb.Append(SourceGenerator.MakeNameSpaceBeginText(this.ProfilerNamespace));
            sb.AppendLine();

            for (int i = 0; i < 128; ++i)
            {
                var f = Util.GetValueOrDefault(fnMap, i);
                if (f != null)
                {
                    var descr = this.GenerateDescr(f, abi[f.Name]);
                    sb.AppendFormat("{0} {1}{2};\r\n", descr.Item1, f.Name, descr.Item2);
                }
            }
            sb.AppendLine();

            sb.AppendLine("void* handlerTable[128] =");
            sb.AppendLine("{");
            for (int i = 0; i < 128; ++i)
            {
                var f = Util.GetValueOrDefault(fnMap, i);
                if (f == null)
                {
                    sb.AppendFormat("    /* {0,3} */  NULL,\r\n", i);
                }
                else
                {
                    sb.AppendFormat("    /* {0,3} */  reinterpret_cast<void*>({1}),\r\n", i, f.Name);
                }
            }
            sb.AppendLine("};");
            sb.AppendLine();

            sb.Append(SourceGenerator.MakeNameSpaceEndText(this.ProfilerNamespace));
            sb.AppendLine();

            sb.Append(SourceGenerator.MakeNameSpaceBeginText(this.Namespace));
            sb.AppendLine();

            sb.AppendLine("void* Intercept(int svcId, void* func)");
            sb.AppendLine("{");
            sb.AppendFormat("    void* old = {0}::handlerTable[svcId];\r\n", ProfilerNamespace);
            sb.AppendFormat("    {0}::handlerTable[svcId] = func;\r\n", ProfilerNamespace);
            sb.AppendLine("    return old;");
            sb.AppendLine("}");
            sb.AppendLine();

            for (int i = 0; i < 128; ++i)
            {
                var f = Util.GetValueOrDefault(fnMap, i);
                if (f != null)
                {
                    var descr = this.GenerateDescr(f, abi[f.Name]);
                    sb.AppendFormat("{0} {1}{2}\r\n", descr.Item1, f.Name, descr.Item2);
                    sb.AppendLine("{");
                    sb.AppendFormat("    using Type = {0}(*){1};\r\n", descr.Item1, descr.Item2);
                    sb.AppendFormat("    auto func = reinterpret_cast<Type>({0}::handlerTable[{1} /* {2} */]);\r\n",
                        ProfilerNamespace, f.FunctionNumber, MakeMacroName(f.Name));
                    sb.AppendFormat("    return func{0};\r\n", descr.Item3);
                    sb.AppendLine("}");
                    sb.AppendLine();
                }
            }

            sb.Append(SourceGenerator.MakeNameSpaceEndText(this.Namespace));

            return sb.ToString();
        }

        private Tuple<string, string, string> GenerateDescr(Operation op, AbiLayout al)
        {
            var paramText = GenerateParamText(op, al, false);

            return new Tuple<string, string, string>(
                op.ReturnType.Name,
                paramText,
                string.Format("({0})", string.Join(", ", op.Parameters.Select(x => x.Name))));
        }
    }
}
