﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SourceGenerator
    {
        protected CodeGenNames Params { get; private set; }

        protected SourceGenerator()
        {
            Params = null;
        }
        protected SourceGenerator(CodeGenNames p)
        {
            Params = p;
        }

        protected void GenerateWithCategory(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string pathTemplate)
        {
            string template;

            try
            {
                template = File.ReadAllText(templatePath, Encoding.UTF8);
            }
            catch (DirectoryNotFoundException e)
            {
                throw new ErrorException("ディレクトリが見つかりませんでした。", e);
            }
            catch (FileNotFoundException e)
            {
                throw new ErrorException("ファイルが見つかりませんでした。", e);
            }

            foreach (var c in ss.Categories)
            {
                var text = Generate(c, abi, svc, conv);
                if (text != null)
                {
                    var path = ApplyPathTemplate(pathTemplate, c.Name);
                    Directory.CreateDirectory(Path.GetDirectoryName(path));
                    File.WriteAllText(path, template + text, Encoding.UTF8);
                }
            }
        }
        protected string ApplyPathTemplate(string template, string name)
        {
            return string.Format(template.Replace("*", "{0}"), name);
        }
        protected virtual string Generate(CategorySet c,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        { return null; }

        protected void Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string path)
        {
            var template = File.ReadAllText(templatePath, Encoding.UTF8);

            var text = Generate(ss, abi, svc, conv);
            Directory.CreateDirectory(Path.GetDirectoryName(path));
            try
            {
                File.WriteAllText(path, template + text, Encoding.UTF8);
            }
            catch (ArgumentException e)
            {
                throw new ErrorException(
                    string.Format("ファイルの作成に失敗しました。\nパス='{0}'", path), e);
            }
        }
        protected virtual string Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        { return null; }

        protected static string GenerateParamText(Operation op, AbiLayout al, bool wrapReference)
        {
            return string.Format("({0})",
                string.Join(", ",
                    op.Parameters.Select(x => GenerateParamText(x, al, wrapReference))));
        }

        private class ParamTextParts
        {
            public string Const { get; private set; }
            public string TypeQualifier { get; private set; }
            public string ArrayMark { get; private set; }

            public ParamTextParts(string c, string tq, string am)
            {
                Const = c;
                TypeQualifier = tq;
                ArrayMark = am;
            }
            public string Format(string typeName, string variableName, bool wrap)
            {
                var typeQualifier = (typeName == "void") ? "*" : TypeQualifier;
                var format = wrap ? CodeGenNames.UserPointerWrapperClassName + "<{0}{1}*> {3}"
                                  : "{0}{1}{2} {3}{4}";

                return string.Format(format, Const, typeName, typeQualifier, variableName, ArrayMark);
            }
        }

        // isReference/isArray, isIn
        private static ParamTextParts[][] s_paramTextparts = new ParamTextParts[][]
        {
            // isArray
            new ParamTextParts[]
            {
                new ParamTextParts(string.Empty, string.Empty, "[]"),           // out
                new ParamTextParts("const ", string.Empty, "[]"),               // in
            },
            // isReference
            new ParamTextParts[]
            {
                new ParamTextParts(string.Empty, "*", string.Empty),            // out
                new ParamTextParts("const ", "&", string.Empty),                // in
            },
            // other
            new ParamTextParts[]
            {
                new ParamTextParts(string.Empty, string.Empty, string.Empty),   // out
                new ParamTextParts(string.Empty, string.Empty, string.Empty),   // in
            },
        };

        private static bool IsRequireWrapping(Layout alin, OperationParameter p)
        {
            return p.IsPointerForced;
        }

        private static string GenerateParamText(OperationParameter op, AbiLayout al, bool wrapReference)
        {
            bool isReference = al.In.GetEntry(op.Name).IsReference;
            bool isIn = op.IsInput;
            bool isArray = op.IsArray;

            bool wrap = wrapReference && IsRequireWrapping(al.In, op);
            int index = isArray ? 0 : (isReference ? 1 : 2);
            var parts = s_paramTextparts[index][isIn ? 1 : 0];

            return parts.Format(op.Type.GlobalName, op.Name, wrap);
        }
        protected static string MakePrototype(Operation op, AbiLayout al, bool wrapReference)
        {
            return string.Format("{0} {1}{2}",
                op.ReturnType.Name,
                op.Name,
                GenerateParamText(op, al, wrapReference));
        }

        protected static string MakeNameSpaceBeginText(string ns)
        {
            var sb = new StringBuilder();
            sb.AppendLine("#ifdef __cplusplus");
            sb.AppendLine();
            foreach (var n in ns.Split(new string[] { "::" }, StringSplitOptions.None))
            {
                sb.AppendFormat("namespace {0} {1}\r\n", n, "{");
            }
            return sb.ToString();
        }
        protected static string MakeNameSpaceEndText(string ns)
        {
            var sb = new StringBuilder();
            foreach (var n in ns.Split(new string[] { "::" }, StringSplitOptions.None).Reverse())
            {
                sb.AppendFormat("{0} // end of namespace {1}\r\n", "}", n);
            }
            sb.AppendLine();
            sb.AppendLine("#endif  // ifdef __cplusplus");
            return sb.ToString();
        }

        protected static string MakeMacroName(string name)
        {
            var ret1 = Regex.Replace(name, "[A-Z]", m => "_" + m);
            var ret2 = ret1.ToUpperInvariant();
            return ret2.TrimStart('_');
        }
        protected static string MakeSymbolName(
            string format,
            Operation op,
            AbiLayout al,
            bool wrapReference)
        {
            var fi = new Mangler.FunctionInfo();
            fi.Name = string.Format(format, op.Name);

            var paramList = new List<Mangler.TypeInfo>();
            foreach (var p in op.Parameters)
            {
                bool wrap = wrapReference && IsRequireWrapping(al.In, p);

                var layout = al.In.GetEntry(p.Name);
                var ti = new Mangler.TypeInfo();
                ti.Name = p.Type.RealGlobalName;
                ti.IsConst = layout.IsReference && p.IsInput;
                if (wrap)
                {
                    ti.IsArray = false;
                    ti.IsPointer = true;
                    ti.IsReference = false;

                    var tiv = new Mangler.TypeInfo();
                    tiv.Name = "void";

                    var tti = new Mangler.TypeInfo();
                    tti.Name = string.Format("{0}::{1}",
                                CodeGenNames.KernelCommonNamespace,
                                CodeGenNames.UserPointerWrapperClassName);
                    tti.IsArray = false;
                    tti.IsConst = false;
                    tti.IsPointer = false;
                    tti.IsReference = false;
                    tti.TemplateParameters = new Mangler.TypeInfo[] { ti, tiv };

                    paramList.Add(tti);
                }
                else
                {
                    ti.IsArray = false;
                    if (p.IsInput)
                    {
                        ti.IsPointer = p.IsArray && layout.IsReference;
                        ti.IsReference = !p.IsArray && layout.IsReference;
                    }
                    else
                    {
                        Debug.Assert(layout.IsReference, "assertion failed");
                        ti.IsPointer = layout.IsReference;
                        ti.IsReference = false;
                    }
                    paramList.Add(ti);
                }
            }
            fi.Parameters = paramList.ToArray();

            return Mangler.Mangle(fi, al.CodeGenParams);
        }

        protected static string Indent(string text, string indent)
        {
            var lines = text.Split(new string[] { "\n" }, StringSplitOptions.None);
            return string.Join("\n", lines.Select(x => (x.Length > 0) ? indent + x : x));
        }

        protected static string MakeIncludeLine(string path)
        {
            var path2 = path.Replace('\\', '/');
            if (path2.StartsWith("nn/") || (!path2.Contains(".")))
            {
                return string.Format("#include <{0}>\r\n", path2);
            }
            else
            {
                return string.Format("#include \"{0}\"\r\n", path2);
            }
        }

        protected static string MakePragmaOnce()
        {
            return "#pragma once\r\n";
        }
    }
}
