﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal abstract class CodeGenerator
    {
        private const int OperandMinimumLength = 24;

        private class Code
        {
            public string Mnemonic { get; private set; }
            public string Operands { get; private set; }
            public string Comment { get; private set; }

            public Code(string m, string o, string c)
            {
                Mnemonic = m;
                Operands = o;
                Comment = c;
            }
            public string Format(int maxOperandLength)
            {
                if (Mnemonic == string.Empty)
                {
                    return "\r\n";
                }
                else if (Comment == string.Empty)
                {
                    if (Operands == string.Empty)
                    {
                        return string.Format("    {0}\r\n", Mnemonic);
                    }
                    else
                    {
                        return string.Format("    {0,-8} {1}\r\n", Mnemonic, Operands);
                    }
                }
                else
                {
                    var format = string.Concat("    {0,-8} {1,-", maxOperandLength, "}  // {2}\r\n");
                    return string.Format(format, Mnemonic, Operands, Comment);
                }
            }
            public static readonly Code Empty = new Code(string.Empty, string.Empty, string.Empty);
        }

        private List<Code> m_codes = new List<Code>();

        public string CodeText
        {
            get
            {
                var maxOperandLength = m_codes.Where(x => x.Comment.Length > 0)
                                              .Select(x => x.Operands.Length)
                                              .Concat(new int[] { OperandMinimumLength })
                                              .Max();
                var lines = m_codes.Select(x => x.Format(maxOperandLength));
                return string.Concat(lines);
            }
        }

        public abstract void CallSystem(string id);
        public abstract void CallFunction(string name);
        public abstract void AllocateFromStack(int size);
        public abstract void FreeToStack(int size);
        public abstract void SaveRegisters(int[] regs, bool withLr, bool isRequirePadding);
        public abstract void RestoreRegisters(int[] regs, bool withPc, bool isRequirePadding);
        public abstract void Move(int to, int from, string name);
        public abstract void Pack64Bit(int to, int hi, int lo, string name);
        public abstract void Unpack64Bit(int hi, int lo, int from, string name);
        public abstract void StoreToStack(int from, int offset, string name);
        public abstract void LoadFromStack(int to, int offset, string name);
        public abstract void StoreToStack(int from, int offset, string name, int index);
        public abstract void LoadFromStack(int to, int offset, string name, int index);
        public abstract void Store(int from, int toBase, int offset, int size, string name, int index);
        public abstract void Load(int to, int fromBase, int offset, int size, string name, int index);
        public abstract void LoadStackAddress(int to, int offset, string name);

        public abstract void Return();

        public void AddEmptyLine()
        {
            m_codes.Add(Code.Empty);
        }

        protected void AddCode(string mnemonic, string operands)
        {
            AddCode(mnemonic, operands, string.Empty);
        }
        protected void AddCode(string mnemonic, string operands, string comment)
        {
            m_codes.Add(new Code(mnemonic, operands, comment));
        }

        public static CodeGenerator FromCodeGenParams(CodeGenParams abiParams, CodeGenParams svcParams)
        {
            switch (abiParams.Archtecture)
            {
            case CodeGenParams.ArchitectureType.AArch32:
                return new AArch32CodeGenerator();
            case CodeGenParams.ArchitectureType.AArch64:
                {
                    var cgp = (svcParams != null) ? svcParams : abiParams;
                    var is32BitStruct = (cgp.StorageSize == 4);
                    return new AArch64CodeGenerator(is32BitStruct);
                }
            }
            throw new ErrorException("内部エラー");
        }
    }
}
