﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    /// <summary>
    /// 一つの関数に対応する ABI での引数レイアウトを表します。
    /// </summary>
    internal class AbiLayout
    {
        /// <summary>
        /// システムコール呼び出し前の引数レイアウト情報を取得します。
        /// </summary>
        public Layout In { get; private set; }
        /// <summary>
        /// システムコール呼び出し後の引数レイアウト情報を取得します。
        /// </summary>
        public Layout Out { get; private set; }

        public CodeGenParams CodeGenParams { get; private set; }

        private AbiLayout(CodeGenParams ap)
        {
            this.CodeGenParams = ap;
            In = new Layout(ap.StorageSize, ap.RegisterCount, ap.PointerSize, ap.RegisterSize);
            Out = new Layout(ap.StorageSize, ap.RegisterCount, ap.PointerSize, ap.RegisterSize);
        }

        public static Dictionary<string, AbiLayout> Arrange(
            Operation[] operations, CodeGenParams ap)
        {
            var map = new Dictionary<string, AbiLayout>();

            foreach (var op in operations)
            {
                map.Add(op.Name, Arrange(op, ap));
            }

            return map;
        }
        public Layout.StorageEntry GetEntry(string name)
        {
            return In.GetEntryOrNull(name) ?? Out.GetEntry(name);
        }
        public Layout.Position[] GetPositions(string name)
        {
            return In.GetPositions(name) ?? Out.GetPositions(name);
        }

        private static AbiLayout Arrange(Operation op, CodeGenParams ap)
        {
            int registerIndex = 0;
            int stackIndex = 0;

            AbiLayout al = new AbiLayout(ap);

            // 返り値
            {
                var t = op.ReturnType;
                if (t.Size > ap.StorageSize)
                {
                    if (t.RealType.IsPredefined)
                    {
                        // long long
                        al.Out.Add("return", Layout.StorageType.Register, 0, t.Size, t.Size);
                    }
                    else
                    {
                        // 構造体等
                        al.In.AddOne("return", Layout.StorageType.Register,
                            registerIndex++, ap.PointerSize, t.Size, true);
                    }
                }
                else
                {
                    // レジスタ 1 個で渡せる
                    al.Out.Add("return", Layout.StorageType.Register, 0, t.Size, t.Size);
                }
            }

            // 引数
            foreach (var p in op.Parameters)
            {
                if (p.Type.Name == "void")
                {
                    throw new ErrorException(
                        string.Format("{0} のパラメータ {1} の型が void です。"
                            + "void 型のパラメータは使用できません。",
                        op.Name,
                        p.Name));
                }

                bool isPointer = IsPassByPointer(p, ap.StorageSize, ap.PointerSize, ap.RegisterSize);
                int dataSize = p.GetSize(ap.PointerSize, ap.RegisterSize);
                int transferSize;

                if (isPointer)
                {
                    transferSize = ap.PointerSize;
                }
                else
                {
                    if (p.IsUnknownSize)
                    {
                        throw new ErrorException(
                            string.Format("{0} のパラメータ {1} のサイズが不定です。"
                                + "サイズが不定の場合 {2} でなければなりません。",
                            op.Name,
                            p.Name,
                            p.IsInput ? "inptr" : "outptr"));
                    }
                    transferSize = dataSize;
                }

                if ((!isPointer) && p.Type.RealType.IsPredefined && dataSize > ap.StorageSize)
                {
                    // long long

                    // レジスタに載るか？
                    if (al.In.IsRegisterAvailable(0) && al.In.IsRegisterAvailable(1))
                    {
                        // r0-r1 に載る
                        al.In.Add(p.Name, Layout.StorageType.Register, 0, transferSize, dataSize);
                        registerIndex = 2;
                    }
                    else if (al.In.IsRegisterAvailable(2) && al.In.IsRegisterAvailable(3))
                    {
                        // r2-r3 に載る
                        al.In.Add(p.Name, Layout.StorageType.Register, 2, transferSize, dataSize);
                        registerIndex = 4;
                    }
                    else
                    {
                        // 載らない
                        stackIndex += (stackIndex & 1);
                        al.In.Add(p.Name, Layout.StorageType.Stack, stackIndex, transferSize, dataSize);
                        stackIndex += 2;

                        // レジスタインデックを更新してこれ以降レジスタにのらないようにする
                        registerIndex = ap.RegisterCount;
                    }
                }
                else
                {
                    // long long 以外

                    int numRegisterAvailables = ap.RegisterCount - registerIndex;

                    // レジスタに載るか？
                    if (transferSize <= numRegisterAvailables * ap.StorageSize)
                    {
                        // 載る
                        registerIndex += al.In.Add(p.Name, Layout.StorageType.Register, registerIndex, transferSize, dataSize, isPointer);
                    }
                    else
                    {
                        // 載らない
                        stackIndex += al.In.Add(p.Name, Layout.StorageType.Stack, stackIndex, transferSize, dataSize, isPointer);

                        // レジスタインデックを更新してこれ以降レジスタにのらないようにする
                        registerIndex = ap.RegisterCount;
                    }
                }
            }

            return al;
        }

        private static bool IsPassByPointer(OperationParameter op, int storageSize, int pointerSize, int registerSize)
        {
            // 配列はポインタとして扱う
            if (op.IsArray)
            {
                return true;
            }

            // in 以外はポインタとして扱う
            if (op.Direction != OperationParameter.DirectionType.In)
            {
                return true;
            }

            // 組み込み型でなく StorageSize 以上ならポインタとして扱う
            if ((!op.Type.RealType.IsPredefined) && (op.GetSize(pointerSize, registerSize) > storageSize))
            {
                return true;
            }

            return false;
        }
    }
}
