﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SvcStubSourceGenerator : SourceGenerator
    {
        private string m_idPathTemplate;

        //--------------------------------------------------------------
        // public

        public SvcStubSourceGenerator(CodeGenNames p) : base(p) { }

        public void Generate(SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string pathTemplate,
            string idPathTemplate)
        {
            this.m_idPathTemplate = idPathTemplate;
            this.GenerateWithCategory(ss, abi, svc, conv, templatePath, pathTemplate);
        }

        //--------------------------------------------------------------
        // override

        /// <summary>
        /// システムコール呼び出し関数定義ファイルの内容全体を生成します。
        /// </summary>
        /// <param name="c">対象のカテゴリのデータ</param>
        /// <param name="abi">各関数の ABI レイアウト</param>
        /// <param name="svc">各関数の SVC レイアウト</param>
        /// <param name="conv">各関数のレイアウト変換処理</param>
        /// <returns>システムコール呼び出し関数定義ファイルの内容全体</returns>
        protected override string Generate(CategorySet c,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        {
            if (c.Operations == null)
            {
                return null;
            }

            var idPath = this.m_idPathTemplate.Replace("*", c.Name).Replace("\\", "/");

            var sb = new StringBuilder();
            sb.Append(SourceGenerator.MakeIncludeLine(idPath));
            sb.AppendLine();
            sb.AppendLine();

            foreach (var op in c.Operations)
            {
                try
                {
                    var text = this.Generate(op, abi[op.Name], svc[op.Name], conv[op.Name]);
                    sb.Append(text);
                    sb.AppendLine();
                    sb.AppendLine();
                }
                catch (ErrorException ee)
                {
                    throw new ErrorException(
                        string.Format("operation: {0}", op.Name), ee);
                }
            }

            sb.AppendLine("    .end");
            sb.AppendLine();
            return sb.ToString();
        }

        //--------------------------------------------------------------
        // private

        /// <summary>
        /// 関数ごとのシステムコール呼び出し関数定義を生成します。
        /// </summary>
        /// <param name="op">対象の関数</param>
        /// <param name="al">対象の関数の ABI レイアウト</param>
        /// <param name="sl">対象の関数の SVC レイアウト</param>
        /// <param name="lc">対象の関数のレイアウト変換処理</param>
        /// <returns>システムコール呼び出し関数定義</returns>
        private string Generate(Operation op, AbiLayout al, SvcLayout sl, LayoutConversion lc)
        {
            var symbolFormat = string.Format("{0}::{1}::{2}",
                CodeGenNames.UserCommonNamespace,
                Params.StubNamespace,
                "{0}");
            var symbolName = MakeSymbolName(symbolFormat, op, al, false);

            var code = GenerateCode(op, al, lc);

            var sb = new StringBuilder();
            sb.AppendFormat("//-------------------------------------------------\r\n");
            sb.AppendFormat("// {0}\r\n", SourceGenerator.MakePrototype(op, al, false));
            sb.AppendFormat("//\r\n");
            sb.Append(MakeLayoutComment(al, sl, lc, op));
            sb.AppendLine();
            sb.AppendFormat("    .section .text.{0}, \"ax\"\r\n", symbolName);
            sb.AppendFormat("    .align   2\r\n");
            sb.AppendFormat("    .global  {0}\r\n", symbolName);
            sb.AppendFormat("    .hidden  {0}\r\n", symbolName);
            sb.AppendFormat("    .type    {0}, %function\r\n", symbolName);
            sb.AppendFormat("{0}:\r\n", symbolName);
            sb.AppendLine();
            sb.Append(code);
            sb.AppendLine();
            sb.AppendFormat("    .size {0}, [.-{0}]\r\n", symbolName);
            return sb.ToString();
        }

        //--------------------------------------------------------------
        // private static

        /// <summary>
        /// 対象の関数の指定された引数または返り値のサイズを表す文字列を取得します。
        /// </summary>
        /// <param name="op">対象の関数</param>
        /// <param name="name">引数の名前。"return" を指定することで返り値を指定できます。</param>
        /// <returns>指定された引数または返り値のサイズを表す文字列</returns>
        private static string GetVariableSize(Operation op, string name, int pointerSize, int registerSize)
        {
            if (name == "return")
            {
                return op.ReturnType.Size.ToString();
            }
            else
            {
                return op.Parameters.First(x => x.Name == name).GetSizeText(pointerSize, registerSize);
            }
        }
        /// <summary>
        /// 対象のレイアウトをダンプします。
        /// </summary>
        /// <param name="layout">対象のレイアウト</param>
        /// <param name="op">対象のレイアウトが対象としている関数</param>
        /// <returns>layout に対して、レイアウト上の位置を表す文字列をキーとし、その内容を表す文字列を値とする辞書</returns>
        private static Dictionary<string, string> FormatParams(Layout layout, Operation op)
        {
            var list = new Dictionary<string, string>();
            foreach (var kv in layout.StorageParams)
            {
                var pos = string.Format("{0,-8} {1}", kv.Item1.Storage, kv.Item1.Index);
                var text = string.Format("{0}  {2,3} {3,2} {1}",
                    pos,
                    kv.Item2.VariableName,
                    kv.Item2.IsReference ? "ref" : string.Empty,
                    GetVariableSize(op, kv.Item2.VariableName, layout.PointerSize, layout.RegisterSize));
                list.Add(pos, text);
            }
            return list;
        }
        /// <summary>
        /// 対象の関数のレイアウトをダンプします。
        /// </summary>
        /// <param name="op">対象の関数</param>
        /// <param name="al">対象の関数の ABI レイアウト</param>
        /// <param name="sl">対象の関数の SVC レイアウト</param>
        /// <param name="lc">対象の関数のレイアウト変換処理</param>
        /// <returns>対象の関数のレイアウトをダンプした文字列</returns>
        private static string MakeLayoutComment(
            AbiLayout al, SvcLayout sl, LayoutConversion lc, Operation op)
        {
            var inAbiList = FormatParams(al.In, op);
            var inSvcList = FormatParams(sl.In, op);
            var outAbiList = FormatParams(al.Out, op);
            var outSvcList = FormatParams(sl.Out, op);

            var abiMaxLength = inAbiList.Concat(outAbiList).Select(x => x.Value).Add(string.Empty).Max(x => x.Length);
            var inKeys = inAbiList.Concat(inSvcList).ToLookup(x => x.Key).Select(x => x.Key).OrderBy(x => x).ToArray();
            var outKeys = outAbiList.Concat(outSvcList).ToLookup(x => x.Key).Select(x => x.Key).OrderBy(x => x).ToArray();

            var lineFormat = string.Concat("// {0,-3} {1,-", abiMaxLength, "}   {2}\r\n");

            var sb = new StringBuilder();
            sb.AppendFormat(lineFormat, string.Empty, al.CodeGenParams, sl.CodeGenParams);
            sb.AppendFormat(lineFormat, "in", "abi", "svc");
            foreach (var key in inKeys)
            {
                var abi = GetOrEmpty(inAbiList, key);
                var svc = GetOrEmpty(inSvcList, key);
                sb.AppendFormat(lineFormat, string.Empty, abi, svc);
            }
            sb.AppendFormat("//\r\n");
            sb.AppendFormat(lineFormat, "out", "abi", "svc");
            foreach (var key in outKeys)
            {
                var abi = GetOrEmpty(outAbiList, key);
                var svc = GetOrEmpty(outSvcList, key);
                sb.AppendFormat(lineFormat, string.Empty, abi, svc);
            }
            sb.AppendFormat("//\r\n");
            sb.AppendFormat("// conversion\r\n");
            foreach (var c in lc.PreOperations.Concat(lc.PostOperations))
            {
                sb.AppendFormat("//   {0,-15} {1}\r\n", c.Type, c.VariableName);
            }
            return sb.ToString();
        }
        /// <summary>
        /// システムコール呼び出し関数の命令列アセンブラコードを生成します。
        /// </summary>
        /// <param name="op">対象の関数</param>
        /// <param name="al">対象の関数の ABI レイアウト</param>
        /// <param name="lc">対象の関数のレイアウト変換処理</param>
        /// <returns>システムコール呼び出し関数の命令列アセンブラコード</returns>
        private static string GenerateCode(Operation op, AbiLayout al, LayoutConversion lc)
        {
            var storageSize = al.In.StorageSize;

            int stackOffset = 0;
            var saveMap = new Dictionary<int, int>();
            CodeGenerator cg = CodeGenerator.FromCodeGenParams(al.CodeGenParams, null);

            // レジスタの保存
            if (lc.SaveRegisters.Length > 0)
            {
                stackOffset = MakeSaveCode(cg, saveMap, lc.SaveRegisters, storageSize);

                cg.AddEmptyLine();
            }

            // SVC 呼び出し前の処理
            //   ABI → SVC
            if (lc.PreOperations.Length > 0)
            {
                var tracer = new RegisterUsageTracer();
                for (int regNo = 0; regNo < al.In.RegisterCount; ++regNo)
                {
                    var name = al.In.GetRegisterParam(regNo);
                    if (name != null)
                    {
                        tracer.Occupy(regNo, "abi " + name);
                    }
                }
                lc.SaveRegisters.ForEach(x => tracer.Release(x));

                foreach (var lco in lc.PreOperations)
                {
                    lco.GenerateCodeInAbiToSvc(cg, tracer, stackOffset);
                }
                cg.AddEmptyLine();
            }

            // SVC 呼び出し
            cg.CallSystem(CodeGenNames.IdNamePrefix + SourceGenerator.MakeMacroName(op.Name));

            // SVC 呼び出し後の処理
            //   SVC → ABI
            if (lc.PostOperations.Length > 0)
            {
                cg.AddEmptyLine();
                foreach (var lco in lc.PostOperations)
                {
                    lco.GenerateCodeOutSvcToAbi(cg, null, saveMap, lc.SafeRegister, stackOffset);
                }
            }

            // レジスタの復帰
            // if 条件は RestoreRegisters ではない
            if (lc.SaveRegisters.Length > 0)
            {
                cg.AddEmptyLine();
                MakeRestoreCode(cg, lc.SaveRegisters, lc.RestoreRegisters, storageSize);
            }

            // return
            cg.Return();

            return cg.CodeText;
        }
        /// <summary>
        /// 関数先頭のレジスタ保存処理アセンブラコードを生成します。
        /// </summary>
        /// <param name="cg">アセンブラ生成器</param>
        /// <param name="saveMap">レジスタの保存先記録先</param>
        /// <param name="saveRegisters">保存すべきレジスタ番号の配列</param>
        /// <param name="storageSize">保存単位サイズ</param>
        /// <returns></returns>
        private static int MakeSaveCode(
            CodeGenerator cg,
            Dictionary<int, int> saveMap,
            int[] saveRegisters,
            int storageSize)
        {
            cg.SaveRegisters(saveRegisters, false, false);

            int stackOffset = 0;
            foreach (var reg in saveRegisters)
            {
                saveMap.Add(reg, stackOffset);
                stackOffset += storageSize;
            }
            return stackOffset;
        }
        /// <summary>
        /// 関数末尾のレジスタ復帰処理アセンブラコードを生成します。
        /// savedRegisters の内容は認識せず、その要素数の restoreRegisters との差のみを使用します。
        /// </summary>
        /// <param name="cg">アセンブラ生成器</param>
        /// <param name="savedRegisters">保存したレジスタ番号の配列</param>
        /// <param name="restoreRegisters">復帰すべきレジスタ番号の配列</param>
        /// <param name="storageSize">保存単位サイズ</param>
        private static void MakeRestoreCode(
            CodeGenerator cg, int[] savedRegisters, int[] restoreRegisters, int storageSize)
        {
            var numAdd = savedRegisters.Length - restoreRegisters.Length;
            if (numAdd > 0)
            {
                cg.FreeToStack(numAdd * storageSize);
            }
            if (restoreRegisters.Length > 0)
            {
                cg.RestoreRegisters(restoreRegisters, false, false);
            }
        }

        //--------------------------------------------------------------
        // private static util

        private static string GetOrEmpty(Dictionary<string, string> dic, string key)
        {
            string v;
            return dic.TryGetValue(key, out v) ? v : string.Empty;
        }
    }
}
