﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using MakeNso.Elf;

namespace MakeNso
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            if (Directory.Exists(Path.Combine(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location), "en")))
            {
                System.Threading.Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
            }

            MakeNsoArgs opt = new MakeNsoArgs();

            // コマンドライン引数のパースはパース処理の中で、エラーを表示するため、try を分ける
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            try
            {
                MakeNsoParams param = opt.Params;
                if (param.Extract)
                {
                    NsoFile nsoFile = new NsoFile(param.InputFileName);
                    using (var outFs = new FileStream(param.OutputFileName, FileMode.Create, FileAccess.Write))
                    {
                        nsoFile.ExtractData(outFs);
                    }
                }
                else
                {
                    bool compressMode = true;
                    NsoFile nsoFile = new NsoFile();

                    if (param.Compress)
                    {
                        compressMode = true;
                    }
                    if (param.NoCompress)
                    {
                        compressMode = false;
                    }
                    nsoFile.CompressMode = compressMode;

                    nsoFile.SetModuleName(param.ModuleName);

                    ElfInfo elf = new ElfInfo(param.InputFileName);

                    foreach (var section in elf.SectionInfos)
                    {
                        if (elf.FileType == ElfFileType.SharedObjectFile && section.AddressAlign > 0x1000)
                        {
                            throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSectionAlign, section.SectionName, section.AddressAlign));
                        }
                    }

                    byte[] buildId = BuildId.GetBuildId(elf);
                    if (buildId != null)
                    {
                        nsoFile.SetModuleId(buildId);
                    }

                    ElfSegmentInfo textInfo = elf.GetExElfSegmentInfo();
                    ElfSegmentInfo roInfo = elf.GetRoElfSegmentInfo();
                    ElfSegmentInfo dataInfo = elf.GetRwElfSegmentInfo();
                    ElfSegmentInfo bssInfo = elf.GetZiElfSegmentInfo();

                    nsoFile.SetTextSegment(elf.GetExElfSegmentInfo());
                    nsoFile.SetRoSegment(elf.GetRoElfSegmentInfo());
                    nsoFile.SetDataSegment(elf.GetRwElfSegmentInfo());
                    nsoFile.SetBssSegment(elf.GetZiElfSegmentInfo());

                    ElfSectionInfo apiInfo = elf.GetApiInfoElfSectionInfo();
                    if (apiInfo != null)
                    {
                        nsoFile.SetApiInfo(apiInfo.VirtualAddress - roInfo.VirtualAddress, apiInfo.SectionSize);
                    }

                    ElfSectionInfo dynStrInfo = elf.GetDynStrElfSectionInfo();
                    if (dynStrInfo != null)
                    {
                        nsoFile.SetDynStrInfo(dynStrInfo.VirtualAddress - roInfo.VirtualAddress, dynStrInfo.SectionSize);
                    }

                    ElfSectionInfo dynSymInfo = elf.GetDynSymElfSectionInfo();
                    if (dynSymInfo != null)
                    {
                        nsoFile.SetDynSymInfo(dynSymInfo.VirtualAddress - roInfo.VirtualAddress, dynSymInfo.SectionSize);
                    }

                    nsoFile.CompressTextSegment();
                    nsoFile.CompressRoSegment();
                    nsoFile.CompressDataSegment();

                    nsoFile.CalcPosition();

                    using (FileStream outputFs = new FileStream(param.OutputFileName, FileMode.Create, FileAccess.Write))
                    {
                        nsoFile.WriteData(outputFs);
                    }

                    if (param.VerboseMode)
                    {
                        nsoFile.PrintNsoHeader();
                    }
                }
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(string.Format("MakeNso INPUT='{0}', OUTPUT='{1}'", opt.Params.InputFileName, opt.Params.OutputFileName));
                Console.Error.WriteLine(ex.Message);
                Environment.ExitCode = 1;
                return;
            }
        }
    }
}
