﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeNro.Elf
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// シンボルテーブルセクションを表します。
    /// </summary>
    public sealed class ElfSymbolTableInfo
    {
        private ElfSymbolTableInfo(ElfInfo elfInfo, long offset)
        {
            this.FileName = elfInfo.FileName;

            this.Architecture = elfInfo.Architecture;

            this.ByteOrder = elfInfo.ByteOrder;

            using (var fs = File.OpenRead(elfInfo.FileName))
            using (var er = new ElfBinaryReader(fs, elfInfo.Architecture, elfInfo.ByteOrder))
            {
                fs.Seek(offset, SeekOrigin.Begin);

                this.SymbolNameOffset = er.ReadUInt32(); // st_name

                // シンボル名のオフセット値以降は、アーキテクチャごとに構造が違うので、別々に処理する
                if (this.Architecture == ElfArchitectureType.Elf32)
                {
                    this.SymbolAddress = er.ReadUInt32(); // st_value
                    er.ReadUInt32(); // st_size
                    er.ReadByte(); // st_info
                    er.ReadByte(); // st_other
                    this.SymbolSectionIndex = er.ReadUInt16(); // st_shndx
                }
                else
                {
                    er.ReadByte(); // st_info
                    er.ReadByte(); // st_other
                    this.SymbolSectionIndex = er.ReadUInt16(); // st_shndx
                    this.SymbolAddress = er.ReadUInt64(); // st_value
                    er.ReadUInt64(); // st_size
                }
            }

            this.SymbolName = string.Empty;
        }

        /// <summary>
        /// ELF ファイルのパスを取得します。
        /// </summary>
        public string FileName { get; private set; }

        /// <summary>
        /// アーキテクチャタイプを取得します。
        /// </summary>
        public ElfArchitectureType Architecture { get; private set; }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        /// <summary>
        /// シンボル名の位置を示すオフセットを取得します。
        /// </summary>
        public ulong SymbolNameOffset { get; private set; }

        /// <summary>
        /// シンボルのセクションオフセット値を取得します。
        /// </summary>
        public ulong SymbolAddress { get; private set; }

        /// <summary>
        /// シンボルの属するセクションのインデックス値を取得します。
        /// </summary>
        public uint SymbolSectionIndex { get; private set; }

        /// <summary>
        /// シンボル文字列テーブルのセクションのオフセット値を取得します。
        /// </summary>
        public uint SymbolStringTableSectionOffset { get; private set; }

        /// <summary>
        /// シンボル文字列テーブルのセクションのサイズ値を取得します。
        /// </summary>
        public uint SymbolStringTableSectionSize { get; private set; }

        /// <summary>
        /// シンボル名を取得します。
        /// </summary>
        public string SymbolName { get; private set; }

        /// <summary>
        /// ELF データのシンボルテーブルからシンボルデータを読み取ります。
        /// </summary>
        /// <param name="elfInfo">ELF データです。</param>
        /// <param name="offset">シンボルテーブルの位置を示すオフセットです。</param>
        /// <param name="size">シンボルテーブルのサイズです。</param>
        /// <param name="entrySize">シンボルテーブルのエントリサイズです。</param>
        /// <param name="symbolStringTableOffset">シンボル文字列テーブルの位置を示すオフセットです。</param>
        /// <param name="SymbolStringTableSectionSize">シンボル文字列テーブルのサイズです。</param>
        internal static IReadOnlyList<ElfSymbolTableInfo> GetElfSymbolTableInfos(ElfInfo elfInfo, ulong offset, ulong size, ulong entrySize, ulong symbolStringTableOffset, ulong SymbolStringTableSectionSize)
        {
            if (!IsValidSymbolTableEntrySize(entrySize, elfInfo.Architecture))
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidSectionHeaderEntrySize);
            }

            var list = new List<ElfSymbolTableInfo>();

            var entryNum = (long)(size / entrySize);
            for (long i = 0; i < entryNum; ++i)
            {
                var entryOffset = (long)offset + ((long)entrySize * i);

                list.Add(new ElfSymbolTableInfo(elfInfo, entryOffset));
            }

            var bytes = GetSymbolTableStringContents(elfInfo, symbolStringTableOffset, SymbolStringTableSectionSize);

            foreach (var si in list)
            {
                /// 各セクションのセクション名オフセットから、'0' がくるまでバイトリードして、それを文字列化する関数を呼び出す。
                si.SymbolName = GetSectionNameFromStringTable(bytes, (uint)si.SymbolNameOffset);
            }

            return new ReadOnlyCollection<ElfSymbolTableInfo>(list);
        }

        private static  byte[] GetSymbolTableStringContents(ElfInfo elfInfo, ulong symbolStringTableOffset, ulong SymbolStringTableSectionSize)
        {
            using (var fs = File.OpenRead(elfInfo.FileName))
            using (var br = new BinaryReader(fs))
            {
                fs.Seek((long)symbolStringTableOffset, SeekOrigin.Begin);

                return br.ReadBytes((int)SymbolStringTableSectionSize);
            }
        }

        /// シンボルテーブルのエントリサイズは arm32 と arm64 で違うので、それが正しいかをチェックするための関数
        private static bool IsValidSymbolTableEntrySize(ulong size, ElfArchitectureType architecture)
        {
            if (architecture == ElfArchitectureType.Elf32 && size == 16)
            {
                return true;
            }

            if (architecture == ElfArchitectureType.Elf64 && size == 24)
            {
                return true;
            }

            return false;
        }

        /// ストリングテーブル全体のバイトデータを受け取り、セクション名オフセットから、'0' までのバイトを読み込んで、それを文字列化して返す関数。
        private static string GetSectionNameFromStringTable(byte[] bytes, uint offset)
        {
            var sb = new StringBuilder();

            for (uint i = offset; i < bytes.Length; ++i)
            {
                if (bytes[i] > 0)
                {
                    sb.Append((char)bytes[i]);
                }
                else
                {
                    return sb.ToString();
                }
            }

            throw new ArgumentException(Properties.Resources.Message_FailedToGetSectionName);
        }
    }
}
