﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("MakeInitialProgramTest")]

namespace MakeInitialProgram
{
    /// <summary>
    /// MakeSvcVeneer で作成したヘッダファイルを読み込み、システムコール名、システムコール番号を抽出します。
    /// </summary>
    internal class SystemCallInfo
    {
        private Dictionary<string, uint> SyscallDictionary { get; set; }
        /// <summary>
        /// SystemCallInfo のコンストラクタ
        /// </summary>
        internal SystemCallInfo()
        {
            SyscallDictionary = new Dictionary<string, uint>();
        }

        /// <summary>
        /// システムコールの情報を読み込みます。
        /// </summary>
        /// <param name="filePath">システムコール番号とシステムコールの名前の定義が書いてあるファイルのパス</param>
        internal void Load(string filePath)
        {
            int curNum = SyscallDictionary.Count;
            using (var sr = new StreamReader(filePath))
            {
                string line;

                while ((line = sr.ReadLine()) != null)
                {
                    var match = Regex.Match(line, @"^#define *([A-Z][A-Z0-9_]+) +([0-9]+)$");

                    if (!match.Success)
                    {
                        continue;
                    }

                    if (match.Groups.Count != 3)
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_InvalidSvcHeaderFormat, filePath));
                    }

                    string syscallName = ConvertCamelNameToSnakeName(filePath, match.Groups[1].Value);
                    string syscallId = match.Groups[2].Value;
                    uint syscallNum;

                    if (!uint.TryParse(syscallId, out syscallNum))
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_InvalidNumberBaseFormat, filePath, syscallName, 10));
                    }

                    if (SyscallDictionary.ContainsKey(syscallName))
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_SvcAlreadyRegistered, filePath, syscallName));
                    }

                    if (SyscallDictionary.ContainsValue(syscallNum))
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_SvcDuplicated, filePath, syscallName, syscallNum));
                    }

                    SyscallDictionary.Add(syscallName, syscallNum);
                }

                if (SyscallDictionary.Count == curNum)
                {
                    throw new ArgumentException(
                        string.Format(Properties.Resources.Message_SvcNotFound, filePath));
                }
            }
        }

        /// <summary>
        /// 与えられた引数から検索可能なシステムコール番号があるかを調べます
        /// </summary>
        /// <param name="syscallName">検索するシステムコール名</param>
        /// <returns>システムコール番号が検索可能かどうか</returns>
        internal bool HasSystemCall(string syscallName)
        {
            return SyscallDictionary.ContainsKey(syscallName);
        }

        /// <summary>
        /// 与えられた引数で検索されたシステムコール番号を取得します。
        /// </summary>
        /// <param name="syscallName">システムコール名</param>
        /// <returns>システムコール番号</returns>
        internal uint GetSystemCallNumber(string syscallName)
        {
            return SyscallDictionary[syscallName];
        }

        /// <summary>
        /// 所持しているシステムコール情報の個数を取得します。
        /// </summary>
        /// <returns>保持しているシステムコール情報の個数</returns>
        internal int GetNumInfo()
        {
            return SyscallDictionary.Values.ToArray().Count();
        }

        private string ConvertCamelNameToSnakeName(string filePath, string name)
        {
            const string PrefixName = "NN_SVC_ID_";
            int index = name.IndexOf(PrefixName);

            if (index != 0)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidSvcDefinition, filePath, name, PrefixName));
            }

            name = name.Replace(PrefixName, string.Empty);

            string[] names = name.Split('_');
            string result = string.Empty;

            foreach (var elem in names)
            {
                char[] buf = elem.ToLower().ToCharArray();
                buf[0] = (char)((int)buf[0] + ((int)'A' - (int)'a'));
                result += new string(buf);
            }

            return result;
        }
    }
}
