﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// コマンドラインオプションを解析して、初期プログラム作成ツールから利用できるようにします。
    /// </summary>
    internal sealed class CommandLineOption
    {
        private static readonly Parameter[] Parameters = new[]
        {
            new Parameter()
            {
                LongName     = "--help",
                ShortName    = "-h",
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = Properties.Resources.Option_Help,
                Action = (opt, x) =>
                {
                    opt.Help = true;
                }
            },
            new Parameter()
            {
                LongName     = "--verbose",
                ShortName    = null,
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = Properties.Resources.Option_Verbose,
                Action = (opt, x) =>
                {
                    opt.Verbose = true;
                }
            },
            new Parameter()
            {
                LongName     = "-o",
                ShortName    = null,
                MetaVariable = "file",
                IsMandatory  = true,
                HelpText     = Properties.Resources.Option_OutputPath,
                Action = (opt, x) =>
                {
                    opt.Output = x;
                }
            },
            new Parameter()
            {
                LongName     = "--elf",
                ShortName    = null,
                MetaVariable = "file",
                IsMandatory  = true,
                HelpText     = Properties.Resources.Option_ElfPath,
                Action = (opt, x) =>
                {
                    opt.Elf = x;
                }
            },
            new Parameter()
            {
                LongName     = "--param_file",
                ShortName    = null,
                MetaVariable = "file",
                IsMandatory  = true,
                HelpText     = Properties.Resources.Option_ParameterFilePath,
                Action = (opt, x) =>
                {
                    opt.ParamFile = x;
                }
            },
            new Parameter()
            {
                LongName     = "--svc_header",
                ShortName    = null,
                MetaVariable = "file",
                IsMandatory  = false,
                HelpText     = Properties.Resources.Option_SvcHeaderPath,
                Action = (opt, x) =>
                {
                    opt.SvcHeader.Add(x);
                }
            },
            new Parameter()
            {
                LongName     = "--testmode",
                ShortName    = null,
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = Properties.Resources.Option_TestMode,
                Action = (opt, x) =>
                {
                    opt.IsTestMode = true;
                }
            },
            new Parameter()
            {
                LongName     = "--utf8",
                ShortName    = null,
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = Properties.Resources.Option_Utf8,
                Action = (opt, x) =>
                {
                    opt.IsUtf8 = true;
                }
            },
        };

        /// <summary>
        /// CommandLineOption クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal CommandLineOption()
        {
            this.Output = string.Empty;

            this.Elf = string.Empty;

            this.ParamFile = string.Empty;

            this.SvcHeader = new List<string>();

            this.Help = false;

            this.Verbose = false;

            this.IsTestMode = false;

            this.IsUtf8 = false;
        }

        /// <summary>
        /// 初期プログラムの出力先となる絶対パスを取得します。
        /// </summary>
        internal string Output { get; private set; }

        /// <summary>
        /// 初期プログラムの作成元となるELFファイルの絶対パスを取得します。
        /// </summary>
        internal string Elf { get; private set; }

        /// <summary>
        /// 初期プログラムの作成時に使用するパラメータ指定ファイルの絶対パスを取得します。
        /// </summary>
        internal string ParamFile { get; private set; }

        /// <summary>
        /// システムコールの ID が定義されたヘッダファイルを取得します。
        /// </summary>
        internal List<string> SvcHeader { get; private set; }

        /// <summary>
        /// ヘルプメッセージを出力するかどうかを示す値を取得します。
        /// </summary>
        internal bool Help { get; private set; }

        /// <summary>
        /// ログメッセージを表示するかどうかを示す値を取得します。
        /// </summary>
        internal bool Verbose { get; private set; }

        /// <summary>
        /// テスト用のデータを作成するかどうかを示す値を取得します。
        /// </summary>
        internal bool IsTestMode { get; private set; }

        /// <summary>
        /// 出力に UTF8 を利用します。
        /// </summary>
        internal bool IsUtf8 { get; private set; }

        /// <summary>
        /// ヘルプメッセージを取得します。
        /// </summary>
        /// <returns>ヘルプメッセージです。</returns>
        internal static string GetHelpMessage()
        {
            var name = System.IO.Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location);

            var sb = new System.Text.StringBuilder();

            sb.AppendFormat("Usage: {0}", name);

            foreach (var param in Parameters)
            {
                if (!param.IsMandatory)
                {
                    continue;
                }

                sb.AppendFormat(" {0}", param.LongName);

                if (param.MetaVariable != null)
                {
                    sb.AppendFormat(" <{0}>", param.MetaVariable);
                }
            }

            sb.AppendLine(" [options]");

            sb.AppendLine();

            sb.Append("options:");

            foreach (var param in Parameters)
            {
                sb.AppendLine();

                var sb_opt = new System.Text.StringBuilder();

                sb_opt.AppendFormat("  {0}", param.LongName);

                if (param.MetaVariable != null)
                {
                    sb_opt.AppendFormat(" <{0}>", param.MetaVariable);
                }

                if (param.ShortName != null)
                {
                    sb_opt.AppendFormat(", {0}", param.ShortName);

                    if (param.MetaVariable != null)
                    {
                        sb_opt.AppendFormat(" <{0}>", param.MetaVariable);
                    }
                }

                sb.AppendFormat("{0,-31} {1}", sb_opt.ToString(), param.HelpText);
            }

            return sb.ToString();
        }

        /// <summary>
        /// コマンドラインオプションをパースします。
        /// </summary>
        /// <param name="args">コマンドラインオプションです。</param>
        internal void Parse(string[] args)
        {
            var mandates = new List<string>();

            foreach (var param in Parameters)
            {
                if (param.IsMandatory)
                {
                    mandates.Add(param.LongName);
                }
            }

            for (var i = 0; i < args.Length; ++i)
            {
                for (var j = 0; j < Parameters.Length; ++j)
                {
                    var key = args[i];

                    var param = Parameters[j];

                    if ((string.Compare(param.LongName, key) == 0) || (string.Compare(param.ShortName, key) == 0))
                    {
                        if (param.MetaVariable == null)
                        {
                            param.Action(this, null);
                        }
                        else
                        {
                            if ((i >= (args.Length - 1)) || args[i + 1].StartsWith("-"))
                            {
                                throw new ArgumentException(string.Format(Properties.Resources.Option_NotFoundArgument, key));
                            }

                            param.Action(this, args[++i]);
                        }

                        mandates.Remove(param.LongName);

                        break;
                    }

                    if (j + 1 < Parameters.Length)
                    {
                        continue;
                    }

                    throw new ArgumentException(string.Format(Properties.Resources.Option_InvalidArgument, args[i]));
                }
            }

            if (!this.Help)
            {
                foreach (var mandate in mandates)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Option_NotFound, mandate));
                }
            }
        }

        private class Parameter
        {
            internal string LongName { get; set; }

            internal string ShortName { get; set; }

            internal string MetaVariable { get; set; }

            internal bool IsMandatory { get; set; }

            internal string HelpText { get; set; }

            internal Action<CommandLineOption, string> Action { get; set; }
        }
    }
}
