﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeInitialProcessImage
{
    internal class InitialProcessImageGenerator
    {
        private class InitialProcessImageHeader
        {
            private static readonly byte[] ValidSignature = new byte[] { (byte)'I', (byte)'N', (byte)'I', (byte)'1' };

            public static readonly int HeaderLength = 16;

            public uint ImageLength { get; set; }
            public uint InitialProcessCount { get; set; }

            public void Write(Stream s)
            {
                using (var bw = new BinaryWriter(s, Encoding.UTF8, true))
                {
                    bw.Write(ValidSignature.ToArray());
                    bw.Write(ImageLength);
                    bw.Write(InitialProcessCount);
                    bw.Write((uint)0);  // パディング
                }
            }
        }

        /// <summary>
        /// INITIAL_PROCESS イメージを生成します。
        /// </summary>
        /// <param name="outputPath">INITIAL_PROCESS イメージの出力先パス</param>
        /// <param name="inputPaths">入力ファイル（.kip ファイル）</param>
        public void Generate(string outputPath, string[] inputPaths)
        {
            if (outputPath == null)
            {
                throw new ArgumentNullException("outputPath");
            }
            if (inputPaths == null)
            {
                throw new ArgumentNullException("inputPaths");
            }

            bool outputFileCreated = false;
            try
            {
                using (var outFile = OpenOutputFileStream(outputPath))
                {
                    outputFileCreated = true;
                    // 仮のヘッダを書いておく
                    new InitialProcessImageHeader().Write(outFile);

                    // 入力ファイルの内容を出力ファイルに書きだす
                    long totalLength = InitialProcessImageHeader.HeaderLength;
                    foreach (var inputPath in inputPaths)
                    {
                        using (var inFile = OpenInputFileStream(inputPath))
                        {
                            totalLength += inFile.Length;
                            inFile.CopyTo(outFile);
                        }
                    }

                    // 念のため、ヘッダの ImageLength に収まるかどうかチェック
                    if (totalLength > uint.MaxValue)
                    {
                        throw new MakeInitialProcessImageException("The length of output file exceeds the 4GiB limit.");
                    }

                    // 先頭にシークして、真のヘッダを書く
                    outFile.Seek(0, SeekOrigin.Begin);
                    var trueHeader = new InitialProcessImageHeader()
                    {
                        InitialProcessCount = (uint)inputPaths.Length,
                        ImageLength = (uint)totalLength,
                    };
                    trueHeader.Write(outFile);
                }
            }
            catch
            {
                // 失敗したら出力ファイルを削除
                if (outputFileCreated)
                {
                    File.Delete(outputPath);
                }
                throw;
            }
        }

        private static FileStream OpenOutputFileStream(string outputPath)
        {
            try
            {
                return new FileStream(outputPath, FileMode.Create, FileAccess.Write, FileShare.None);
            }
            catch (Exception e)
            {
                throw new MakeInitialProcessImageException(string.Format("Failed to create output file {0} : {1}", outputPath, e.Message), e);
            }
        }

        private static FileStream OpenInputFileStream(string inputPath)
        {
            try
            {
                return new FileStream(inputPath, FileMode.Open, FileAccess.Read, FileShare.Read);
            }
            catch (Exception e)
            {
                throw new MakeInitialProcessImageException(string.Format("Failed to open input file {0} : {1}", inputPath, e.Message), e);
            }
        }
    }
}
